/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository;

import java.util.Map;
import java.util.Set;

import org.ametys.core.group.GroupIdentity;
import org.ametys.core.right.ProfileAssignmentStorage;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.repository.jcr.ACLJCRAmetysObjectHelper;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Implementation of {@link ProfileAssignmentStorage} for {@link ACLAmetysObject}s which stores profile assignments in subnodes of the node representing this object.
 */
public class ACLAmetysObjectProfileAssignmentStorage extends AbstractLogEnabled implements ProfileAssignmentStorage
{
    /* -------------- */
    /* HAS PERMISSION */
    /* -------------- */

    public Set<String> hasUserAnyAllowedProfile(Set< ? extends Object> rootContexts, UserIdentity user, Set<String> profileIds)
    {
        return ACLJCRAmetysObjectHelper.hasUserAnyAllowedProfile(rootContexts, user, profileIds);
    }
    
    public Set<String> hasGroupAnyAllowedProfile(Set< ? extends Object> rootContexts, Set<GroupIdentity> groups, Set<String> profileIds)
    {
        return ACLJCRAmetysObjectHelper.hasGroupAnyAllowedProfile(rootContexts, groups, profileIds);
    }
    
    public Set<String> hasAnyConnectedAnyAllowedProfile(Set< ? extends Object> rootContexts, Set<String> profileIds)
    {
        return ACLJCRAmetysObjectHelper.hasAnyConnectedAnyAllowedProfile(rootContexts, profileIds);
    }
    
    public Set<String> hasAnonymousAnyAllowedProfile(Set< ? extends Object> rootContexts, Set<String> profileIds)
    {
        return ACLJCRAmetysObjectHelper.hasAnonymousAnyAllowedProfile(rootContexts, profileIds);
    }
    
    /* ----------------------- */
    /* GET ALL PROFILES */
    /* ----------------------- */
    
    public Map<Object, Map<AnonymousOrAnyConnectedKeys, Set<String>>> getAllProfilesForAnonymousAndAnyConnectedUser(Set< ? extends Object> rootContexts)
    {
        return ACLJCRAmetysObjectHelper.getAllProfilesForAnonymousAndAnyConnectedUser(rootContexts);
    }
    
    public Map<Object, Map<GroupIdentity, Map<UserOrGroup, Set<String>>>> getAllProfilesForGroups(Set< ? extends Object> rootContexts, Set<GroupIdentity> groups)
    {
        return ACLJCRAmetysObjectHelper.getAllProfilesForGroups(rootContexts, groups);
    }
    
    public Map<Object, Map<UserOrGroup, Set<String>>> getAllProfilesForUser(Set< ? extends Object> rootContexts, UserIdentity user)
    {
        return ACLJCRAmetysObjectHelper.getAllProfilesForUser(rootContexts, user);
    }
    
    public Map<Object, Set<AnonymousOrAnyConnectedKeys>> getAllAssignmentsForAnonymousAndAnyConnectedUser(Set< ? extends Object> rootContexts, String profileId)
    {
        return ACLJCRAmetysObjectHelper.getProfilePermissionsForAnonymousAndAnyConnectedUser(rootContexts, profileId);
    }
    
    public Map<Object, Map<UserIdentity, Set<UserOrGroup>>> getAllAssignmentsForUsers(Set< ? extends Object> rootContexts, String profileId)
    {
        return ACLJCRAmetysObjectHelper.getProfilePermissionsForUsers(rootContexts, profileId);
    }
    
    public Map<Object, Map<GroupIdentity, Set<UserOrGroup>>> getAllAssignmentsForGroups(Set< ? extends Object> rootContexts, String profileId)
    {
        return ACLJCRAmetysObjectHelper.getProfilePermissionsForGroups(rootContexts, profileId);
    }
    
    /* ----------------------- */
    /* ALLOWED/DENIED PROFILES */
    /* ----------------------- */
    
    public Map<AnonymousOrAnyConnectedKeys, Set<String>> getProfilesForAnonymousAndAnyConnectedUser(Object object)
    {
        ACLAmetysObject ametysObj = (ACLAmetysObject) object;
        return ametysObj.getProfilesForAnonymousAndAnyConnectedUser();
    }

    public Map<UserIdentity, Map<UserOrGroup, Set<String>>> getProfilesForUsers(Object object, UserIdentity user)
    {
        ACLAmetysObject ametysObj = (ACLAmetysObject) object;
        return ametysObj.getProfilesForUsers(user);
    }

    public Map<GroupIdentity, Map<UserOrGroup, Set<String>>> getProfilesForGroups(Object object, Set<GroupIdentity> groups)
    {
        ACLAmetysObject ametysObj = (ACLAmetysObject) object;
        return ametysObj.getProfilesForGroups(groups);
    }
    
    /* ----------- */
    /* INHERITANCE */
    /* ----------- */

    public boolean isInheritanceDisallowed(Object object)
    {
        ACLAmetysObject ametysObj = (ACLAmetysObject) object;
        return ametysObj.isInheritanceDisallowed();
    }

    
    /* ------------------------------ */
    /* SUPPORT OF OBJECT AND PRIORITY */
    /* ------------------------------ */
    
    public boolean supports(Object object)
    {
        return object instanceof ACLAmetysObject;
    }
    
    public boolean isRootContextSupported(Object rootContext)
    {
        return rootContext instanceof JCRAmetysObject;
    }

    @Override
    public int getPriority()
    {
        // Minor priority than ModifiableACLAmetysObjectProfileAssignmentStorage
        return ProfileAssignmentStorage.MAX_PRIORITY + 100;
    }
}
