/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository;

import java.util.HashMap;
import java.util.Map;

import org.ametys.plugins.repository.jcr.JCRAmetysObjectFactory;
import org.ametys.runtime.plugin.ExtensionPoint;
import org.ametys.runtime.plugin.component.AbstractThreadSafeComponentExtensionPoint;

/**
 * {@link ExtensionPoint} hosting all {@link AmetysObjectFactory}.
 */
public class AmetysObjectFactoryExtensionPoint extends AbstractThreadSafeComponentExtensionPoint<AmetysObjectFactory>
{
    /** Avalon Role */
    public static final String ROLE = AmetysObjectFactoryExtensionPoint.class.getName();
    
    private Map<String, AmetysObjectFactory> _schemes = new HashMap<>();
    private Map<String, AmetysObjectFactory> _nodetypes = new HashMap<>();
    
    @Override
    public void initializeExtensions() throws Exception
    {
        super.initializeExtensions();
        
        for (String id : getExtensionsIds())
        {
            AmetysObjectFactory factory = getExtension(id);
            _schemes.put(factory.getScheme(), factory);
            
            if (factory instanceof JCRAmetysObjectFactory)
            {
                JCRAmetysObjectFactory<?> jcrFactory = (JCRAmetysObjectFactory) factory;
                
                for (String nodetype : jcrFactory.getNodetypes())
                {
                    if (_nodetypes.containsKey(nodetype))
                    {
                        throw new IllegalArgumentException("Nodetype '" + nodetype + "' cannot be managed by more than one factory.");
                    }
                    
                    _nodetypes.put(nodetype, factory);
                }
            }
        }
    }
    
    /**
     * Returns the {@link AmetysObjectFactory} corresponding to the given scheme.
     * @param scheme the scheme.
     * @return the {@link AmetysObjectFactory} or <code>null</code> if not found.
     */
    public AmetysObjectFactory getFactoryForScheme(String scheme)
    {
        return _schemes.get(scheme);
    }
    
    /**
     * Returns the {@link AmetysObjectFactory} corresponding to the given nodetype.
     * @param nodetype the nodetype.
     * @return the {@link AmetysObjectFactory} or <code>null</code> if not found.
     */
    public AmetysObjectFactory getFactoryForNodetype(String nodetype)
    {
        return _nodetypes.get(nodetype);
    }
}
