/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository;

import java.io.Closeable;
import java.util.Spliterator;
import java.util.Spliterators;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

/**
 * Allows iteration through {@link AmetysObject}s.<br>
 * It implements {@link Iterable}, allowing foreach statements.
 * @param <A> the actual type of {@link AmetysObject}s.
 */
public interface AmetysObjectIterable<A extends AmetysObject> extends Iterable<A>, Closeable
{
    /**
     * Returns the number of elements in this iterable.
     * If this information is unavailable, returns -1.
     * @return a long
     */
    long getSize();
    
    AmetysObjectIterator<A> iterator();
    
    /**
     * Close the associated resources.<br>
     * An {@link AmetysObjectIterable} must NOT be closed if any of the contained {@link AmetysObject} is still in use.
     */
    void close();
    
    /**
     * Returns a sequential {@code Stream} with this iterable as its source.
     * @return a sequential {@code Stream} over the objects in this iterable.
     */
    default Stream<A> stream()
    {
        return StreamSupport.stream(spliterator(), false);
    }
    
    @Override
    default Spliterator<A> spliterator()
    {
        long size = getSize();
        
        if (size < 0)
        {
            // Spliterator with unknown size.
            return Iterable.super.spliterator();
        }
        else
        {
            // Spliterator with known size.
            return Spliterators.spliterator(iterator(), size, 0);
        }
    }
}
