/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository;

import java.util.Iterator;
import java.util.NoSuchElementException;

/**
 * Iterates over {@link AmetysObject}s.
 * @param <A> the actual type of {@link AmetysObject}s.
 */
public interface AmetysObjectIterator<A extends AmetysObject> extends Iterator<A>
{
    /**
     * Returns the current position within the iterator. The number
     * returned is the 0-based index of the next element in the iterator,
     * i.e. the one that will be returned on the subsequent <code>next</code> call.
     * <br>
     * Note that this method does not check if there is a next element,
     * i.e. an empty iterator will always return 0.
     *
     * @return a long
     */
    long getPosition();
    
    /**
     * Returns the number of elements in the iterator.
     * If this information is unavailable, returns -1.
     * @return a long
     */
    long getSize();
    
    /**
     * Skip a number of elements in the iterator. <br>
     * The default implementation simply calls <code>skipNum</code> times {@link #next()}.
     * @param skipNum the non-negative number of elements to skip
     * @throws java.util.NoSuchElementException if skipped past the last element in the iterator.
     */
    default void skip(long skipNum)
    {
        if (skipNum == 0)
        {
            return;
        }
        
        if (skipNum < 0)
        {
            throw new AmetysRepositoryException("Cannot skip with a negative number");
        }
        
        long size = getSize();
        if (size != -1 && skipNum > size - getPosition())
        {
            throw new NoSuchElementException();
        }
        
        for (long i = skipNum; i > 0; i--)
        {
            next();
        }
    }
}
