/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.repository;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import javax.jcr.ItemExistsException;
import javax.jcr.NamespaceRegistry;
import javax.jcr.Node;
import javax.jcr.PathNotFoundException;
import javax.jcr.Repository;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.Value;
import javax.jcr.nodetype.NodeType;
import javax.jcr.query.Query;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;
import org.apache.jackrabbit.core.nodetype.InvalidNodeTypeDefException;
import org.apache.jackrabbit.core.nodetype.NodeTypeDefStore;
import org.apache.jackrabbit.core.nodetype.NodeTypeManagerImpl;
import org.apache.jackrabbit.core.nodetype.NodeTypeRegistry;
import org.apache.jackrabbit.spi.Name;
import org.apache.jackrabbit.spi.QNodeTypeDefinition;
import org.slf4j.Logger;

import org.ametys.core.util.LambdaUtils;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.plugins.repository.jcr.JCRAmetysObjectFactory;
import org.ametys.plugins.repository.jcr.NodeTypeHelper;
import org.ametys.plugins.repository.migration.jcr.repository.VersionsFactory;
import org.ametys.plugins.repository.provider.AbstractRepository;
import org.ametys.plugins.repository.provider.JackrabbitRepository;
import org.ametys.plugins.repository.virtual.VirtualAmetysObjectFactory;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Base component for accessing {@link AmetysObject}s.
 */
public class AmetysObjectResolver extends AbstractLogEnabled implements Serviceable, Initializable, Component
{
    /** Avalon ROLE. */
    public static final String ROLE = AmetysObjectResolver.class.getName();

    /** JCR Relative Path to root. */
    public static final String ROOT_REPO = "ametys:root";

    /** JCR type for root node. */
    public static final String ROOT_TYPE = "ametys:root";

    /** JCR mixin type for objects. */
    public static final String OBJECT_TYPE = "ametys:object";

    /** JCR property name for virtual objects. */
    public static final String VIRTUAL_PROPERTY = "ametys-internal:virtual";

    private AmetysObjectFactoryExtensionPoint _ametysFactoryExtensionPoint;
    private NamespacesExtensionPoint _namespacesExtensionPoint;
    private NodeTypeDefinitionsExtensionPoint _nodetypeDefsExtensionPoint;
    private Repository _repository;
    private SourceResolver _resolver;

    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _repository = (Repository) manager.lookup(AbstractRepository.ROLE);
        _ametysFactoryExtensionPoint = (AmetysObjectFactoryExtensionPoint) manager.lookup(AmetysObjectFactoryExtensionPoint.ROLE);
        _namespacesExtensionPoint = (NamespacesExtensionPoint) manager.lookup(NamespacesExtensionPoint.ROLE);
        _nodetypeDefsExtensionPoint = (NodeTypeDefinitionsExtensionPoint) manager.lookup(NodeTypeDefinitionsExtensionPoint.ROLE);
    }
    
    @Override
    public void initialize() throws Exception
    {
        // On vérifie que le root soit bien créé
        Session session = _repository.login();

        _initNamespaces(session);
        _initNodetypes(session);

        if (!session.getRootNode().hasNode(ROOT_REPO))
        {
            getLogger().info("Creating ametys root Node");
            
            session.getRootNode().addNode(ROOT_REPO, ROOT_TYPE);
            
            initRepoNodes(session, getLogger());
        }
        
        if (session.hasPendingChanges())
        {
            session.save();
        }
        
        session.logout();
        if (_repository instanceof JackrabbitRepository)
        {
            // Now that custom_nodetypes.xml have been (re)created, compare it with the previous version and delete the backup if they are the same
            ((JackrabbitRepository) _repository).compareCustomNodetypes();
        }
    }
    
    /**
     * Init the repository nodes directly inside ROOT node
     * @param session the session to use
     * @param logger the logger to log actions
     * @return true if the sesion needs changes
     * @throws RepositoryException something went wrong
     */
    public static boolean initRepoNodes(Session session, Logger logger) throws RepositoryException
    {
        // Create Ametys migration versions node if we just created the repository.
        logger.info("Creating ametys migration versions root Node");
        session.getRootNode().getNode(ROOT_REPO)
                    .addNode(VersionsFactory.VERSIONS_NODENAME, VersionsFactory.VERSIONS_NODETYPE);
        
        return session.hasPendingChanges();
    }
    
    private void _initNamespaces(Session session) throws RepositoryException
    {
        NamespaceRegistry registry = session.getWorkspace().getNamespaceRegistry();
        Collection prefixes = Arrays.asList(registry.getPrefixes());
        
        _namespacesExtensionPoint.getExtensionsIds().stream()
                                                    .filter(prefix -> !prefixes.contains(prefix))
                                                    .forEach(LambdaUtils.wrapConsumer(prefix ->
                                                    {
                                                        String namespace = _namespacesExtensionPoint.getNamespace(prefix);
                                                        getLogger().debug("Adding {} namespace", prefix);
                                                        registry.registerNamespace(prefix, namespace);
                                                    }));
    }

    private void _initNodetypes(Session session) throws RepositoryException, InvalidNodeTypeDefException, IOException
    {
        NodeTypeDefStore store = new NodeTypeDefStore();

        // Hard-coded first nodetypes file
        Source fsource = _resolver.resolveURI("plugin:repository://nodetypes/ametys_nodetypes.xml");
        try (InputStream is = fsource.getInputStream())
        {
            store.load(is);
        }
        finally
        {
            _resolver.release(fsource);
        }
        
        // Load all declared nodetype definitions in the store.
        for (String nodetypeDef : _nodetypeDefsExtensionPoint.getNodeTypeDefinitions())
        {
            Source source = _resolver.resolveURI(nodetypeDef);
            try (InputStream is = source.getInputStream())
            {
                store.load(is);
            }
            finally
            {
                _resolver.release(source);
            }
        }
        
        NodeTypeManagerImpl ntManager = (NodeTypeManagerImpl) session.getWorkspace().getNodeTypeManager();
        NodeTypeRegistry registry = ntManager.getNodeTypeRegistry();
        
        // Remove all already registered nodetypes from the store.
        for (Name name : registry.getRegisteredNodeTypes())
        {
            store.remove(name);
        }
        
        // Register the "new" nodetype definitions.
        Collection<QNodeTypeDefinition> ntDefs = store.all();
        if (!ntDefs.isEmpty())
        {
            registry.registerNodeTypes(ntDefs);
        }
    }

    /**
     * Retrieves an {@link AmetysObject} from an absolute path.
     * The given path is absolute in the Ametys tree.<br>
     * The path may omit the leading <code>'/'</code>, but the path
     * is always considered absolute, <code>null</code> path is forbidden.<br>
     * @param <A> the actual type of {@link AmetysObject}.
     * @param absolutePath the path to use.
     * @return the corresponding AmetysObject.
     * @throws AmetysRepositoryException if an error occurs.
     * @throws UnknownAmetysObjectException if no such object exists for the given path.
     * @deprecated Use resolveByPath instead
     */
    @Deprecated
    public <A extends AmetysObject> A resolve(String absolutePath) throws AmetysRepositoryException, UnknownAmetysObjectException
    {
        return resolveByPath(absolutePath);
    }
    
    /**
     * Retrieves an {@link AmetysObject} from an absolute path.
     * The given path is absolute in the Ametys tree.<br>
     * The path may omit the leading <code>'/'</code>, but the path
     * is always considered absolute, <code>null</code> path is forbidden.<br>
     * @param <A> the actual type of {@link AmetysObject}.
     * @param absolutePath the path to use.
     * @return the corresponding AmetysObject.
     * @throws AmetysRepositoryException if an error occurs.
     * @throws UnknownAmetysObjectException if no such object exists for the given path.
     */
    public <A extends AmetysObject> A resolveByPath(String absolutePath) throws AmetysRepositoryException, UnknownAmetysObjectException
    {
        return resolveByPath(absolutePath, null);
    }
    
    /**
     * Retrieves an {@link AmetysObject} from an absolute path.
     * The given path is absolute in the Ametys tree.<br>
     * The path may omit the leading <code>'/'</code>, but the path
     * is always considered absolute, <code>null</code> path is forbidden.<br>
     * @param <A> the actual type of {@link AmetysObject}.
     * @param absolutePath the path to use.
     * @param session the JCR Session to use to retrieve the {@link AmetysObject}.
     * @return the corresponding AmetysObject.
     * @throws AmetysRepositoryException if an error occurs.
     * @throws UnknownAmetysObjectException if no such object exists for the given path.
     */
    public <A extends AmetysObject> A resolveByPath(String absolutePath, Session session) throws AmetysRepositoryException, UnknownAmetysObjectException
    {
        if (getLogger().isDebugEnabled())
        {
            getLogger().debug("Resolving " + absolutePath);
        }
        
        if (absolutePath == null)
        {
            throw new AmetysRepositoryException("Absolute path cannot be null");
        }
        
        Node rootNode;
        Session jcrSession = null;
        try
        {
            jcrSession = session != null ? session : _repository.login();
            rootNode = jcrSession.getRootNode().getNode(ROOT_REPO);
        }
        catch (PathNotFoundException e)
        {
            if (session == null && jcrSession != null)
            {
                // logout only if the session was created here
                jcrSession.logout();
            }

            throw new AmetysRepositoryException("Unable to get ametys:root Node", e);
        }
        catch (RepositoryException e)
        {
            if (session == null && jcrSession != null)
            {
                // logout only if the session was created here
                jcrSession.logout();
            }

            throw new AmetysRepositoryException("An error occured while getting ametys:root node", e);
        }
        
        try
        {
            return this.<A>_resolve(null, rootNode, absolutePath, false);
        }
        catch (RepositoryException e)
        {
            if (session == null)
            {
                // logout only if the session was created here
                jcrSession.logout();
            }

            throw new AmetysRepositoryException("An error occured while resolving " + absolutePath, e);
        }
    }

    /**
     * Retrieves an {@link AmetysObject} by its unique id.
     * @param <A> the actual type of {@link AmetysObject}.
     * @param id the identifier representing the wanted {@link AmetysObject} is the Ametys repository.
     * @return the corresponding {@link AmetysObject}.
     * @throws AmetysRepositoryException if an error occurs.
     * @throws UnknownAmetysObjectException if no such object exists for the given id.
     */
    public <A extends AmetysObject> A resolveById(String id) throws AmetysRepositoryException, UnknownAmetysObjectException
    {
        if (getLogger().isDebugEnabled())
        {
            getLogger().debug("Resolving " + id);
        }
        
        if (StringUtils.isBlank(id))
        {
            throw new AmetysRepositoryException("An object id must conform to the <protocol>://<protocol-specific-part> syntax but id is blank or null");
        }
        
        int index = id.indexOf("://");
        if (index == -1)
        {
            throw new AmetysRepositoryException("An object id must conform to the <protocol>://<protocol-specific-part> syntax: " + id);
        }
        
        String scheme = id.substring(0, index);
        
        AmetysObjectFactory<A> factory = _ametysFactoryExtensionPoint.getFactoryForScheme(scheme);
        
        if (factory == null)
        {
            throw new UnknownAmetysObjectException("There's no object for id " + id);
        }
        
        return factory.getAmetysObjectById(id);
    }
    
    /**
     * <b>Expert</b>. Retrieves an {@link AmetysObject} by its unique id and the provided JCR Session.<br>
     * It only works with id corresponding to a {@link JCRAmetysObjectFactory}.<br>
     * This method should be uses to avoid useless Session creation.
     * @param <A> the actual type of {@link AmetysObject}.
     * @param id the identifier representing the wanted {@link AmetysObject} is the Ametys repository.
     * @param session the JCR Session to use to retrieve the {@link AmetysObject}.
     * @return the corresponding {@link AmetysObject}.
     * @throws AmetysRepositoryException if an error occurs.
     * @throws UnknownAmetysObjectException if no such object exists for the given id.
     * @throws RepositoryException if a JCR error occurs.
     */
    public <A extends AmetysObject> A resolveById(String id, Session session) throws AmetysRepositoryException, UnknownAmetysObjectException, RepositoryException
    {
        if (getLogger().isDebugEnabled())
        {
            getLogger().debug("Resolving " + id);
        }
        
        if (StringUtils.isBlank(id))
        {
            throw new AmetysRepositoryException("An object id must conform to the <protocol>://<protocol-specific-part> syntax but id is blank or null");
        }
        
        int index = id.indexOf("://");
        if (index == -1)
        {
            throw new AmetysRepositoryException("An object id must conform to the <protocol>://<protocol-specific-part> syntax: " + id);
        }
        
        String scheme = id.substring(0, index);
        
        AmetysObjectFactory<A> factory = _ametysFactoryExtensionPoint.getFactoryForScheme(scheme);
        
        if (factory == null)
        {
            throw new UnknownAmetysObjectException("There's no object for id " + id);
        }
        
        if (!(factory instanceof JCRAmetysObjectFactory))
        {
            throw new IllegalArgumentException("The expert method resolveById(String, Session) should only be called for id corresponding to a JCRAmetysObjectFactory");
        }
        
        return ((JCRAmetysObjectFactory<A>) factory).getAmetysObjectById(id, session);
    }

    /**
     * Return true if the specified id correspond to an existing {@link AmetysObject}.
     * @param id the identifier.
     * @return true if the specified id correspond to an existing {@link AmetysObject}.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public boolean hasAmetysObjectForId(String id) throws AmetysRepositoryException
    {
        int index = id.indexOf("://");
        if (index == -1)
        {
            throw new AmetysRepositoryException("An object id must conform to the <protocol>://<protocol-specific-part> syntax: " + id);
        }
        
        String scheme = id.substring(0, index);
        
        AmetysObjectFactory factory = _ametysFactoryExtensionPoint.getFactoryForScheme(scheme);
        
        if (factory == null)
        {
            return false;
        }
        
        return factory.hasAmetysObjectForId(id);
    }
    
    /**
     * <b>Expert</b>. Returns the {@link AmetysObject} corresponding to a given JCR Node.<br>
     * It is strictly equivalent to call <code>resolve(null, node, null, allowUnknownNode)</code>
     * @param <A> the actual type of {@link AmetysObject}s
     * @param node an existing node in the underlying JCR repository.
     * @param allowUnknownNode if <code>true</code>, returns <code>null</code> if the node type
     *                         does not correspond to a factory. If <code>false</code> and no factory
     *                         corresponds, an {@link AmetysRepositoryException} is thrown.
     * @return the {@link AmetysObject} corresponding to a given JCR node.
     * @throws AmetysRepositoryException if an error occurs.
     * @throws RepositoryException if a JCR error occurs.
     */
    public <A extends AmetysObject> A resolve(Node node, boolean allowUnknownNode) throws AmetysRepositoryException, RepositoryException
    {
        return this.<A>_resolve(null, node, null, allowUnknownNode);
    }
    
    /**
     * <b>Expert</b>. Retrieves an {@link AmetysObject}, given a JCR Node, a relative path
     * and the parentPath in the Ametys hierarchy.<br>
     * The path is always relative, even if it begins with a <code>'/'</code>,
     * <code>null</code> path or empty path are equivalent.<br>
     * May return null if ignoreUnknownNodes is true.
     * @param <A> the actual type of {@link AmetysObject}.
     * @param parentPath the parentPath of the returned AmetysObject, in the Ametys hierarchy.
     * @param node the context JCR node.
     * @param childPath the path relative to the JCR node.
     * @param allowUnknownNode if <code>true</code>, returns <code>null</code> if the node type
     *                         does not correspond to a factory. If <code>false</code> and no factory
     *                         corresponds, an {@link AmetysRepositoryException} is thrown.
     * @return the corresponding AmetysObject.
     * @throws AmetysRepositoryException if an error occurs.
     * @throws UnknownAmetysObjectException if no such object exists for the given path.
     * @throws RepositoryException if a JCR error occurs.
     */
    public <A extends AmetysObject> A resolve(String parentPath, Node node, String childPath, boolean allowUnknownNode) throws AmetysRepositoryException, UnknownAmetysObjectException, RepositoryException
    {
        return this.<A>_resolve(parentPath, node, childPath, allowUnknownNode);
    }
    
    @SuppressWarnings("unchecked")
    private <T extends AmetysObject> T _resolve(String parentPath, Node node, String childPath, boolean allowUnknownNode) throws AmetysRepositoryException, UnknownAmetysObjectException, RepositoryException
    {
        if (getLogger().isDebugEnabled())
        {
            getLogger().debug("Entering _resolve with parentPath=" + parentPath + ", node=" + node.getPath() + ", childPath=" + childPath + ", ignoreUnknownNodes=" + allowUnknownNode);
        }
        
        String path = childPath == null ? "" : childPath;
        path = path.length() == 0 || path.charAt(0) != '/' ? path : path.substring(1);

        if (path.length() != 0 && (Character.isSpaceChar(path.charAt(0)) || Character.isSpaceChar(path.charAt(path.length() - 1))))
        {
            throw new AmetysRepositoryException("Path cannot begin or end with a space character");
        }
        
        String nodeType = NodeTypeHelper.getNodeTypeName(node);
        
        JCRAmetysObjectFactory jcrFactory = _getJCRFactory(nodeType, allowUnknownNode, parentPath, childPath);
        
        if (jcrFactory == null)
        {
            return null;
        }
        
        AmetysObject rootObject = jcrFactory.getAmetysObject(node, parentPath);

        if (path.length() != 0)
        {
            if (!(rootObject instanceof TraversableAmetysObject))
            {
                throw new AmetysRepositoryException("The node of type '" + nodeType + "' at path '" + node.getPath() + "' does not corresponds to a TraversableAmetysObject");
            }

            return (T) ((TraversableAmetysObject) rootObject).getChild(path);
        }
        else
        {
            return (T) rootObject;
        }
    }
    
    
    private JCRAmetysObjectFactory _getJCRFactory(String nodeType, boolean allowUnknownNode, String parentPath, String childPath)
    {
        if (getLogger().isDebugEnabled())
        {
            getLogger().debug("Nodetype is " + nodeType);
        }

        AmetysObjectFactory<?> factory = _ametysFactoryExtensionPoint.getFactoryForNodetype(nodeType);
        
        if (factory == null)
        {
            if (allowUnknownNode)
            {
                if (getLogger().isDebugEnabled())
                {
                    getLogger().debug("No factory for nodetype " + nodeType + ". Unknown node is allowed, returning null.");
                }

                return null;
            }
            
            throw new UnknownAmetysObjectException("Cannot get factory for node '" + childPath +  "' under '" + parentPath + "': There's no factory for nodetype: " + nodeType);
        }

        if (getLogger().isDebugEnabled())
        {
            getLogger().debug("Factory is " + factory.getClass().getName());
        }

        if (!(factory instanceof JCRAmetysObjectFactory))
        {
            throw new AmetysRepositoryException("A factory resolving JCR nodes must implements JCRAmetysObjectFactory");
        }

        JCRAmetysObjectFactory jcrFactory = (JCRAmetysObjectFactory) factory;
        
        return jcrFactory;
    }
    
    /**
     * <b>Expert</b>. Retrieves the virtual children of a concrete JCR Node.<br>
     * @param <A> the actual type of {@link AmetysObject}s.
     * @param parent the {@link JCRAmetysObject} "hosting" the {@link VirtualAmetysObjectFactory} reference.
     * @return all virtual children under the given JCR Node in the Ametys hierarchy.
     * @throws AmetysRepositoryException if an error occurs.
     * @throws RepositoryException if a JCR error occurs.
     */
    public <A extends AmetysObject> AmetysObjectIterable<A> resolveVirtualChildren(JCRAmetysObject parent) throws AmetysRepositoryException, RepositoryException
    {
        Node contextNode = parent.getNode();
        
        if (getLogger().isDebugEnabled())
        {
            getLogger().debug("Entering resolveVirtualChildren with parent=" + parent);
        }

        if (!contextNode.hasProperty(VIRTUAL_PROPERTY))
        {
            return null;
        }
        
        Value[] values = contextNode.getProperty(VIRTUAL_PROPERTY).getValues();
        List<AmetysObjectIterable<A>> children = new ArrayList<>(values.length);
        for (Value value : values)
        {
            String id = value.getString();
            
            if (getLogger().isDebugEnabled())
            {
                getLogger().debug("Found virtual factory id: " + id);
            }

            AmetysObjectFactory<A> factory = _ametysFactoryExtensionPoint.getExtension(id);
            
            if (factory == null)
            {
                throw new AmetysRepositoryException("There's no virtual factory for id " + id);
            }
            
            if (getLogger().isDebugEnabled())
            {
                getLogger().debug("Found factory: " + factory.getClass().getName());
            }

            if (!(factory instanceof VirtualAmetysObjectFactory))
            {
                throw new AmetysRepositoryException("A factory handling virtual objects must implement VirtualAmetysObjectFactory");
            }
            
            VirtualAmetysObjectFactory<A> virtualFactory = (VirtualAmetysObjectFactory<A>) factory;
            children.add(virtualFactory.getChildren(parent));
        }
        
        return new ChainedAmetysObjectIterable<>(children);
    }
    
    /**
     * <b>Expert</b>. Retrieves the virtual child of a concrete JCR Node.<br>
     * @param parent the {@link JCRAmetysObject} "hosting" the {@link VirtualAmetysObjectFactory} reference.
     * @param childPath  the name of the virtual child.
     * @return a named child under the given JCR Node in the Ametys hierarchy.
     * @throws AmetysRepositoryException if an error occurs.
     * @throws RepositoryException if a JCR error occurs.
     * @throws UnknownAmetysObjectException if the named child does not exist
     */
    public AmetysObject resolveVirtualChild(JCRAmetysObject parent, String childPath) throws AmetysRepositoryException, RepositoryException, UnknownAmetysObjectException
    {
        Node contextNode = parent.getNode();
        
        if (getLogger().isDebugEnabled())
        {
            getLogger().debug("Entering resolveVirtualChild with parent=" + parent);
        }

        if (!contextNode.hasProperty(VIRTUAL_PROPERTY))
        {
            throw new UnknownAmetysObjectException("There's no virtual child at Ametys path " + parent.getPath());
        }
        
        String path = childPath == null ? "" : childPath;
        path = path.length() == 0 || path.charAt(0) != '/' ? path : path.substring(1);
        int index = path.indexOf('/');
        String childName = index == -1 ? path : path.substring(0, index);
        String subPath = index == -1 ? null : path.substring(index + 1);

        if (childName.length() == 0)
        {
            throw new AmetysRepositoryException("A path element cannot be empty in " + childPath);
        }
        else if (Character.isSpaceChar(path.charAt(0)) || Character.isSpaceChar(path.charAt(path.length() - 1)))
        {
            throw new AmetysRepositoryException("Path element cannot begin or end with a space character: " + childName);
        }
        
        Value[] values = contextNode.getProperty(VIRTUAL_PROPERTY).getValues();
        AmetysObject object = _getVirtualChild(parent, childName, values);
        
        if (object == null)
        {
            throw new UnknownAmetysObjectException("There's no virtual object named " + childName + " at Ametys path " + parent.getPath());
        }
        
        if (subPath != null)
        {
            if (!(object instanceof TraversableAmetysObject))
            {
                throw new AmetysRepositoryException("The virtual object " + childName + "at path '" + childPath + "' does not corresponds to a TraversableAmetysObject");
            }

            return ((TraversableAmetysObject) object).getChild(subPath);
        }
        else
        {
            return object;
        }
    }

    /**
     * Executes the given JCR XPath query and resolves results as
     * {@link AmetysObject}s.<br>
     * The resulting {@link AmetysObjectIterable} supports lazy loading, but
     * will also fail lazily if one if the result nodes does not correspond to
     * an {@link AmetysObject}.
     * @param <A> the actual type of the results.
     * @param jcrQuery a JCR XPath query.
     * @return an Iterator over the resulting {@link AmetysObject}.
     */
    public <A extends AmetysObject> AmetysObjectIterable<A> query(String jcrQuery)
    {
        Session session = null;
        try
        {
            session = _repository.login();
            return query(jcrQuery, session);
        }
        catch (RepositoryException ex)
        {
            if (session != null)
            {
                session.logout();
            }

            throw new AmetysRepositoryException("An error occured executing the JCR query : " + jcrQuery, ex);
        }
    }
    
    /**
     * <b>Expert</b>. Executes the given JCR XPath query with the provided JCR Session and resolves results as
     * {@link AmetysObject}s.<br>
     * The resulting {@link AmetysObjectIterable} supports lazy loading, but
     * will also fail lazily if one if the result nodes does not correspond to
     * an {@link AmetysObject}.
     * @param <A> the actual type of the results.
     * @param jcrQuery a JCR XPath query.
     * @param session the JCR Session to use to execute the request.
     * @return an Iterator over the resulting {@link AmetysObject}.
     * @throws RepositoryException if a JCR error occurs.
     */
    @SuppressWarnings("deprecation")
    public <A extends AmetysObject> AmetysObjectIterable<A> query(String jcrQuery, Session session) throws RepositoryException
    {
        if (getLogger().isDebugEnabled())
        {
            getLogger().debug("Executing XPath query: '" + jcrQuery + "'");
        }
        
        Query query = session.getWorkspace().getQueryManager().createQuery(jcrQuery, Query.XPATH);

        long t1 = System.currentTimeMillis();
        AmetysObjectIterable<A> it = new NodeIteratorIterable<>(this, query.execute().getNodes(), null, session);
        
        if (getLogger().isInfoEnabled())
        {
            getLogger().info("JCR query '" + jcrQuery + "' executed in " + (System.currentTimeMillis() - t1) + " ms");
        }
        
        return it;
    }

    private AmetysObject _getVirtualChild(JCRAmetysObject parent, String childName, Value[] values) throws RepositoryException
    {
        int i = 0;
        AmetysObject object = null;
        
        while (object == null && i < values.length)
        {
            Value value = values[i];
            String id = value.getString();

            if (getLogger().isDebugEnabled())
            {
                getLogger().debug("Found virtual factory id: " + id);
            }

            AmetysObjectFactory factory = _ametysFactoryExtensionPoint.getExtension(id);
            
            if (factory == null)
            {
                throw new AmetysRepositoryException("There's no virtual factory for id " + id);
            }
            
            if (getLogger().isDebugEnabled())
            {
                getLogger().debug("Found factory: " + factory.getClass().getName());
            }

            if (!(factory instanceof VirtualAmetysObjectFactory))
            {
                throw new AmetysRepositoryException("A factory handling virtual objects must implement VirtualAmetysObjectFactory: " + id);
            }
            
            VirtualAmetysObjectFactory virtualFactory = (VirtualAmetysObjectFactory) factory;
            
            try
            {
                object = virtualFactory.getChild(parent, childName);
            }
            catch (UnknownAmetysObjectException e)
            {
                // Not an error
                if (getLogger().isDebugEnabled())
                {
                    getLogger().debug("The factory: " + factory.getClass().getName() + " has no child named" + childName, e);
                }

                i++;
            }
        }
        
        return object;
    }
    
    /**
     * <b>Expert</b>. Creates a child object in the JCR tree and resolve it to an {@link AmetysObject}.
     * @param <A> the actual type of {@link AmetysObject}s
     * @param parentPath the parentPath of the new object.
     * @param parentNode the parent JCR Node of the new object.
     * @param childName the name of the new object.
     * @param nodetype the type of the Node backing the new object.
     * @return the newly created {@link AmetysObject}.
     * @throws AmetysRepositoryException if an error occurs.
     * @throws RepositoryIntegrityViolationException if an object with the same name already
     *         exists and same name siblings is not allowed.
     * @throws RepositoryException if a JCR error occurs.
     */
    public <A extends AmetysObject> A createAndResolve(String parentPath, Node parentNode, String childName, String nodetype) throws AmetysRepositoryException, RepositoryIntegrityViolationException, RepositoryException
    {
        if (getLogger().isDebugEnabled())
        {
            getLogger().debug("Entering createAndResolve with parentPath=" + parentPath + ", parentNode=" + parentNode.getPath() + ", childName=" + childName + ", nodetype=" + nodetype);
        }

        if (_ametysFactoryExtensionPoint.getFactoryForNodetype(nodetype) == null)
        {
            throw new AmetysRepositoryException("Cannot create a node '" + childName +  "' under '" + parentPath + "': There's no factory for nodetype: " + nodetype);
        }
        
        try
        {
            Node node = parentNode.addNode(childName, nodetype);
            NodeType[] mixinNodeTypes = node.getMixinNodeTypes();
            boolean foundMixin = false;
            
            int i = 0;
            while (!foundMixin && i < mixinNodeTypes.length)
            {
                if (OBJECT_TYPE.equals(mixinNodeTypes[i].getName()))
                {
                    foundMixin = true;
                }
                
                i++;
            }
            
            if (!foundMixin)
            {
                node.addMixin(OBJECT_TYPE);
            }
            
            return this.<A>resolve(parentPath, node, null, false);
        }
        catch (ItemExistsException e)
        {
            throw new RepositoryIntegrityViolationException("The object " + childName + " already exist at path " + parentPath, e);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to add child node for the underlying node for object at path " + parentPath, e);
        }
    }
}
