/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository;

import java.util.List;
import java.util.NoSuchElementException;
import java.util.stream.Collectors;

/**
 * Implementation of a {@link AmetysObjectIterable} based on a list of others {@link AmetysObjectIterable}.
 * @param <A> the actual type of {@link AmetysObject}s.
 */
public class ChainedAmetysObjectIterable<A extends AmetysObject> implements AmetysObjectIterable<A>
{
    long _size;
    private List<AmetysObjectIterable<A>> _iterables;
    private boolean _sizeComputed;
    
    /**
     * Creates a {@link ChainedAmetysObjectIterable}.
     * @param iterables a list of {@link AmetysObjectIterable}.
     */
    public ChainedAmetysObjectIterable(List<AmetysObjectIterable<A>> iterables)
    {
        _iterables = iterables;
    }
    
    public long getSize()
    {
        if (!_sizeComputed)
        {
            _size = _computeSize();
        }
        
        return _size;
    }
    
    private long _computeSize()
    {
        _sizeComputed = true;
        
        long size = 0;
        
        for (AmetysObjectIterable it : _iterables)
        {
            long s = it.getSize();
            
            if (s == -1)
            {
                return -1;
            }
            
            size += s;
        }
        
        return size;
    }
    
    public AmetysObjectIterator<A> iterator()
    {
        List<AmetysObjectIterator<A>> its = _iterables.stream().map(it -> it.iterator()).collect(Collectors.toList());
        return new ChainedIterator(its);
    }
    
    public void close()
    {
        for (AmetysObjectIterable<A> it : _iterables)
        {
            it.close();
        }
    }
    
    class ChainedIterator implements AmetysObjectIterator<A>
    {
        private long _position;
        private int _currentIterator;
        private List<AmetysObjectIterator<A>> _its;
        
        public ChainedIterator(List<AmetysObjectIterator<A>> its)
        {
            _its = its;
        }
        
        public long getPosition()
        {
            return _position;
        }
        
        public long getSize()
        {
            return ChainedAmetysObjectIterable.this.getSize();
        }
        
        public void skip(long skipNum)
        {
            if (skipNum == 0)
            {
                return;
            }
            
            if (skipNum < 0)
            {
                throw new AmetysRepositoryException("Cannot skip with a negative number");
            }
            
            for (long i = skipNum; i > 0; i--)
            {
                if (!hasNext())
                {
                    throw new NoSuchElementException();
                }
                
                _position++;
                _its.get(_currentIterator).skip(1);
            }
        }

        public boolean hasNext()
        {
            while (_currentIterator < _its.size() && !_its.get(_currentIterator).hasNext())
            {
                _currentIterator++;
            }
            
            return _currentIterator < _its.size() && _its.get(_currentIterator).hasNext();
        }
        
        public A next()
        {
            if (!hasNext())
            {
                throw new NoSuchElementException();
            }
            
            _position++;
            return _its.get(_currentIterator).next();
        }
    }
}
