/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository;

import java.util.Collection;
import java.util.Iterator;

/**
 * Implementation of an {@link AmetysObjectIterable} based on a {@link Collection} of AmetysObject IDs.
 * The AmetysObjects are resolved when next() is called.
 * @param <A> the actual type of {@link AmetysObject}s
 */
public class IdCollectionIterable<A extends AmetysObject> implements AmetysObjectIterable<A>
{
    AmetysObjectResolver _resolver;
    private Collection<String> _ids;
    
    /**
     * Creates a {@link IdCollectionIterable}.
     * @param resolver The AmetysObject resolver.
     * @param ids the {@link AmetysObject}s collection
     */
    public IdCollectionIterable(AmetysObjectResolver resolver, Collection<String> ids)
    {
        _resolver = resolver;
        _ids = ids;
    }
    
    @Override
    public long getSize()
    {
        return _ids.size();
    }
    
    @Override
    public AmetysObjectIterator<A> iterator()
    {
        return new IdIterator(_ids.iterator(), _ids.size());
    }
    
    @Override
    public void close()
    {
        // Nothing to do.
    }
    
    class IdIterator implements AmetysObjectIterator<A>
    {
        private Iterator<String> _it;
        private int _pos;
        private long _size;
        
        public IdIterator(Iterator<String> it, long size)
        {
            _it = it;
            _size = size;
        }
        
        @Override
        public long getSize()
        {
            return _size;
        }
        
        @Override
        public long getPosition()
        {
            return _pos;
        }
        
        @Override
        public boolean hasNext()
        {
            return _it.hasNext();
        }
        
        @Override
        public A next()
        {
            _pos++;
            return _resolver.resolveById(_it.next());
        }
        
        @Override
        public void remove()
        {
            _it.remove();
        }
    }
}
