/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository;

import java.util.Set;

import org.ametys.core.group.GroupIdentity;
import org.ametys.core.user.UserIdentity;

/**
 * {@link ACLAmetysObject} that can store and modify its profile assignements.
 */
public interface ModifiableACLAmetysObject extends ACLAmetysObject
{
    /**
     * Adds allowed profiles any connected user has on this ametys object
     * @param profileIds The profiles to add
     */
    public void addAllowedProfilesForAnyConnectedUser(Set<String> profileIds);
    
    /**
     * Removes allowed profiles any connected user has on this ametys object
     * @param profileIds The profiles to remove
     */
    public void removeAllowedProfilesForAnyConnectedUser(Set<String> profileIds);
    
    /**
     * Adds denied profiles any connected user has on this ametys object
     * @param profileIds The profiles to add
     */
    public void addDeniedProfilesForAnyConnectedUser(Set<String> profileIds);
    
    /**
     * Removes denied profiles any connected user has on this ametys object
     * @param profileIds The profiles to remove
     */
    public void removeDeniedProfilesForAnyConnectedUser(Set<String> profileIds);
    
    /**
     * Adds allowed profiles an anonymous user has on this ametys object
     * @param profileIds The profiles to add
     */
    public void addAllowedProfilesForAnonymous(Set<String> profileIds);
    
    /**
     * Removes allowed profiles an anonymous user has on this ametys object
     * @param profileIds The profiles to remove
     */
    public void removeAllowedProfilesForAnonymous(Set<String> profileIds);
    
    /**
     * Adds denied profiles an anonymous user has on this ametys object
     * @param profileIds The profiles to add
     */
    public void addDeniedProfilesForAnonymous(Set<String> profileIds);
    
    /**
     * Removes denied profiles an anonymous user has on this ametys object
     * @param profileIds The profiles to remove
     */
    public void removeDeniedProfilesForAnonymous(Set<String> profileIds);
    
    /**
     * Associates some users with an allowed profile on this ametys object
     * @param users The users to add
     * @param profileId The id of the profile
     */
    public void addAllowedUsers(Set<UserIdentity> users, String profileId);
    
    /**
     * Removes the association between some users and an allowed profile on this ametys object
     * @param users The users to remove
     * @param profileId The id of the profile
     */
    public void removeAllowedUsers(Set<UserIdentity> users, String profileId);
    
    /**
     * Removes the association between some users and all allowed profiles on this ametys object
     * @param users The users to remove
     */
    public void removeAllowedUsers(Set<UserIdentity> users);
    
    /**
     * Associates some groups with an allowed profile on this ametys object
     * @param groups The groups to add
     * @param profileId The id of the profile
     */
    public void addAllowedGroups(Set<GroupIdentity> groups, String profileId);
    
    /**
     * Removes the association between some groups and an allowed profile on this ametys object
     * @param groups The groups to remove
     * @param profileId The id of the profile
     */
    public void removeAllowedGroups(Set<GroupIdentity> groups, String profileId);
    
    /**
     * Removes the association between some groups and all allowed profiles on this ametys object
     * @param groups The groups to remove
     */
    public void removeAllowedGroups(Set<GroupIdentity> groups);
    
    /**
     * Associates some users with a denied profile on this ametys object
     * @param users The users to add
     * @param profileId The id of the profile
     */
    public void addDeniedUsers(Set<UserIdentity> users, String profileId);
    
    /**
     * Removes the association between some users and an denied profile on this ametys object
     * @param users The users to remove
     * @param profileId The id of the profile
     */
    public void removeDeniedUsers(Set<UserIdentity> users, String profileId);
    
    /**
     * Removes the association between some users and all denied profiles on this ametys object
     * @param users The users to remove
     */
    public void removeDeniedUsers(Set<UserIdentity> users);
    
    /**
     * Associates some groups with an allowed profile on this ametys object
     * @param groups The groups to add
     * @param profileId The id of the profile
     */
    public void addDeniedGroups(Set<GroupIdentity> groups, String profileId);
    
    /**
     * Removes the association between some groups and an allowed profile on this ametys object
     * @param groups The groups to remove
     * @param profileId The id of the profile
     */
    public void removeDeniedGroups(Set<GroupIdentity> groups, String profileId);
    
    /**
     * Removes the association between some groups and all allowed profiles on this ametys object
     * @param groups The groups to remove
     */
    public void removeDeniedGroups(Set<GroupIdentity> groups);
    
    /**
     * Allow or disallow inheritance of permissions
     * @param disallow true to disallow inheritance
     */
    public void disallowInheritance(boolean disallow);
}
