/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.repository;

/**
 * {@link AmetysObject} that can be ordered or moved
 */
public interface MovableAmetysObject extends AmetysObject
{
    /**
     * Move the current object as a new child of the given object. This node will be the last child. You should call canMoveTo to know if this is a supported operation.
     * @param newParent The new parent for the current object. Can not be null. Can not be a child of the current node. Must be a TraversableAmetyObject.
     * @param renameIfExist true to rename moved page if a page with same name already exist
     * @throws AmetysRepositoryException if an error occurs.
     * @throws RepositoryIntegrityViolationException if a page with the same name already exists.
     */
    public void moveTo(AmetysObject newParent, boolean renameIfExist) throws AmetysRepositoryException, RepositoryIntegrityViolationException;

    /**
     * Test if a move can be a success or if it is impossible (e.g. due to the implementation of the target)
     * @param newParent See moveTo.
     * @return true if the move operation may succeed. If false is returned and you call moveTo anyway, you may encontered a RuntimeException (such as UnsupportedOperationException)
     * @throws AmetysRepositoryException if an error occurs.
     */
    public boolean canMoveTo(AmetysObject newParent) throws AmetysRepositoryException;
    
    /**
     * Order a node before another sibling node (or as the last node)
     * @param siblingNode The node that will be the next sibling node of the current node. Must have the same parent as the current node. Can be null to set the current node as the last node.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public void orderBefore(AmetysObject siblingNode) throws AmetysRepositoryException;
}
