/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;

import org.ametys.runtime.plugin.ExtensionPoint;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * {@link ExtensionPoint} hosting the JCR namespaces.
 */
public class NamespacesExtensionPoint extends AbstractLogEnabled implements ExtensionPoint<String>, Component
{
    /** Avalon Role */
    public static final String ROLE = NamespacesExtensionPoint.class.getName();
    
    private Map<String, String> _namespaces = new HashMap<>();
    
    @Override
    public void addExtension(String id, String pluginName, String featureName, Configuration configuration) throws ConfigurationException
    {
        String namespace = configuration.getChild("namespace", true).getValue(null);
        
        // Add the nodetype definition 
        if (namespace != null)
        {
            if (_namespaces.containsKey(id))
            {
                getLogger().error("The namespace for prefix " + id + " is already defined. It will be ignored");
            }
            else
            {
                _namespaces.put(id, namespace);
            }
        }
    }

    @Override
    public void initializeExtensions() throws Exception
    {
        // Empty
    }

    @Override
    public boolean hasExtension(String id)
    {
        return _namespaces.containsKey(id);
    }

    @Override
    public String getExtension(String id)
    {
        if (hasExtension(id))
        {
            return _namespaces.get(id);
        }
        
        return null;
    }

    @Override
    public Set<String> getExtensionsIds()
    {
        return _namespaces.keySet();
    }
    
    /**
     * Returns the namespace for the given prefix or null if none.
     * @param prefix the prefix.
     * @return the namespace for the given prefix or null if none.
     */
    public String getNamespace(String prefix)
    {
        return _namespaces.get(prefix);
    }
}
