/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.repository;

import java.util.ArrayList;
import java.util.List;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

/**
 * Implementation of {@link AmetysObjectIterable} based on a underlying JCR {@link NodeIterator}.<br><br>
 * <b>Please note that this implementation only works with NodeIterators containing nodes actually bound to AmetysObjects.<br>
 * If a node exists in the iterator and does not correspond to an {@link AmetysObject}, an {@link AmetysRepositoryException} will be thrown.</b><br>
 * Unless {@link AmetysObjectIterator#skip(long)} is called on the underlying Iterator, 
 * results are buffered so that one may safely call {@link #iterator()} more than once, 
 * even if the underlying {@link NodeIterator} will be consumed only once.<br>
 * If {@link AmetysObjectIterator#skip(long)} is called, any subsequent call to {@link #iterator()} will throw an {@link IllegalStateException}. 
 * @param <A> the actual type of {@link AmetysObject}s.
 */
public class NodeIteratorIterable<A extends AmetysObject> implements AmetysObjectIterable<A>
{
    AmetysObjectResolver _resolver;
    String _parentPath;
    Session _session;
    List<A> _buffer = new ArrayList<>();
    boolean _skipCalled;
    private NodeIterator _iterator;
    
    
    /**
     * Creates a {@link NodeIteratorIterable}.
     * @param resolver the application {@link AmetysObjectResolver}.
     * @param iterator the underlying {@link NodeIterator}.
     * @param parentPath the parent path in the Ametys hierarchy of all
     *                   {@link AmetysObject} being returned.
     * @param session the JCR Session corresponding to the specified NodeIterator
     */
    public NodeIteratorIterable(AmetysObjectResolver resolver, NodeIterator iterator, String parentPath, Session session)
    {
        _resolver = resolver;
        _iterator = iterator;
        _parentPath = parentPath;
        _session  = session;
    }
    
    public long getSize()
    {
        return _iterator.getSize();
    }
    
    public AmetysObjectIterator<A> iterator()
    {
        if (_skipCalled)
        {
            throw new IllegalStateException("iterator() cannot be called on NodeIteratorIterable after skip() have been called. Results would be inconsistent.");
        }
        
        return new NodeIteratorIterator(_iterator);
    }
    
    public void close()
    {
        if (_session != null && _session.isLive())
        {
            _session.logout();
        }
    }
    
    class NodeIteratorIterator implements AmetysObjectIterator<A>
    {
        private NodeIterator _it;
        private int _position;
        
        public NodeIteratorIterator(NodeIterator it)
        {
            _it = it;
        }
        
        public long getPosition()
        {
            return _position;
        }
        
        public long getSize()
        {
            return _it.getSize();
        }
        
        public void skip(long skipNum)
        {
            if (skipNum > _buffer.size() - _position)
            {
                _skipCalled = true;
                _it.skip(skipNum - (_buffer.size() - _position));
                
                for (int i = _buffer.size() - _position; i < skipNum; i++)
                {
                    _buffer.add(null);
                }
            }
            
            _position += skipNum;
        }
        
        public boolean hasNext()
        {
            return _position < _buffer.size() || _it.hasNext();
        }
        
        public A next()
        {
            if  (_position < _buffer.size())
            {
                return _buffer.get(_position++);
            }
            
            Node node = _it.nextNode();
            _position++;
            
            A result = null;
            try
            {
                result = _resolver.resolve(_parentPath, node, null, false);
                return result;
            }
            catch (RepositoryException e)
            {
                throw new AmetysRepositoryException("An error occured while resolving", e);
            }
            finally
            {
                // add something to the buffer, even if its null, 
                // so that the buffer size matches the current position
                _buffer.add(result);
            }
        }
        
        public void remove()
        {
            _it.remove();
        }
    }
}
