/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository;

import java.util.NoSuchElementException;

/**
 * {@link AmetysObject} which is traversable and allow to create children.
 */
public interface TraversableAmetysObject extends AmetysObject
{
    /**
     * Retrieves children of the current object.
     * @param <A> the actual type of {@link AmetysObject}s.
     * @return the children or an empty {@link AmetysObjectIterable}.
     * @throws AmetysRepositoryException if an error occurs.
     */
    <A extends AmetysObject> AmetysObjectIterable<A> getChildren() throws AmetysRepositoryException;

    /**
     * Tests if this Object has at least one child with the given name.<br>
     * @param name the name to test. Cannot be <code>null</code> nor empty nor
     * begin with a <code>'/'</code> and it cannot contain <code>'/'</code>.
     * @return <code>true</code> if the named child exists,
     * <code>false</code> otherwise.
     * @throws AmetysRepositoryException if an error occurs.
     */
    boolean hasChild(String name) throws AmetysRepositoryException;

    /**
     * Retrieves a given child from its relative path.
     * The path cannot be <code>null</code>, empty nor begin with a <code>'/'</code>.
     * @param <A> the actual type of {@link AmetysObject}.
     * @param path the path of the child which can contains a position with.
     * <code>[n]</code> if same name sibling is allowed.
     * @return the child found.
     * @throws AmetysRepositoryException if an error occurs.
     * @throws UnknownAmetysObjectException if the object does not exist.
     */
    <A extends AmetysObject> A getChild(String path) throws AmetysRepositoryException, UnknownAmetysObjectException;
    
    /**
     * Gets the child {@link AmetysObject} at the given position
     * @param <A> The type of children objects
     * @param index the position of the {@link AmetysObject} within the ordered set of its sibling objects.
     * @return the {@link AmetysObject} if found
     * @throws AmetysRepositoryException if an error occurs
     * @throws UnknownAmetysObjectException if no object was found at this position
     */
    @SuppressWarnings("unchecked")
    default <A extends AmetysObject> A getChildAt(long index) throws AmetysRepositoryException, UnknownAmetysObjectException
    {
        if (index < 0)
        {
            throw new AmetysRepositoryException("Child index cannot be negative");
        }
        
        AmetysObjectIterable<AmetysObject> children = getChildren();
        AmetysObjectIterator<AmetysObject> it = children.iterator();
        
        try
        {
            it.skip(index);
            return (A) it.next();
        }
        catch (NoSuchElementException e)
        {
            throw new UnknownAmetysObjectException("There's no child at index " + index + " for object " + this.getId());
        }
    }
    
    /**
     * Returns the position of this {@link AmetysObject} within the ordered set of its sibling objects.
     * @param ao The Ametys object
     * @return the position of this  {@link AmetysObject} within the ordered set of its sibling objects.
     * @throws AmetysRepositoryException if an error occurs.
     */
    default long getChildPosition(AmetysObject ao) throws AmetysRepositoryException
    {
        AmetysObjectIterable<AmetysObject> children = getChildren();
        
        long count = 0;
        for (AmetysObject child : children)
        {
            if (child.getId().equals(ao.getId()))
            {
                return count;
            }
            count++;
        }
        
        // Not found
        return -1;
    }
}
