/*
 *  Copyright 2012 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository;

import java.util.HashSet;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * Implementation of a {@link AmetysObjectIterable} based on a list of others {@link AmetysObjectIterable},
 * returning duplicate objects only once.
 * @param <A> the actual type of {@link AmetysObject}s.
 */
public class UniqueChainedAmetysObjectIterable<A extends AmetysObject> implements AmetysObjectIterable<A>
{
    private List<AmetysObjectIterable<A>> _iterables;
    
    /**
     * Creates a {@link UniqueChainedAmetysObjectIterable}.
     * @param iterables a list of {@link AmetysObjectIterable}.
     */
    public UniqueChainedAmetysObjectIterable(List<AmetysObjectIterable<A>> iterables)
    {
        _iterables = iterables;
    }
    
    @Override
    public long getSize()
    {
        return -1;
    }
    
    public AmetysObjectIterator<A> iterator()
    {
        List<AmetysObjectIterator<A>> its = _iterables.stream().map(it -> it.iterator()).collect(Collectors.toList());
        return new UniqueChainedIterator(its);
    }
    
    public void close()
    {
        for (AmetysObjectIterable<A> it : _iterables)
        {
            it.close();
        }
    }
    
    class UniqueChainedIterator implements AmetysObjectIterator<A>
    {
        private Set<String> _identifiers = new HashSet<>(); 
        private A _nextObject;
        private boolean _nextObjectSet;
        private long _position;
        private int _currentIterator;
        private List<AmetysObjectIterator<A>> _its;
        
        public UniqueChainedIterator(List<AmetysObjectIterator<A>> its)
        {
            _its = its;
        }
        
        @Override
        public long getSize()
        {
            return -1;
        }
        
        public long getPosition()
        {
            return _position;
        }
        
        @Override
        public boolean hasNext()
        {
            if (_nextObjectSet)
            {
                return true;
            }
            else
            {
                return _fetchNextObject();
            }
        }
        
        @Override
        public A next()
        {
            if (!_nextObjectSet)
            {
                if (!_fetchNextObject())
                {
                    throw new NoSuchElementException();
                }
            }
            
            _nextObjectSet = false;
            _position++;
            return _nextObject;
        }
        
        /**
         * Set nextObject to the next object. 
         * @return If there are no more objects then return false. Otherwise, return true.
         */
        private boolean _fetchNextObject()
        {
            while (_internalHasNext())
            {
                A object = _internalNext();
                // Will return true if the object did not exist.
                if (_identifiers.add(object.getId()))
                {
                    _nextObject = object;
                    _nextObjectSet = true;
                    return true;
                }
            }
            return false;
        }
        
        private boolean _internalHasNext()
        {
            while (_currentIterator < _its.size() && !_its.get(_currentIterator).hasNext())
            {
                _currentIterator++;
            }
            
            return _currentIterator < _its.size() && _its.get(_currentIterator).hasNext();
        }
        
        private A _internalNext()
        {
            return _its.get(_currentIterator).next();
        }
    }
}
