/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.activities;

import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.jcr.RepositoryException;

import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.observation.Event;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.core.user.UserHelper;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.type.DataContext;
import org.ametys.runtime.model.type.ElementType;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.runtime.plugin.component.PluginAware;

/**
 * Abstract implementation for {@link ActivityType} storing activity in JCR
 */
public abstract class AbstractActivityType extends AbstractLogEnabled implements Configurable, ActivityType, Serviceable, PluginAware
{
    /** The default time range to merge compatible activities : 5 minutes */
    private static final int __MAX_MERGE_ACTIVITY_RANGE = 5 * 60;
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    /** Helper to get users */
    protected UserHelper _userHelper;
    
    /** The plugin name */
    protected String _pluginName;

    private Map<String, I18nizableText> _supportedEventTypes;
    private String _id;
    
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
    }
    
    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        _supportedEventTypes = new HashMap<>();
        
        _id = configuration.getAttribute("id");
        
        Configuration[] supportedTypesConf = configuration.getChild("supported-types").getChildren("supported-type");
        if (supportedTypesConf.length == 0)
        {
            throw new ConfigurationException("Missing 'supported-types' configuration", configuration);
        }
        
        for (Configuration conf : supportedTypesConf)
        {
            String id = conf.getAttribute("id");
            I18nizableText label = I18nizableText.parseI18nizableText(conf.getChild("label"), "plugin." + _pluginName, id);
            _supportedEventTypes.put(id, label);
        }
    }
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _currentUserProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
        _userHelper = (UserHelper) serviceManager.lookup(UserHelper.ROLE);
    }

    public boolean supports(Event element)
    {
        // Support all events
        return true;
    }
    
    public String getId()
    {
        return _id;
    }
    
    @Override
    public Map<String, I18nizableText> getSupportedEventTypes()
    {
        return _supportedEventTypes;
    }
    
    @Override
    public boolean isMergeable (Activity activity1, Activity activity2)
    {
        if (activity1 == null || activity2 == null)
        {
            return false;
        }
        
        String type1 = activity1.getEventType();
        UserIdentity author1 = activity1.getAuthor();
        ZonedDateTime date1 = activity1.getDate();
        
        if (type1 == null || author1 == null || date1 == null)
        {
            getLogger().error("Mandatory type, date or author is missing for activity '{}'. Consider it as no mergeable", activity1.getId());
            return false;
        }
        
        String type2 = activity2.getEventType();
        UserIdentity author2 = activity2.getAuthor();
        ZonedDateTime date2 = activity2.getDate();
        
        if (type2 == null || author2 == null || date2 == null)
        {
            getLogger().error("Mandatory type, date or author is missing for activity '{}'. Consider it as no mergeable", activity2.getId());
            return false;
        }
        
        return type1.equals(type2) && author1.equals(author2) && Math.abs(ChronoUnit.SECONDS.between(date1, date2)) < __MAX_MERGE_ACTIVITY_RANGE;
    }
    
    @Override
    public Map<String, Object> mergeActivities(List<Activity> activities)
    {
        Map<String, Object> mergedActivity = activities.get(0).toJSONForClient();
        
        mergedActivity.put("amount", activities.size());
        
        // Date range
        if (activities.size() > 1)
        {
            ZonedDateTime startDate = null;
            ZonedDateTime endDate = null;
            
            for (Activity activity : activities)
            {
                ZonedDateTime activityDate = activity.getDate();
                if (startDate == null || activityDate.isBefore(startDate))
                {
                    startDate = activityDate;
                }
                
                if (endDate == null || activityDate.isAfter(endDate))
                {
                    endDate = activityDate;
                }
            }
            DataContext dataContext = DataContext.newInstance();
            ElementType dateElementType = (ElementType) activities.get(0).getType(ActivityFactory.DATE);
            mergedActivity.put(ActivityFactory.DATE, dateElementType.valueToJSONForClient(startDate, dataContext));
            
            if (endDate != null
                && startDate != null
                && ChronoUnit.SECONDS.between(startDate, endDate) > 1)
            {
                // Ignore end date if events are less than 1 minute apart
                mergedActivity.put("endDate", dateElementType.valueToJSONForClient(endDate, dataContext));
            }
        }
        
        return mergedActivity;
    }
    
    public void handleEvent(Event event) throws RepositoryException
    {
        for (ActivityArguments activityArguments : getActivitiesArguments(event))
        {
            ActivityHolder holder = activityArguments.holder();
            Map<String, Object> arguments = new HashMap<>(event.getArguments());
            // Merge event arguments and activity's parameters
            arguments.putAll(activityArguments.parameters());
            // use the event issuer to determine the author. Some activity (like adding a member) can happened with no user authenticated
            holder.addActivity(this, arguments, event.getIssuer(), event.getId());
        }
    }

    /**
     * Get from the event the activity's parameters for each handled holder
     * @param event the event
     * @return the list of activities arguments
     */
    public abstract List<ActivityArguments> getActivitiesArguments(Event event);
    
    /**
     * Record for an activity holder and activity parameters specific to this holder
     * @param holder the activity holder
     * @param parameters the parameters
     */
    public record ActivityArguments(ActivityHolder holder, Map<String, Object> parameters) { /** */ }
}
