/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.activities;

import java.time.ZonedDateTime;
import java.util.HashMap;
import java.util.Map;

import javax.jcr.Node;

import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.repository.data.ametysobject.ModifiableModelLessDataAwareAmetysObject;
import org.ametys.plugins.repository.data.holder.ModifiableModelLessDataHolder;
import org.ametys.plugins.repository.data.holder.impl.DefaultModifiableModelLessDataHolder;
import org.ametys.plugins.repository.data.repositorydata.ModifiableRepositoryData;
import org.ametys.plugins.repository.data.repositorydata.impl.JCRRepositoryData;
import org.ametys.plugins.repository.jcr.SimpleAmetysObject;
import org.ametys.plugins.repository.model.RepositoryDataContext;
import org.ametys.runtime.model.type.DataContext;
import org.ametys.runtime.model.type.ModelItemTypeConstants;

/**
 * Object representing an activity (such as the creation of content)
 */
public class Activity extends SimpleAmetysObject<ActivityFactory> implements ModifiableModelLessDataAwareAmetysObject
{
    /** key used to store the activity id when serialized */
    public static final String ACTIVITY_ID_KEY = "id";
    /** key used to store the activity name when serialized */
    public static final String ACTIVITY_NAME_KEY = "name";

    /**
     * Creates an {@link Activity}.
     * @param node the node backing this {@link Activity}
     * @param parentPath the parentPath in the Ametys hierarchy
     * @param factory the factory which created the activity
     */
    public Activity(Node node, String parentPath, ActivityFactory factory)
    {
        super(node, parentPath, factory);
    }

    public ModifiableModelLessDataHolder getDataHolder()
    {
        ModifiableRepositoryData repositoryData = new JCRRepositoryData(getNode());
        return new DefaultModifiableModelLessDataHolder(_getFactory().getElementTypesExtensionPoint(), repositoryData);
    }
    
    /**
     * Get the type of the activity
     * @return the activity type
     */
    public ActivityType getActivityType()
    {
        return _getFactory().getActivityType(getValue(ActivityFactory.ACTIVITY_TYPE_ID));
    }
    
    /**
     * Get the event type that generated this activity
     * @return the event id
     */
    public String getEventType()
    {
        return getValueOfType(ActivityFactory.TYPE, ModelItemTypeConstants.STRING_TYPE_ID);
    }
    
    /**
     * Get the date of the activity
     * @return the date time
     */
    public ZonedDateTime getDate()
    {
        return getValueOfType(ActivityFactory.DATE, ModelItemTypeConstants.DATETIME_TYPE_ID);
    }
    
    /**
     * Get the author of the activity
     * @return the author identity
     */
    public UserIdentity getAuthor()
    {
        // FIXME REPOSITORY-504 user elementType is not implemented in repository.
        // but the available elementType for Activity are all defined in CMS so anyway it wouldn't work without CMS
        return getValueOfType(ActivityFactory.AUTHOR, "user");
    }
    
    /**
     * Generate SAX event to represent all the activity
     * @param contentHandler the targeted contentHandler
     * @throws SAXException if an error occurred
     */
    public void toSAX(ContentHandler contentHandler) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute(ACTIVITY_ID_KEY, getId());
        attrs.addCDATAAttribute(ACTIVITY_NAME_KEY, getName());
        XMLUtils.startElement(contentHandler, "activity", attrs);
        dataToSAX(contentHandler);
        XMLUtils.endElement(contentHandler, "activity");
    }

    
    /**
     * Serialize the activity as a JSON
     * @return the JSON
     */
    public Map<String, Object> toJSONForClient()
    {
        Map<String, Object> json = new HashMap<>();
        json.put(ACTIVITY_ID_KEY, getId());
        json.put(ACTIVITY_NAME_KEY, getName());
        DataContext dataContext = RepositoryDataContext.newInstance()
                                                       .withObject(this);
        
        json.putAll(dataToJSON(dataContext));
        
        json.putAll(getActivityType().additionnalDataToJSONForClient(this));
        
        return json;
    }
}
