/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.activities;

import javax.jcr.Node;
import javax.jcr.RepositoryException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.data.type.ModelItemTypeExtensionPoint;
import org.ametys.plugins.repository.jcr.SimpleAmetysObjectFactory;

/**
 * AmetysObject factory associated with {@link Activity}
 */
public class ActivityFactory extends SimpleAmetysObjectFactory
{
    /** Avalon Role for activity data types */
    public static final String MODEL_ITEM_TYPE_EXTENSION_ROLE = ModelItemTypeExtensionPoint.class.getName() + ".Activity";
    
    /** activity node type name. */
    public static final String NODE_TYPE = RepositoryConstants.NAMESPACE_PREFIX + ":activity";
    
    /** data name for the activity type id */
    public static final String ACTIVITY_TYPE_ID = "activityTypeId";
    
    /** data name for the date of the activity */
    public static final String DATE = "date";

    /** data name for the type of the event represented by the activity */
    public static final String TYPE = "type";

    /** data name for the author of the activity */
    public static final String AUTHOR = "author";
    
    private ModelItemTypeExtensionPoint _modelItemTypeEP;

    private ActivityTypeExtensionPoint _activityTypeEP;

    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _modelItemTypeEP = (ModelItemTypeExtensionPoint) serviceManager.lookup(MODEL_ITEM_TYPE_EXTENSION_ROLE);
        _activityTypeEP = (ActivityTypeExtensionPoint) serviceManager.lookup(ActivityTypeExtensionPoint.ROLE);
    }
    
    @Override
    public Activity getAmetysObject(Node node, String parentPath) throws AmetysRepositoryException, RepositoryException
    {
        return new Activity(node, parentPath, this);
    }

    /**
     * Retrieves the extension point with available modelItem types for
     * {@link Activity}
     * 
     * @return the extension point with available modelItem types for
     *         {@link Activity}
     */
    public ModelItemTypeExtensionPoint getElementTypesExtensionPoint()
    {
        return _modelItemTypeEP;
    }

    /**
     * Retrieve an {@link ActivityType}
     * @param activityTypeId the type id
     * @return the activity type
     */
    public ActivityType getActivityType(String activityTypeId)
    {
        return _activityTypeEP.getExtension(activityTypeId);
    }
}
