/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.activities;

import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Map;

import javax.jcr.RepositoryException;

import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.RepositoryConstants;

/**
 * Interface for activity holder
 */
public interface ActivityHolder
{
    /** the activity root node name */
    public static final String ACTIVITIES_ROOT_NODE_NAME = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":activities";
    /** Prefix used for naming the activity node */
    public static final String ACTIVITY_NAME_PREFIX = "ametys-activity_";
    /** formatter to store date time in jcr node name */
    public static final DateTimeFormatter JCR_UTC_FORMAT = DateTimeFormatter.ofPattern("yyyy_MM_dd_HH_mm_ss_SSS");
    
    /**
     * Returns the activities stored by this activity holder
     * @return The activities
     * @throws RepositoryException if failed to get activity nodes
     */
    public AmetysObjectIterable<Activity> getActivities() throws RepositoryException;
    
    /**
     * Add an activity to the activity holder
     * @param type the type
     * @param parameters the parameters
     * @param eventId the id of the event
     * @return the activity
     * @throws RepositoryException when an error occurred
     */
    public Activity addActivity(ActivityType type, Map<String, Object> parameters, String eventId) throws RepositoryException;
    
    /**
     * Add an activity to the activity holder
     * 
     * This convenient method will call {@link #addActivity(ZonedDateTime, ActivityType, Map, UserIdentity, String)} with the current time.
     * 
     * @param type the type
     * @param parameters the parameters
     * @param author the author
     * @param eventId the id of the event
     * @return the activity
     * @throws RepositoryException when an error occurred
     */
    public default Activity addActivity(ActivityType type, Map<String, Object> parameters, UserIdentity author, String eventId) throws RepositoryException
    {
        return addActivity(ZonedDateTime.now(), type, parameters, author, eventId);
    }
    
    /**
     * Add an activity to the activity holder
     * @param date the date
     * @param type the type
     * @param parameters the parameters
     * @param author the author
     * @param eventId the id of the event
     * @return the activity
     * @throws RepositoryException when an error occurred
     */
    public Activity addActivity(ZonedDateTime date, ActivityType type, Map<String, Object> parameters, UserIdentity author, String eventId) throws RepositoryException;
}
