/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.activities;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.jcr.RepositoryException;

import org.ametys.core.observation.Event;
import org.ametys.plugins.repository.data.holder.DataHolder;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.Supporter;

/**
 * Interface for activity types
 */
public interface ActivityType extends Supporter<Event>
{
    /**
     * Get the id of the activity type
     * @return the id of the activity type
     */
    public String getId();
    
    /**
     * Get the ids of the supported event types
     * @return the ids of the supported event types
     */
    public Map<String, I18nizableText> getSupportedEventTypes();
    
    /**
     * Add the specific values to store in activity for this activity type
     * @param activity the activity
     * @param parameters the activity parameters
     * @throws RepositoryException if an error occurs while manipulating the repository
     */
    public default void setAdditionalActivityData (Activity activity, Map<String, Object> parameters) throws RepositoryException
    {
        // Nothing to set
    }

    /**
     * Determines if two activities can be merged
     * @param activity1 The first activity
     * @param activity2 The second activity
     * @return true if the events can be merged
     */
    public boolean isMergeable (Activity activity1, Activity activity2);
    
    /**
     * Merge activities into one activity.
     * Be careful : be sure that the activities can be merged calling isMergeable method before
     * @param activities The activities to merge
     * @return Merged activities
     */
    public Map<String, Object> mergeActivities(List<Activity> activities);

    /**
     * Retrieve data specific to the {@link ActivityType} that needs to be serialized but are not stored in the {@link DataHolder}.
     * 
     * Those data will be added to the data of the dataHolder
     * 
     * @param activity the activity to serialize
     * @return a json map of the specific data
     */
    public default Map<String, Object> additionnalDataToJSONForClient(Activity activity)
    {
        return new HashMap<>();
    }
    
    /**
     * Handle the event
     * @param event the event
     * @throws RepositoryException if a repository exception
     */
    public void handleEvent(Event event) throws RepositoryException;
    
}
