/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.activities;

import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.MetadataExpression;

/**
 * Constructs an {@link Expression} corresponding to a activity'type comparison.
 */
public class ActivityTypeExpression implements Expression
{
    private static final String __METADATA_EXPRESSION = new MetadataExpression(ActivityFactory.TYPE).build();
    private Operator _operator;
    private String[] _values;

    /**
     * Creates the expression.
     * @param operator the operator to make the comparison (only Operator.EQ and Operator.NE allowed)
     * @param value the activity type id value 
     */
    public ActivityTypeExpression(Operator operator, String value)
    {
        if (Operator.EQ != operator && Operator.NE != operator)
        {
            throw new AmetysRepositoryException("Test operator '" + "' is unknown for test's expression.");
        }

        _operator = operator;
        _values = new String[]{value};
    }
    
    /**
     * Creates the expression.
     * @param operator the operator to make the comparison (only Operator.EQ and Operator.NE allowed)
     * @param values the activity' type values 
     */
    public ActivityTypeExpression(Operator operator, String... values)
    {
        _operator = operator;
        _values = values;
    }
    
    @Override
    public String build()
    {
        StringBuilder xpath = new StringBuilder();
        
        if (_values.length == 1)
        {
            xpath.append(__METADATA_EXPRESSION)
                 .append(' ').append(_operator).append(" '").append(_values[0].replaceAll("'", "''")).append("'");
            
            return xpath.toString();
        }
        
        xpath.append(Operator.EQ.equals(_operator) ? '(' : "not(");
        
        for (int i = 0; i < _values.length; i++)
        {
            if (i > 0)
            {
                xpath.append(LogicalOperator.OR.toString());
            }
            
            xpath.append(__METADATA_EXPRESSION)
                 .append(" = '").append(_values[i].replaceAll("'", "''")).append("'");
        }
        
        xpath.append(')');
        
        return xpath.toString();
    }
}
