/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.activities;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.ametys.core.observation.Event;
import org.ametys.runtime.plugin.component.AbstractThreadSafeComponentSupporterExtensionPoint;
import org.ametys.runtime.plugin.component.LazyInitializeExtensions;

/**
 * This class is in charge of loading the various {@link ActivityType}.
 * It also allows to communicate with them for the retrieval of activity in JSON
 */
@LazyInitializeExtensions
public class ActivityTypeExtensionPoint extends AbstractThreadSafeComponentSupporterExtensionPoint<ActivityType, Event>
{
    /** The Avalon Role */
    public static final String ROLE = ActivityTypeExtensionPoint.class.getName();
    
    /**
     * Merge activities
     * @param initialActivities The activities to merge
     * @return The merged activities
     */
    public List<Map<String, Object>> mergeActivities(List<Activity> initialActivities)
    {
        List<Map<String, Object>> mergedActivities = new ArrayList<>();
        
        List<Activity> activitiesToMerge = new ArrayList<>(initialActivities);
        Map<String, List<Activity>> mergeableActivities = new LinkedHashMap<>();
        
        for (Activity activity : initialActivities)
        {
            String activityId = activity.getId();
            
            List<Activity> clonedActivities = new ArrayList<>(activitiesToMerge);
            for (Activity clonedActivity : clonedActivities)
            {
                String clonedActivityId = clonedActivity.getId();
                
                if (!activityId.equals(clonedActivityId))
                {
                    ActivityType activityType = activity.getActivityType();
                    ActivityType clonedActivityType = activity.getActivityType();
                    
                    if (activityType == clonedActivityType && activityType.isMergeable(activity, clonedActivity))
                    {
                        mergeableActivities.computeIfAbsent(activityId, str -> new ArrayList<>()).add(clonedActivity);
                        
                        // The events were merged : no need to re-process them
                        activitiesToMerge.remove(activity);
                        activitiesToMerge.remove(clonedActivity);
                    }
                }
                else
                {
                    mergeableActivities.computeIfAbsent(activityId, str -> new ArrayList<>()).add(activity);
                }
            }
        }
        
        for (List<Activity> acts : mergeableActivities.values())
        {
            // FIXME GG not sure it's fine to take the first one. Quite random
            ActivityType eventType = acts.get(0).getActivityType();
            try
            {
                mergedActivities.add(eventType.mergeActivities(acts));
            }
            catch (Exception e)
            {
                getLogger().error("An error occurred while trying to merge activities. The original activities are kept", e);
                for (Activity act : acts)
                {
                    try
                    {
                        mergedActivities.add(act.toJSONForClient());
                    }
                    catch (Exception e2)
                    {
                        getLogger().error("An error occured while trying to serialize activity '{}'. The activity will be ignored.", act.getId(), e2);
                    }
                }
            }
        }
        
        return mergedActivities;
    }
    
    /**
     * Returns all the {@link ActivityType} matching the provided event id 
     * @param eventId the id of the event
     * @return The event type
     */
    public List<ActivityType> getActivityTypes(String eventId)
    {
        List<ActivityType> activities = new ArrayList<>(); 
        for (String extensionId : getExtensionsIds())
        {
            ActivityType eventType = getExtension(extensionId);
            if (eventType.getSupportedEventTypes().keySet().contains(eventId))
            {
                activities.add(eventType);
            }
        }
        
        return activities;
    }
}
