/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.activities;

import java.time.ZonedDateTime;
import java.util.HashMap;
import java.util.Map;

import javax.jcr.RepositoryException;

import org.ametys.core.observation.Event;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.ObservationConstants;
import org.ametys.runtime.model.type.ModelItemTypeConstants;

/**
 * Default implementation for an {@link ActivityHolder}
 * This implementation is backed by a {@link ModifiableTraversableAmetysObject}
 */
public class DefaultActivityHolder implements ActivityHolder
{
    private ModifiableTraversableAmetysObject _rootCollection;
    private DefaultActivityHolderFactory _activityHolderFactory;

    /**
     * Create a default {@link ActivityHolder}
     * @param rootCollection the ametys object backing this activity holder
     * @param factory the {@link DefaultActivityHolderFactory} which creates the AmetysObject
     */
    public DefaultActivityHolder(ModifiableTraversableAmetysObject rootCollection, DefaultActivityHolderFactory factory)
    {
        _rootCollection = rootCollection;
        _activityHolderFactory = factory;
    }

    /**
     * Returns the activity nodes
     * @return The activities
     * @throws RepositoryException if failed to get activity nodes
     */
    public AmetysObjectIterable<Activity> getActivities() throws RepositoryException
    {
        return _rootCollection.getChildren();
    }
    
    public Activity addActivity(ActivityType type, Map<String, Object> parameters, String eventId) throws RepositoryException
    {
        return addActivity(ZonedDateTime.now(), type, parameters, _activityHolderFactory.getCurrentUser(), eventId);
    }
    
    /**
     * Add an activity to the activity holder
     * @param date the date
     * @param type the type
     * @param parameters the parameters
     * @param author the author
     * @param eventId the id of the event
     * @return the activity
     * @throws RepositoryException when an error occurred
     */
    public Activity addActivity(ZonedDateTime date, ActivityType type, Map<String, Object> parameters, UserIdentity author, String eventId) throws RepositoryException
    {
        String activityName = ACTIVITY_NAME_PREFIX + date.format(JCR_UTC_FORMAT);
        
        Activity activity = _rootCollection.createChild(activityName, ActivityFactory.NODE_TYPE);
        
        // Type
        activity.setValue(ActivityFactory.ACTIVITY_TYPE_ID, type.getId(), ModelItemTypeConstants.STRING_TYPE_ID);
        
        // Date
        activity.setValue(ActivityFactory.DATE, date, ModelItemTypeConstants.DATETIME_TYPE_ID);
        
        // Type
        activity.setValue(ActivityFactory.TYPE, eventId, ModelItemTypeConstants.STRING_TYPE_ID);
        
        // Author
        // TODO GG we should use a constant here
        activity.setValue(ActivityFactory.AUTHOR, author, "user");
        
        type.setAdditionalActivityData(activity, parameters);
        _rootCollection.saveChanges();
        
        _notifyObserver(activity);
        
        return activity;
    }
    
    private void _notifyObserver(Activity activity)
    {
        Map<String, Object> params = new HashMap<>();
        // Provide the activity and the id for both sync and async observer
        params.put(ObservationConstants.ARGS_ACTIVITY, activity);
        params.put(ObservationConstants.ARGS_ACTIVITY_ID, activity.getId());
        Event event = new Event(ObservationConstants.EVENT_ACTIVITY_CREATED, activity.getAuthor(), params);
        _activityHolderFactory.getObservationManager().notify(event);
    }
}
