/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.repository.collection;

import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import javax.jcr.Node;
import javax.jcr.PathNotFoundException;
import javax.jcr.RepositoryException;

import org.ametys.core.group.GroupIdentity;
import org.ametys.core.right.ProfileAssignmentStorage.AnonymousOrAnyConnectedKeys;
import org.ametys.core.right.ProfileAssignmentStorage.UserOrGroup;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableACLAmetysObject;
import org.ametys.plugins.repository.RepositoryIntegrityViolationException;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.jcr.ACLJCRAmetysObjectHelper;
import org.ametys.plugins.repository.jcr.JCRTraversableAmetysObject;
import org.ametys.plugins.repository.jcr.NodeHelper;
import org.ametys.plugins.repository.jcr.SimpleAmetysObject;

/**
 * An {@link AmetysObject} representing a collection of other {@link AmetysObject}s.<br>
 * The collection stored its contents in the JCR Repository in a hash tree for
 * optimizing performances.<br>
 * Please note that this implementation does not keep the insertion order of elements.
 * @param <F> the type of the actual factory.
 * @param <A> the type of the composite {@link AmetysObject}.
 */
public class AmetysObjectCollection<F extends AmetysObjectCollectionFactory, A extends AmetysObject> extends SimpleAmetysObject<F> implements JCRTraversableAmetysObject, Iterable<A>, ModifiableACLAmetysObject
{
    AmetysObjectCollectionFactory _factory;
    
    /**
     * Creates an {@link AmetysObjectCollection}.
     * @param node the node backing this AmetysObject.
     * @param parentPath the parentPath in the Ametys hierarchy.
     * @param factory the {@link AmetysObjectCollectionFactory} which created the AmetysObject.
     */
    public AmetysObjectCollection(Node node, String parentPath, F factory)
    {
        super(node, parentPath, factory);
        _factory = factory;
    }

    @SuppressWarnings("unchecked")
    public AmetysObjectIterable<A> getChildren() throws AmetysRepositoryException
    {
        return _factory.getChildren(getPath(), getNode());
    }
    
    public Iterator<A> iterator()
    {
        return _factory.getChildren(getPath(), getNode()).iterator();
    }
    
    public boolean hasChild(String name) throws AmetysRepositoryException
    {
        if (name == null || "".equals(name) || name.charAt(0) == '/')
        {
            throw new AmetysRepositoryException("Child name cannot be null, empty or absolute");
        }
        
        if (".".equals(name) || "..".equals(name))
        {
            throw new AmetysRepositoryException("Child name cannot be . or ..");
        }
        
        try
        {
            return getNode().hasNode(NodeHelper.getFullHashPath(name));
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("An error occured getting child object " + name + " for object " + getName(), e);
        }
    }
    
    @SuppressWarnings("unchecked")
    public A getChild(String path) throws AmetysRepositoryException, UnknownAmetysObjectException
    {
        if (path == null || "".equals(path) || path.charAt(0) == '/')
        {
            throw new AmetysRepositoryException("Child path cannot be null, empty or absolute");
        }
        
        String subPath = null;
        String name;
        
        int i = path.indexOf('/');
        if (i == -1)
        {
            name = path;
        }
        else
        {
            name = path.substring(0, i);
            subPath = path.substring(i + 1);
        }
        
        if (".".equals(name) || "..".equals(name))
        {
            throw new AmetysRepositoryException("Path cannot contain segment with . or ..");
        }
        
        try
        {
            Node childNode = getNode().getNode(NodeHelper.getFullHashPath(name));
            
            return (A) _factory.getObject(null, childNode, subPath);
        }
        catch (PathNotFoundException e)
        {
            throw new UnknownAmetysObjectException("Unable to retrieve child node for path: " + path, e);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException(e);
        }
    }
    
    @SuppressWarnings("unchecked")
    public A createChild(String name, String type) throws AmetysRepositoryException, RepositoryIntegrityViolationException
    {
        Node contextNode = NodeHelper.getOrCreateFinalHashNode(getNode(), name);
        return (A) _factory.createChild(getPath(), contextNode, name, type);
    }
    
    @Override
    @SuppressWarnings("unchecked")
    public AmetysObject getParent() throws AmetysRepositoryException
    {
        return _factory.getParent(this);
    }
    
    public Map<AnonymousOrAnyConnectedKeys, Set<String>> getProfilesForAnonymousAndAnyConnectedUser()
    {
        return ACLJCRAmetysObjectHelper.getProfilesForAnonymousAndAnyConnectedUser(getNode());
    }
    
    public Map<GroupIdentity, Map<UserOrGroup, Set<String>>> getProfilesForGroups(Set<GroupIdentity> groups)
    {
        return ACLJCRAmetysObjectHelper.getProfilesForGroups(getNode(), groups);
    }
    
    public Map<UserIdentity, Map<UserOrGroup, Set<String>>> getProfilesForUsers(UserIdentity user)
    {
        return ACLJCRAmetysObjectHelper.getProfilesForUsers(getNode(), user);
    }
    
    public void addAllowedProfilesForAnyConnectedUser(Set<String> profileIds)
    {
        ACLJCRAmetysObjectHelper.addAllowedProfilesForAnyConnectedUser(getNode(), profileIds);
    }
    
    public void removeAllowedProfilesForAnyConnectedUser(Set<String> profileIds)
    {
        ACLJCRAmetysObjectHelper.removeAllowedProfilesForAnyConnectedUser(getNode(), profileIds);
    }
    
    public void addDeniedProfilesForAnyConnectedUser(Set<String> profileIds)
    {
        ACLJCRAmetysObjectHelper.addDeniedProfilesForAnyConnectedUser(getNode(), profileIds);
    }
    
    public void removeDeniedProfilesForAnyConnectedUser(Set<String> profileIds)
    {
        ACLJCRAmetysObjectHelper.removeDeniedProfilesForAnyConnectedUser(getNode(), profileIds);
    }
    
    public void addAllowedProfilesForAnonymous(Set<String> profileIds)
    {
        ACLJCRAmetysObjectHelper.addAllowedProfilesForAnonymous(getNode(), profileIds);
    }
    
    public void removeAllowedProfilesForAnonymous(Set<String> profileIds)
    {
        ACLJCRAmetysObjectHelper.removeAllowedProfilesForAnonymous(getNode(), profileIds);
    }
    
    public void addDeniedProfilesForAnonymous(Set<String> profileIds)
    {
        ACLJCRAmetysObjectHelper.addDeniedProfilesForAnonymous(getNode(), profileIds);
    }
    
    public void removeDeniedProfilesForAnonymous(Set<String> profileIds)
    {
        ACLJCRAmetysObjectHelper.removeDeniedProfilesForAnonymous(getNode(), profileIds);
    }
    
    public void addAllowedUsers(Set<UserIdentity> users, String profileId)
    {
        ACLJCRAmetysObjectHelper.addAllowedUsers(users, getNode(), profileId);
    }
    
    public void removeAllowedUsers(Set<UserIdentity> users, String profileId)
    {
        ACLJCRAmetysObjectHelper.removeAllowedUsers(users, getNode(), profileId);
    }
    
    public void removeAllowedUsers(Set<UserIdentity> users)
    {
        ACLJCRAmetysObjectHelper.removeAllowedUsers(users, getNode());
    }
    
    public void addAllowedGroups(Set<GroupIdentity> groups, String profileId)
    {
        ACLJCRAmetysObjectHelper.addAllowedGroups(groups, getNode(), profileId);
    }
    
    public void removeAllowedGroups(Set<GroupIdentity> groups, String profileId)
    {
        ACLJCRAmetysObjectHelper.removeAllowedGroups(groups, getNode(), profileId);
    }
    
    public void removeAllowedGroups(Set<GroupIdentity> groups)
    {
        ACLJCRAmetysObjectHelper.removeAllowedGroups(groups, getNode());
    }
    
    public void addDeniedUsers(Set<UserIdentity> users, String profileId)
    {
        ACLJCRAmetysObjectHelper.addDeniedUsers(users, getNode(), profileId);
    }
    
    public void removeDeniedUsers(Set<UserIdentity> users, String profileId)
    {
        ACLJCRAmetysObjectHelper.removeDeniedUsers(users, getNode(), profileId);
    }
    
    public void removeDeniedUsers(Set<UserIdentity> users)
    {
        ACLJCRAmetysObjectHelper.removeDeniedUsers(users, getNode());
    }
    
    public void addDeniedGroups(Set<GroupIdentity> groups, String profileId)
    {
        ACLJCRAmetysObjectHelper.addDeniedGroups(groups, getNode(), profileId);
    }
    
    public void removeDeniedGroups(Set<GroupIdentity> groups, String profileId)
    {
        ACLJCRAmetysObjectHelper.removeDeniedGroups(groups, getNode(), profileId);
    }
    
    public void removeDeniedGroups(Set<GroupIdentity> groups)
    {
        ACLJCRAmetysObjectHelper.removeDeniedGroups(groups, getNode());
    }
    
    public boolean isInheritanceDisallowed()
    {
        return ACLJCRAmetysObjectHelper.isInheritanceDisallowed(getNode());
    }
    
    public void disallowInheritance(boolean disallow)
    {
        ACLJCRAmetysObjectHelper.disallowInheritance(getNode(), disallow);
    }
}
