/*
 *  Copyright 2010 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.repository.collection;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.RepositoryException;
import javax.jcr.Session;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;

import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ChainedAmetysObjectIterable;
import org.ametys.plugins.repository.NodeIteratorIterable;
import org.ametys.plugins.repository.jcr.SimpleAmetysObjectFactory;

/**
 * Factory for {@link AmetysObjectCollection}.
 */
public class AmetysObjectCollectionFactory extends SimpleAmetysObjectFactory
{
    /** JCR nodetype for the collection itself */
    public static final String COLLECTION_NODETYPE = "ametys:collection";
    
    /** JCR nodetype for collection elements (ie. nodes "between" the collection and contained contents) */
    public static final String COLLECTION_ELEMENT_NODETYPE = "ametys:collectionElement";
    
    @Override
    public void configure(Configuration configuration) throws ConfigurationException
    {
        // does nothing, scheme and nodetypes are not obtained through configuration
    }

    @Override
    public String getScheme()
    {
        return "collection";
    }
    
    @Override
    public Collection<String> getNodetypes()
    {
        ArrayList<String> nodetypes = new ArrayList<>();
        
        nodetypes.add(COLLECTION_NODETYPE);
        nodetypes.add(COLLECTION_ELEMENT_NODETYPE);
        
        return Collections.unmodifiableCollection(nodetypes);
    }
    
    @Override
    @SuppressWarnings("unchecked")
    public AmetysObjectCollection getAmetysObject(Node node, String parentPath) throws AmetysRepositoryException, RepositoryException
    {
        String nodeType = node.getPrimaryNodeType().getName();
        
        if (nodeType.equals(COLLECTION_NODETYPE))
        {
            // c'est directement une collection
            return new AmetysObjectCollection(node, parentPath, this);
        }
        
        // sinon, c'est un élément de collection, on remonte jusqu'à la collection
        Node contextNode = node;
        while (!COLLECTION_NODETYPE.equals(contextNode.getPrimaryNodeType().getName()))
        {
            contextNode = contextNode.getParent();
        }
        
        return new AmetysObjectCollection(contextNode, parentPath, this);
    }
    
    /**
     * Returns the parent of the given {@link AmetysObjectCollection}
     * @param object an {@link AmetysObjectCollection}
     * @return the parent of the given {@link AmetysObjectCollection}
     * @throws AmetysRepositoryException if an error occurs
     */
    public AmetysObject getParent(AmetysObjectCollection object) throws AmetysRepositoryException
    {
        Node node = object.getNode();
        try
        {
            Node parentNode = node.getParent();
            return _resolver.resolve(parentNode, false);
        }
        catch (RepositoryException ex)
        {
            throw new AmetysRepositoryException("An error occured during resolving parent object of object " + object.getName(), ex);
        }
    }

    /**
     * Returns a single {@link AmetysObject} given its path and JCR Node.<br>
     * This method should never been called by clients.
     * @param <A> the type of the composite {@link AmetysObject}.
     * @param parentPath the path of the collection
     * @param node the node of the child
     * @param subPath the subpath in the Ametys hierarchy
     * @return an {@link AmetysObject}
     * @throws AmetysRepositoryException if an error occurs.
     */
    @SuppressWarnings("unchecked")
    public <A extends AmetysObject> A getObject(String parentPath, Node node, String subPath) throws AmetysRepositoryException
    {
        try
        {
            return (A) _resolver.resolve(parentPath, node, subPath, false);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("An error occured while resolving Node", e);
        }
    }
    
    /**
     * Creates a child object in the collection and resolve it to an {@link AmetysObject}.
     * @param <A> the type of the composite {@link AmetysObject}.
     * @param parentPath the parentPath of the new object.
     * @param parentNode the parent JCR Node of the new object, corresponding to a collection element.
     * @param name the name of the new object.
     * @param type the type of the Node backing the new object.
     * @return the newly created {@link AmetysObject}.
     * @throws AmetysRepositoryException if an error occurs.
     */
    @SuppressWarnings("unchecked")
    public <A extends AmetysObject> A createChild(String parentPath, Node parentNode, String name, String type) throws AmetysRepositoryException
    {
        try
        {
            return (A) _resolver.createAndResolve(parentPath, parentNode, name, type);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("An error occured while creating Node", e);
        }
    }
    
    /**
     * Returns the {@link AmetysObject}s children of the JCR Node backing an {@link AmetysObjectCollection}.<br>
     * This method should never been called by clients.
     * @param parentPath the parent path in the Ametys hierarchy of all {@link AmetysObject} being returned.
     * @param collectionNode the JCR Node backing the {@link AmetysObjectCollection}.
     * @return the {@link AmetysObject}s children.
     */
    @SuppressWarnings("unchecked")
    public AmetysObjectIterable getChildren(String parentPath, Node collectionNode)
    {
        List<AmetysObjectIterable> iterators = new ArrayList<>();
            
        try
        {
            NodeIterator it = collectionNode.getNodes();
            _addFirstLevelChildren(parentPath, iterators, it, collectionNode.getSession());
        }
        catch (RepositoryException ex)
        {
            throw new AmetysRepositoryException("An error occured while iterating inside the collection", ex);
        }
        
        return new ChainedAmetysObjectIterable(iterators);
    }
    
    private void _addFirstLevelChildren(String parentPath, List<AmetysObjectIterable> iterators, NodeIterator it, Session session) throws RepositoryException
    {
        // the collection itself could have some other child nodes which should be ignored here
        while (it.hasNext())
        {
            Node nextChild = it.nextNode();
            if (nextChild.getPrimaryNodeType().getName().equals(AmetysObjectCollectionFactory.COLLECTION_ELEMENT_NODETYPE))
            {
                _addNextLevelChildren(parentPath, iterators, nextChild.getNodes(), session);
            }
        }
    }
    
    private void _addNextLevelChildren(String parentPath, List<AmetysObjectIterable> iterators, NodeIterator it, Session session) throws RepositoryException
    {
        if (it.hasNext())
        {
            Node firstNode = it.nextNode();
            if (firstNode.getPrimaryNodeType().getName().equals(AmetysObjectCollectionFactory.COLLECTION_ELEMENT_NODETYPE))
            {
                NodeIterator firstChildIt = firstNode.getNodes();
                _addNextLevelChildren(parentPath, iterators, firstChildIt, session);
                
                while (it.hasNext())
                {
                    NodeIterator childIt = it.nextNode().getNodes();
                    _addNextLevelChildren(parentPath, iterators, childIt, session);
                }
            }
            else
            {
                iterators.add(new NodeIteratorIterable(_resolver, new WrapperNodeIterator(firstNode, it), parentPath, session));
            }
        }
    }
    
    private class WrapperNodeIterator implements NodeIterator
    {
        private Node _firstElement;
        private NodeIterator _it;
        
        private boolean _firstElementUsed;
        
        public WrapperNodeIterator(Node firstElement, NodeIterator it)
        {
            _firstElement = firstElement;
            _it = it;
            _firstElementUsed = false;
        }
        
        public Node nextNode()
        {
            if (!_firstElementUsed)
            {
                _firstElementUsed = true;
                return _firstElement;
            }
            
            return _it.nextNode();
        }

        public long getPosition()
        {
            if (!_firstElementUsed)
            {
                return 0;
            }
            
            return _it.getPosition();
        }

        public long getSize()
        {
            return _it.getSize();
        }

        public void skip(long skipNum)
        {
            if (skipNum < 0)
            {
                throw new IllegalArgumentException("skipNum must not be negative");
            }

            if (skipNum == 0)
            {
                return;
            }
            
            if (!_firstElementUsed)
            {
                _firstElementUsed = true;
                if (skipNum > 1)
                {
                    _it.skip(skipNum - 1);
                }
            }
            else
            {
                _it.skip(skipNum);
            }
        }

        public boolean hasNext()
        {
            if (!_firstElementUsed)
            {
                return true;
            }
            
            return _it.hasNext();
        }

        public Object next()
        {
            return nextNode();
        }

        public void remove()
        {
            throw new UnsupportedOperationException("remove is unsupported");
        }
    }
}
