/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.data.ametysobject;

import java.util.Collection;
import java.util.Map;
import java.util.Optional;

import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.data.external.ExternalizableDataProvider.ExternalizableDataStatus;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.plugins.repository.data.holder.group.ModelAwareComposite;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.impl.DataHolderHelper;
import org.ametys.plugins.repository.data.holder.values.SynchronizationContext;
import org.ametys.runtime.model.Model;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ViewItemAccessor;
import org.ametys.runtime.model.exception.BadDataPathCardinalityException;
import org.ametys.runtime.model.exception.BadItemTypeException;
import org.ametys.runtime.model.exception.UndefinedItemPathException;
import org.ametys.runtime.model.type.DataContext;

/**
 * Model aware {@link AmetysObject} that can handle data.
 */
public interface ModelAwareDataAwareAmetysObject extends DataAwareAmetysObject, ModelAwareDataHolder
{
    @Override
    public ModelAwareDataHolder getDataHolder();
    
    public default ModelAwareComposite getComposite(String compositePath) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException
    {
        return getDataHolder().getComposite(compositePath);
    }
    
    public default ModelAwareComposite getLocalComposite(String compositePath) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException
    {
        return getDataHolder().getLocalComposite(compositePath);
    }
    
    public default ModelAwareComposite getExternalComposite(String compositePath) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException
    {
        return getDataHolder().getExternalComposite(compositePath);
    }

    public default ModelAwareRepeater getRepeater(String repeaterPath) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException
    {
        return getDataHolder().getRepeater(repeaterPath);
    }

    public default ModelAwareRepeater getLocalRepeater(String repeaterPath) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException
    {
        return getDataHolder().getLocalRepeater(repeaterPath);
    }

    public default ModelAwareRepeater getExternalRepeater(String repeaterPath) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException
    {
        return getDataHolder().getExternalRepeater(repeaterPath);
    }
    
    public default boolean hasValue(String dataPath) throws IllegalArgumentException, BadDataPathCardinalityException
    {
        return DataAwareAmetysObject.super.hasValue(dataPath);
    }
    
    public default boolean hasLocalValue(String dataPath) throws IllegalArgumentException, BadDataPathCardinalityException
    {
        return getDataHolder().hasLocalValue(dataPath);
    }
    
    public default boolean hasExternalValue(String dataPath) throws IllegalArgumentException, BadDataPathCardinalityException
    {
        return getDataHolder().hasExternalValue(dataPath);
    }
    
    public default boolean hasValueOrEmpty(String dataPath) throws IllegalArgumentException, BadDataPathCardinalityException
    {
        return DataAwareAmetysObject.super.hasValueOrEmpty(dataPath);
    }
    
    public default boolean hasLocalValueOrEmpty(String dataPath) throws IllegalArgumentException, BadDataPathCardinalityException
    {
        return getDataHolder().hasLocalValueOrEmpty(dataPath);
    }
    
    public default boolean hasExternalValueOrEmpty(String dataPath) throws IllegalArgumentException, BadDataPathCardinalityException
    {
        return getDataHolder().hasExternalValueOrEmpty(dataPath);
    }

    public default <T extends Object> T getValue(String dataPath, boolean allowMultiValuedPathSegments) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException
    {
        return getDataHolder().getValue(dataPath, allowMultiValuedPathSegments);
    }
    
    public default <T> T getValue(String dataPath, boolean useDefaultFromModel, T defaultValue) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException
    {
        return getDataHolder().getValue(dataPath, useDefaultFromModel, defaultValue);
    }
    
    public default <T> T getLocalValue(String dataPath) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException
    {
        return getDataHolder().getLocalValue(dataPath);
    }
    
    public default <T> T getExternalValue(String dataPath) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException
    {
        return getDataHolder().getExternalValue(dataPath);
    }
    
    public default ExternalizableDataStatus getStatus(String dataPath) throws IllegalArgumentException, UndefinedItemPathException, BadDataPathCardinalityException
    {
        return getDataHolder().getStatus(dataPath);
    }
    
    @SuppressWarnings("unchecked")
    public default Collection<? extends Model> getModel()
    {
        return (Collection<? extends Model>) getDataHolder().getModel();
    }
    
    public default ModelItem getDefinition(String path) throws IllegalArgumentException, UndefinedItemPathException
    {
        return getDataHolder().getDefinition(path);
    }
    
    default boolean hasDefinition(String path) throws IllegalArgumentException
    {
        return getDataHolder().hasDefinition(path);
    }
    
    public default Collection<String> getDataNames()
    {
        return getDataHolder().getDataNames();
    }
    
    public default void dataToSAX(ContentHandler contentHandler, ViewItemAccessor viewItemAccessor, DataContext context) throws SAXException, BadItemTypeException
    {
        getDataHolder().dataToSAX(contentHandler, viewItemAccessor, context);
    }
    
    public default void dataToSAXForEdition(ContentHandler contentHandler, ViewItemAccessor viewItemAccessor, DataContext context) throws SAXException, BadItemTypeException
    {
        getDataHolder().dataToSAXForEdition(contentHandler, viewItemAccessor, context);
    }
    
    public default Map<String, Object> dataToJSON(ViewItemAccessor viewItemAccessor, DataContext context) throws BadItemTypeException
    {
        return getDataHolder().dataToJSON(viewItemAccessor, context);
    }
    
    public default Map<String, Object> dataToJSONForEdition(ViewItemAccessor viewItemAccessor, DataContext context) throws BadItemTypeException
    {
        return getDataHolder().dataToJSONForEdition(viewItemAccessor, context);
    }
    
    public default Map<String, Object> dataToMap(ViewItemAccessor viewItemAccessor, DataContext context)
    {
        return getDataHolder().dataToMap(viewItemAccessor, context);
    }
    
    /**
     * Check if there are differences between the given values and the current ones
     * @param values the values to check
     * @return <code>true</code> if there are differences, <code>false</code> otherwise
     * @throws UndefinedItemPathException if a key in the given Map refers to a data that is not defined by the model
     * @throws BadItemTypeException if the type defined by the model of one of the Map's key doesn't match the corresponding value
     */
    public default boolean hasDifferences(Map<String, Object> values) throws UndefinedItemPathException, BadItemTypeException
    {
        return hasDifferences(values, SynchronizationContext.newInstance());
    }
    
    /**
     * Check if there are differences between the given values and the current ones
     * @param values the values to check
     * @param context the context of the synchronization
     * @return <code>true</code> if there are differences, <code>false</code> otherwise
     * @throws UndefinedItemPathException if a key in the given Map refers to a data that is not defined by the model
     * @throws BadItemTypeException if the type defined by the model of one of the Map's key doesn't match the corresponding value
     */
    public default boolean hasDifferences(Map<String, Object> values, SynchronizationContext context) throws UndefinedItemPathException, BadItemTypeException
    {
        ViewItemAccessor viewItemAccessor = DataHolderHelper.createViewItemAccessorFromValues(getModel(), values);
        return hasDifferences(viewItemAccessor, values, context);
    }
    
    public default boolean hasDifferences(ViewItemAccessor viewItemAccessor, Map<String, Object> values) throws UndefinedItemPathException, BadItemTypeException
    {
        return hasDifferences(viewItemAccessor, values, SynchronizationContext.newInstance());
    }
    
    public default boolean hasDifferences(ViewItemAccessor viewItemAccessor, Map<String, Object> values, SynchronizationContext context) throws UndefinedItemPathException, BadItemTypeException
    {
        Map<String, Object> convertedValues = DataHolderHelper.convertValues(viewItemAccessor, values, context.ignoreIncompatibleValues()
                                                                                                            ? DataHolderHelper::convertValueIgnoringIncompatibleOnes
                                                                                                            : DataHolderHelper::convertValue);
        return getDataHolder().hasDifferences(viewItemAccessor, convertedValues, context);
    }

    /**
     * Get the collection of model items where there are differences between the given values and the current ones
     * @param values the values to check
     * @return a collection of model items with differences
     * @throws UndefinedItemPathException if a key in the given Map refers to a data that is not defined by the model
     * @throws BadItemTypeException if the type defined by the model of one of the Map's key doesn't match the corresponding value
     */
    public default Collection<ModelItem> getDifferences(Map<String, Object> values) throws UndefinedItemPathException, BadItemTypeException
    {
        return getDifferences(values, SynchronizationContext.newInstance());
    }
    
    /**
     * Get the collection of model items where there are differences between the given values and the current ones
     * @param values the values to check
     * @param context the context of the synchronization
     * @return a collection of model items with differences
     * @throws UndefinedItemPathException if a key in the given Map refers to a data that is not defined by the model
     * @throws BadItemTypeException if the type defined by the model of one of the Map's key doesn't match the corresponding value
     */
    public default Collection<ModelItem> getDifferences(Map<String, Object> values, SynchronizationContext context) throws UndefinedItemPathException, BadItemTypeException
    {
        ViewItemAccessor viewItemAccessor = DataHolderHelper.createViewItemAccessorFromValues(getModel(), values);
        return getDifferences(viewItemAccessor, values, context);
    }
    
    public default Collection<ModelItem> getDifferences(ViewItemAccessor viewItemAccessor, Map<String, Object> values) throws UndefinedItemPathException, BadItemTypeException
    {
        return getDifferences(viewItemAccessor, values, SynchronizationContext.newInstance());
    }
    
    public default Collection<ModelItem> getDifferences(ViewItemAccessor viewItemAccessor, Map<String, Object> values, SynchronizationContext context) throws UndefinedItemPathException, BadItemTypeException
    {
        Map<String, Object> convertedValues = DataHolderHelper.convertValues(viewItemAccessor, values, context.ignoreIncompatibleValues()
                                                                                                            ? DataHolderHelper::convertValueIgnoringIncompatibleOnes
                                                                                                            : DataHolderHelper::convertValue);
        return getDataHolder().getDifferences(viewItemAccessor, convertedValues, context);
    }
    
    public default Optional<? extends ModelAwareDataHolder> getParentDataHolder()
    {
        return getDataHolder().getParentDataHolder();
    }
    
    public default ModelAwareDataHolder getRootDataHolder()
    {
        return getDataHolder().getRootDataHolder();
    }
}
