/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.data.external;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.plugin.component.AbstractThreadSafeComponentExtensionPoint;

/**
 * Extension point for {@link ExternalizableDataProvider}s.
 */
public class ExternalizableDataProviderExtensionPoint extends AbstractThreadSafeComponentExtensionPoint<ExternalizableDataProvider>
{
    /** Avalon Role */
    public static final String ROLE = ExternalizableDataProviderExtensionPoint.class.getName();
    
    /**
     * Get the path of data that can be valued externally or locally by a {@link ExternalizableDataProvider}
     * @param dataHolder The externalizable data holder
     * @return The paths of data that can be valued externally or locally
     */
    public Set<String> getExternalizableDataPaths(ModelAwareDataHolder dataHolder)
    {
        Set<String> dataPaths = new HashSet<>();
        
        for (String id : getExtensionsIds())
        {
            ExternalizableDataProvider provider = getExtension(id);
            
            for (String dataPath : provider.getExternalizableDataPaths(dataHolder))
            {
                if (dataPaths.contains(dataPath))
                {
                    throw new UnsupportedOperationException("The data of path " + dataPath + " is already handled by another externalizable data provider");
                }
                dataPaths.add(dataPath);
            }
        }
        return dataPaths;
    }

    /**
     * Checks if the data of given model item is externalizable
     * @param dataHolder The externalizable data holder
     * @param modelItem the model item
     * @return <code>true</code> if the data of given model item is externalizable, <code>false</code> otherwise
     */
    public boolean isDataExternalizable(ModelAwareDataHolder dataHolder, ModelItem modelItem)
    {
        return isDataExternalizable(dataHolder, modelItem, Map.of());
    }
    
    /**
     * Checks if the data of given model item is externalizable in the given context
     * @param dataHolder The externalizable data holder
     * @param modelItem the model item
     * @param context the context that can be used to determine if the data is externalizable
     * @return <code>true</code> if the data of given model item is externalizable, <code>false</code> otherwise
     */
    public boolean isDataExternalizable(ModelAwareDataHolder dataHolder, ModelItem modelItem, Map<String, Object> context)
    {
        for (String id : getExtensionsIds())
        {
            ExternalizableDataProvider provider = getExtension(id);
            if (provider.isDataExternalizable(dataHolder, modelItem, context))
            {
                return true;
            }
        }
        
        // no provider set this data as externalizable
        return false;
    }
}
