/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.data.extractor.xml;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.xpath.XPathAPI;
import org.w3c.dom.Element;

import org.ametys.core.util.dom.DOMUtils;
import org.ametys.plugins.repository.data.extractor.ModelAwareValuesExtractor;
import org.ametys.plugins.repository.data.holder.impl.DataHolderHelper;
import org.ametys.plugins.repository.model.CompositeDefinition;
import org.ametys.plugins.repository.model.RepeaterDefinition;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.Model;
import org.ametys.runtime.model.ModelHelper;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ModelItemContainer;
import org.ametys.runtime.model.ModelItemGroup;
import org.ametys.runtime.model.ModelViewItem;
import org.ametys.runtime.model.ModelViewItemGroup;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.ViewElement;
import org.ametys.runtime.model.ViewItem;
import org.ametys.runtime.model.ViewItemContainer;
import org.ametys.runtime.model.exception.BadDataPathCardinalityException;
import org.ametys.runtime.model.exception.BadItemTypeException;
import org.ametys.runtime.model.exception.UndefinedItemPathException;
import org.ametys.runtime.model.type.ElementType;

/**
 * This class provides methods to extract values from an XML document, using a model
 */
public class ModelAwareXMLValuesExtractor implements ModelAwareValuesExtractor
{
    /** The DOM element containing the XML values */
    protected Element _element;
    
    /** The getter that retrieves needed additional data by types */
    protected Optional<XMLValuesExtractorAdditionalDataGetter> _additionalDataGetter;
    
    /** The model of the extracted values */
    protected Collection<? extends ModelItemContainer> _modelItemContainers;
    
    /**
     * Creates a model aware XML values extractor
     * @param element the DOM element containing the XML values
     * @param models the model of the extracted values
     */
    public ModelAwareXMLValuesExtractor(Element element, Model... models)
    {
        this(element, Arrays.asList(models));
    }

    /**
     * Creates a model aware XML values extractor
     * @param element the DOM element containing the XML values
     * @param modelItemContainers the model of the extracted values
     */
    public ModelAwareXMLValuesExtractor(Element element, Collection<? extends ModelItemContainer> modelItemContainers)
    {
        this(element, null, modelItemContainers);
    }

    /**
     * Creates a model aware XML values extractor
     * @param element the DOM element containing the XML values
     * @param additionalDataGetter the getter that retrieves needed additional data by types
     * @param models the model of the extracted values
     */
    public ModelAwareXMLValuesExtractor(Element element, XMLValuesExtractorAdditionalDataGetter additionalDataGetter, Model... models)
    {
        this(element, additionalDataGetter, Arrays.asList(models));
    }

    /**
     * Creates a model aware XML values extractor
     * @param element the DOM element containing the XML values
     * @param additionalDataGetter the getter that retrieves needed additional data by types
     * @param modelItemContainers the model of the extracted values
     */
    public ModelAwareXMLValuesExtractor(Element element, XMLValuesExtractorAdditionalDataGetter additionalDataGetter, Collection<? extends ModelItemContainer> modelItemContainers)
    {
        _element = element;
        _additionalDataGetter = Optional.ofNullable(additionalDataGetter);
        _modelItemContainers = modelItemContainers;
    }
    
    public Map<String, Object> extractValues() throws Exception
    {
        View view = new View();
        _fillViewItemContainerFromXML(_element, view, _modelItemContainers);
        return extractValues(view);
    }
    
    /**
     * Fill the given view item container with the data found in the given element.
     * @param element The DOM element containing the values
     * @param viewItemContainer The view item container to fill
     * @param modelItemContainer The model item containing the items that could be in the element
     * @throws Exception if an error occurs
     */
    protected void _fillViewItemContainerFromXML(Element element, ViewItemContainer viewItemContainer, ModelItemContainer modelItemContainer) throws Exception
    {
        _fillViewItemContainerFromXML(element, viewItemContainer, List.of(modelItemContainer));
    }
    
    /**
     * Fill the given view item container with the data found in the given element.
     * @param element The DOM element containing the values
     * @param viewItemContainer The view item container to fill
     * @param modelItemContainers The model items containing the items that could be in the element
     * @throws Exception if an error occurs
     */
    @SuppressWarnings("unchecked")
    protected void _fillViewItemContainerFromXML(Element element, ViewItemContainer viewItemContainer, Collection<? extends ModelItemContainer> modelItemContainers) throws Exception
    {
        List<Element> children = DOMUtils.getUniqueChildElements(element);
        for (Element child : children)
        {
            Optional<ModelItem> optionalModelItem = _getModelItemFromNodeName(element, child.getNodeName(), modelItemContainers);
            if (optionalModelItem.isPresent())
            {
                ModelItem modelItem = optionalModelItem.get();
                ModelViewItem modelViewItem;
                if (modelItem instanceof ModelItemGroup)
                {
                    Element groupNode = child;
                    if (modelItem instanceof RepeaterDefinition)
                    {
                        // Use the first entry to create the view - assume that all entries will have the same data
                        groupNode = DOMUtils.getChildElementByTagName(groupNode, "entry");
                    }
                    
                    modelViewItem = new ModelViewItemGroup();
                    
                    if (groupNode != null)
                    {
                        _fillViewItemContainerFromXML(groupNode, (ModelViewItemGroup) modelViewItem, (ModelItemGroup) modelItem);
                    }
                }
                else
                {
                    modelViewItem = new ViewElement();
                }

                modelViewItem.setDefinition(modelItem);
                if (!viewItemContainer.hasModelViewItem(modelViewItem))
                {
                    viewItemContainer.addViewItem(modelViewItem);
                }
            }
        }
    }
    
    /**
     * Retrieves the model item corresponding to the given node name
     * @param parent the DOM parent element
     * @param nodeName the node name
     * @param modelItemContainers the model item containers where to search for the model item
     * @return the model item corresponding to the given node name
     */
    protected Optional<ModelItem> _getModelItemFromNodeName(Element parent, String nodeName, Collection<? extends ModelItemContainer> modelItemContainers)
    {
        if (ModelHelper.hasModelItem(nodeName, modelItemContainers))
        {
            return Optional.of(ModelHelper.getModelItem(nodeName, modelItemContainers));
        }
        else
        {
            return Optional.empty();
        }
    }
    
    public Map<String, Object> extractValues(View view) throws Exception
    {
        return _extractValues(_element, view, "");
    }
    
    /**
     * Extracts the values of all items in the given view item container
     * @param currentElement the DOM element containing the values of the items in the container
     * @param viewItemContainer the view item container
     * @param prefix the path of the item represented by the view item container (prefix of all contained items)
     * @return the values of all items in the given view item containers
     * @throws Exception if an error occurs
     */
    protected Map<String, Object> _extractValues(Element currentElement, ViewItemContainer viewItemContainer, String prefix) throws Exception
    {
        Map<String, Object> values = new HashMap<>();
        for (ViewItem viewItem : viewItemContainer.getViewItems())
        {
            if (viewItem instanceof ModelViewItem)
            {
                ModelItem modelItem = ((ModelViewItem) viewItem).getDefinition();
                String dataName = modelItem.getName();
                if (_hasChildForAttribute(currentElement, dataName))
                {
                    Object value;
                    if (viewItem instanceof ModelViewItemGroup)
                    {
                        value = _extractGroupValues(currentElement, (ModelViewItemGroup) viewItem, dataName, prefix);
                    }
                    else
                    {
                        value = _extractElementValue(currentElement, (ElementDefinition) modelItem, prefix);
                    }
                        
                    values.put(dataName, value);
                }
            }
            else if (viewItem instanceof ViewItemContainer)
            {
                values.putAll(_extractValues(currentElement, (ViewItemContainer) viewItem, prefix));
            }
        }
        
        return values;
    }
    
    /**
     * Checks if the given element contains a child for the given attribute name
     * @param element the element to check
     * @param attributeName the name of the attribute to search
     * @return <code>true</code> if the element contains a child corresponding to the given attribute name, <code>false</code> otherwise
     */
    protected boolean _hasChildForAttribute(Element element, String attributeName)
    {
        return DOMUtils.hasChildElement(element, attributeName);
    }

    public <T> T extractValue(String dataPath) throws Exception
    {
        // Check that there is an item at the given path
        if (!ModelHelper.hasModelItem(dataPath, _modelItemContainers))
        {
            throw new UndefinedItemPathException("Unable to retrieve the value at path '" + dataPath + "'. There is no such item defined by the model.");
        }
        
        return _extractValue(_modelItemContainers, _element, dataPath, StringUtils.EMPTY);
    }
    
    /**
     * Extracts the value at the given path
     * @param <T> type of the value to retrieve
     * @param modelItemContainer The model item containing the item of the value to extract
     * @param currentElement the DOM element containing the model item container's values
     * @param relativeDataPath The data path relative to the model item container
     * @param prefix the path of the item represented by the model item container (prefix of all contained items)
     * @return the value
     * @throws Exception if an error occurs
     */
    protected <T> T _extractValue(ModelItemContainer modelItemContainer, Element currentElement, String relativeDataPath, String prefix) throws Exception
    {
        return _extractValue(List.of(modelItemContainer), currentElement, relativeDataPath, prefix);
    }
    
    /**
     * Extracts the value at the given path
     * @param <T> type of the value to retrieve
     * @param modelItemContainers The model items containing the item of the value to extract
     * @param currentElement the DOM element containing the model item containers' values
     * @param relativeDataPath The data path relative to the model item containers
     * @param prefix the path of the item represented by the model item containers (prefix of all contained items)
     * @return the value
     * @throws Exception if an error occurs
     */
    protected <T> T _extractValue(Collection<? extends ModelItemContainer> modelItemContainers, Element currentElement, String relativeDataPath, String prefix) throws Exception
    {
        String[] pathSegments = StringUtils.split(relativeDataPath, ModelItem.ITEM_PATH_SEPARATOR);

        if (pathSegments == null || pathSegments.length < 1)
        {
            throw new IllegalArgumentException("Unable to extract the value of the data at the given path. This path is empty.");
        }
        else if (pathSegments.length == 1)
        {
            String dataName = relativeDataPath;
            ModelItem modelItem = ModelHelper.getModelItem(dataName, modelItemContainers);
            if (modelItem instanceof ElementDefinition)
            {
                return _extractElementValue(currentElement, (ElementDefinition) modelItem, prefix);
            }
            else
            {
                ModelViewItemGroup modelViewItemGroup = ModelViewItemGroup.of((ModelItemGroup) modelItem);
                return _extractGroupValues(currentElement, modelViewItemGroup, dataName, prefix);
            }
        }
        else
        {
            String firstSegmentDataName = pathSegments[0];
            String newRelativeDataPath = StringUtils.join(pathSegments, ModelItem.ITEM_PATH_SEPARATOR, 1, pathSegments.length);
            String newPrefix = StringUtils.isNotEmpty(prefix) ? prefix + ModelItem.ITEM_PATH_SEPARATOR + firstSegmentDataName : firstSegmentDataName;

            ModelItem modelItem = ModelHelper.getModelItem(firstSegmentDataName, modelItemContainers);
            
            if (modelItem instanceof RepeaterDefinition)
            {
                if (DataHolderHelper.isRepeaterEntryPath(firstSegmentDataName))
                {
                    Pair<String, Integer> repeaterNameAndEntryPosition = DataHolderHelper.getRepeaterNameAndEntryPosition(firstSegmentDataName);
                    Element repeaterEntryElement = (Element) XPathAPI.selectSingleNode(currentElement, repeaterNameAndEntryPosition.getLeft() + "/entry[@name='" + repeaterNameAndEntryPosition.getRight() + "']");
                    if (repeaterEntryElement != null)
                    {
                        return _extractValue((RepeaterDefinition) modelItem, repeaterEntryElement, newRelativeDataPath, newPrefix);
                    }
                    else
                    {
                        return null;
                    }
                }
                else
                {
                    throw new BadDataPathCardinalityException("Unable to extract the value at path '" + relativeDataPath + "'. The segment '" + pathSegments[0] + "' refers to a repeater but not an entry.");
                }
            }
            else if (modelItem instanceof CompositeDefinition)
            {
                Element compositeElement = DOMUtils.getChildElementByTagName(currentElement, firstSegmentDataName);
                return compositeElement != null ? _extractValue((CompositeDefinition) modelItem, compositeElement, newRelativeDataPath, newPrefix) : null;
            }
            else
            {
                throw new BadItemTypeException("Unable to extract the value at path '" + relativeDataPath + "'. The segment '" + pathSegments[0] + "' does not represent a group item.");
            }
        }
    }
    
    /**
     * Extracts the value of the given element
     * @param <T> type of the value to retrieve
     * @param parent the DOM element of the element definition's parent
     * @param definition the element's definition
     * @param prefix the path of the element's parent
     * @return the value
     * @throws Exception if an error occurs
     */
    @SuppressWarnings("unchecked")
    protected <T> T _extractElementValue(Element parent, ElementDefinition definition, String prefix) throws Exception
    {
        ElementType type = definition.getType();
        String dataName = definition.getName();
       
        String absoluteDataPath = StringUtils.isNotEmpty(prefix) ? prefix + ModelItem.ITEM_PATH_SEPARATOR + dataName : dataName;
        
        Optional<Object> additionalData = _additionalDataGetter.flatMap(dataGetter -> dataGetter.getAdditionalData(absoluteDataPath, type));
        
        Object value = _extractElementValue(parent, definition, additionalData);
        
        if (definition.isMultiple() && value != null && !value.getClass().isArray())
        {
            // The value is single but should be an array. Create the array with the single value
            return (T) new Object[] {value};
        }
        else if (!definition.isMultiple() && value != null && value.getClass().isArray())
        {
            // The value is multiple but should be single. Retrieve the first value of the array
            return Array.getLength(value) > 0 ? (T) Array.get(value, 0) : null;
        }
        else
        {
            return (T) value;
        }
    }
    
    /**
     * Extracts the value of the given element
     * @param <T> type of the element definition
     * @param parent the DOM element of the element definition's parent
     * @param definition the element's definition
     * @param additionalData the additional data needed to extract the value
     * @return the value
     * @throws Exception if an error occurs
     */
    protected <T> Object _extractElementValue(Element parent, ElementDefinition<T> definition, Optional<Object> additionalData) throws Exception
    {
        ElementType<T> type = definition.getType();
        String dataName = definition.getName();
        return type.valueFromXML(parent, dataName, additionalData);
    }
    
    /**
     * Extracts the values of the given group
     * @param <T> type of the values to retrieve (example: {@link Map} for a composite, {@link List} for a repeater
     * @param parent the DOM element of the group's parent
     * @param modelViewItemGroup view item corresponding to the group
     * @param dataName the name of the data to extract
     * @param prefix the path of the group's parent
     * @return the value
     * @throws Exception if an error occurs
     */
    @SuppressWarnings("unchecked")
    protected <T> T _extractGroupValues(Element parent, ModelViewItemGroup modelViewItemGroup, String dataName, String prefix) throws Exception
    {
        ModelItemGroup modelItemGroup = modelViewItemGroup.getDefinition();
        String newPrefix = StringUtils.isNotEmpty(prefix) ? prefix + ModelItem.ITEM_PATH_SEPARATOR + dataName : dataName;

        if (modelItemGroup instanceof RepeaterDefinition)
        {
            if (DataHolderHelper.isRepeaterEntryPath(dataName))
            {
                Pair<String, Integer> repeaterNameAndEntryPosition = DataHolderHelper.getRepeaterNameAndEntryPosition(dataName);
                Element repeaterEntryElement = (Element) XPathAPI.selectSingleNode(parent, repeaterNameAndEntryPosition.getLeft() + "/entry[@name='" + repeaterNameAndEntryPosition.getRight() + "']");
                return repeaterEntryElement != null ? (T) _extractValues(repeaterEntryElement, modelViewItemGroup, newPrefix) : null;
            }
            else
            {
                Element repeaterElement = DOMUtils.getChildElementByTagName(parent, dataName);
                if (repeaterElement != null)
                {
                    List<Map<String, Object>> repeaterValues = new ArrayList<>();
                    int repeaterSize = Integer.valueOf(XPathAPI.eval(repeaterElement, "count(entry)").str());
                    for (int i = 1; i <= repeaterSize; i++)
                    {
                        Element repeaterEntryElement = (Element) XPathAPI.selectSingleNode(repeaterElement, "entry[@name='" + i + "']");
                        Map<String, Object> repeaterEntryValues = _extractValues(repeaterEntryElement, modelViewItemGroup, newPrefix + "[" + i + "]");
                        repeaterValues.add(repeaterEntryValues);
                    }
                    return (T) repeaterValues;
                }
                else
                {
                    return null;
                }
            }
        }
        else
        {
            Element compositeElement = DOMUtils.getChildElementByTagName(parent, dataName);
            return compositeElement != null ? (T) _extractValues(compositeElement, modelViewItemGroup, newPrefix) : null;
        }
    }
}
