/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.data.extractor.xml;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.apache.commons.lang3.StringUtils;
import org.apache.xpath.XPathAPI;
import org.w3c.dom.Element;

import org.ametys.core.util.dom.DOMUtils;
import org.ametys.plugins.repository.data.extractor.ModelLessValuesExtractor;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.type.ElementType;
import org.ametys.runtime.model.type.ModelItemType;
import org.ametys.runtime.plugin.component.AbstractThreadSafeComponentExtensionPoint;

/**
 * This class provides methods to extract values from an XML document
 */
public class ModelLessXMLValuesExtractor implements ModelLessValuesExtractor
{
    /** The DOM element containing the XML values */
    protected Element _element;
    
    /** The getter that retrieves needed additional data by types */
    protected Optional<XMLValuesExtractorAdditionalDataGetter> _additionalDataGetter;
    
    /** Extension point providing available data types */
    protected AbstractThreadSafeComponentExtensionPoint<? extends ModelItemType> _dataTypesExtensionPoint;

    /**
     * Creates a model less XML values extractor
     * @param element the DOM element containing the XML values
     * @param dataTypesExtensionPoint the extension point providing available data types
     */
    public ModelLessXMLValuesExtractor(Element element, AbstractThreadSafeComponentExtensionPoint<? extends ModelItemType> dataTypesExtensionPoint)
    {
        this(element, null, dataTypesExtensionPoint);
    }

    /**
     * Creates a model less XML values extractor
     * @param element the DOM element containing the XML values
     * @param additionalDataGetter the getter that retrieves needed additional data by types
     * @param dataTypesExtensionPoint the extension point providing available data types
     */
    public ModelLessXMLValuesExtractor(Element element, XMLValuesExtractorAdditionalDataGetter additionalDataGetter, AbstractThreadSafeComponentExtensionPoint<? extends ModelItemType> dataTypesExtensionPoint)
    {
        _element = element;
        _additionalDataGetter = Optional.ofNullable(additionalDataGetter);
        _dataTypesExtensionPoint = dataTypesExtensionPoint;
    }
    
    public Map<String, Object> extractValues() throws Exception
    {
        return _extractValues(_element, "");
    }
    
    /**
     * Extracts all the values in the current element
     * @param currentElement the current element
     * @param prefix the path of the item represented by the current element (prefix of all contained items)
     * @return the values
     * @throws Exception if an error occurs
     */
    protected Map<String, Object> _extractValues(Element currentElement, String prefix) throws Exception
    {
        Map<String, Object> values = new HashMap<>();
        
        List<Element> children = DOMUtils.getUniqueChildElements(currentElement);
        for (Element child : children)
        {
            String dataName = child.getNodeName();
            Object value = _extractValue(currentElement, dataName, prefix);
            if (value != null)
            {
                values.put(dataName, value);
            }
        }
        
        return values;
    }

    public <T> T extractValue(String dataPath) throws Exception
    {
        return _extractValue(_element, dataPath, "");
    }
    
    /**
     * Extracts the value at the given path
     * @param <T> type of the value to retrieve
     * @param currentElement the DOM element containing the items' values
     * @param relativeDataPath The data path relative to the current element
     * @param prefix the path of the item represented by the current element (prefix of all contained items)
     * @return the value
     * @throws Exception if an error occurs
     */
    @SuppressWarnings("unchecked")
    protected <T> T _extractValue(Element currentElement, String relativeDataPath, String prefix) throws Exception
    {
        String[] pathSegments = StringUtils.split(relativeDataPath, ModelItem.ITEM_PATH_SEPARATOR);

        if (pathSegments == null || pathSegments.length < 1)
        {
            throw new IllegalArgumentException("Unable to extract the value of the data at the given path. This path is empty.");
        }
        else if (pathSegments.length == 1)
        {
            String dataName = relativeDataPath;
            String dataTypeId = XPathAPI.eval(currentElement, dataName + "/@typeId").str();
            ModelItemType type = _dataTypesExtensionPoint.getExtension(dataTypeId);
            
            if (type instanceof ElementType)
            {
                ElementType elementType = (ElementType) type;
                String absoluteDataPath = StringUtils.isNotEmpty(prefix) ? prefix + ModelItem.ITEM_PATH_SEPARATOR + dataName : dataName;
                Optional<Object> additionalData = _additionalDataGetter.flatMap(dataGetter -> dataGetter.getAdditionalData(absoluteDataPath, elementType));
                
                return (T) elementType.valueFromXML(currentElement, dataName, additionalData);
            }
            else
            {
                String newPrefix = StringUtils.isNotEmpty(prefix) ? prefix + ModelItem.ITEM_PATH_SEPARATOR + dataName : dataName;

                Element compositeNode = (Element) XPathAPI.selectSingleNode(currentElement, dataName);
                return compositeNode != null ? (T) _extractValues(compositeNode, newPrefix) : null;
            }
        }
        else
        {
            String firstSegmentDataName = pathSegments[0];
            String newRelativeDataPath = StringUtils.join(pathSegments, ModelItem.ITEM_PATH_SEPARATOR, 1, pathSegments.length);
            String newPrefix = StringUtils.isNotEmpty(prefix) ? prefix + ModelItem.ITEM_PATH_SEPARATOR + firstSegmentDataName : firstSegmentDataName;
            
            Element compositeNode = (Element) XPathAPI.selectSingleNode(currentElement, firstSegmentDataName);
            return compositeNode != null ? _extractValue(compositeNode, newRelativeDataPath, newPrefix) : null;
        }
    }
}
