/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.data.holder;

import java.util.Collection;
import java.util.Optional;

import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.plugins.repository.data.holder.group.Composite;
import org.ametys.plugins.repository.data.holder.impl.DataHolderHelper;
import org.ametys.plugins.repository.data.repositorydata.RepositoryData;
import org.ametys.runtime.model.exception.BadItemTypeException;
import org.ametys.runtime.model.exception.NotUniqueTypeException;
import org.ametys.runtime.model.exception.UndefinedItemPathException;
import org.ametys.runtime.model.exception.UnknownTypeException;
import org.ametys.runtime.model.type.DataContext;

/**
 * Interface for data containers
 */
public interface DataHolder
{
    /**
     * Retrieves the composite at the given path
     * @param compositePath path of the composite to retrieve
     * @return the composite or <code>null</code> if not exists or is empty
     * @throws IllegalArgumentException if the given composite path is null or empty
     * @throws BadItemTypeException if the stored value at the given path is not a composite
     */
    public Composite getComposite(String compositePath) throws IllegalArgumentException, BadItemTypeException;
    
    /**
     * Checks if there is a non empty value for the data at the given path
     * @param dataPath path of the data
     * @return <code>true</code> if there is a non empty value for the data, <code>false</code> otherwise
     * @throws IllegalArgumentException if the given data path is null or empty
     */
    public boolean hasValue(String dataPath) throws IllegalArgumentException;
    
    /**
     * Checks if there is a value, even empty, for the data at the given path
     * @param dataPath path of the data
     * @return <code>true</code> if there is value, even empty, for the data, <code>false</code> otherwise
     * @throws IllegalArgumentException if the given data path is null or empty
     */
    public boolean hasValueOrEmpty(String dataPath) throws IllegalArgumentException;
    
    /**
     * Retrieves the names of data contained by this data holder
     * Retrieves only the data at first level, does not check composite data 
     * @return the names of all data contained by this data holder
     */
    public Collection<String> getDataNames();
    
    /**
     * Copies the current {@link DataHolder} to the given {@link ModifiableModelAwareDataHolder}.
     * @param dataHolder The destination dataHolder. Can not be null.
     * @throws UndefinedItemPathException if one of the copied data is not defined by the model of the destination {@link ModifiableModelAwareDataHolder}
     * @throws BadItemTypeException if the type defined by the model of the destination {@link ModifiableModelAwareDataHolder} doesn't match the copied value
     * @throws UnknownTypeException if there is no available type compatible with the copied value for the type extension point of the destination {@link ModifiableModelLessDataHolder}
     * @throws NotUniqueTypeException if there is more than one available types compatibles with the copied value for the type extension point of the destination {@link ModifiableModelLessDataHolder}
     */
    public default void copyTo(ModifiableDataHolder dataHolder) throws UndefinedItemPathException, BadItemTypeException, UnknownTypeException, NotUniqueTypeException
    {
        copyTo(dataHolder, DataContext.newInstance());
    }
    
    /**
     * Copies the current {@link DataHolder} to the given {@link ModifiableModelAwareDataHolder}.
     * @param dataHolder The destination dataHolder. Can not be null.
     * @param context The context of the data to copy
     * @throws UndefinedItemPathException if one of the copied data is not defined by the model of the destination {@link ModifiableModelAwareDataHolder}
     * @throws BadItemTypeException if the type defined by the model of the destination {@link ModifiableModelAwareDataHolder} doesn't match the copied value
     * @throws UnknownTypeException if there is no available type compatible with the copied value for the type extension point of the destination {@link ModifiableModelLessDataHolder}
     * @throws NotUniqueTypeException if there is more than one available types compatibles with the copied value for the type extension point of the destination {@link ModifiableModelLessDataHolder}
     */
    public default void copyTo(ModifiableDataHolder dataHolder, DataContext context) throws UndefinedItemPathException, BadItemTypeException, UnknownTypeException, NotUniqueTypeException
    {
        DataHolderHelper.copyTo(this, dataHolder, context);
    }
    
    /**
     * Generates SAX events for the data at the given data path in the current {@link DataHolder}
     * Do not generate any event if there is no values at the given path
     * @param contentHandler the {@link ContentHandler} that will receive the SAX events
     * @param dataPath the path of the data to SAX
     * @throws SAXException if an error occurs during the SAX events generation
     */
    public default void dataToSAX(ContentHandler contentHandler, String dataPath) throws SAXException
    {
        dataToSAX(contentHandler, dataPath, DataContext.newInstance());
    }
    
    /**
     * Generates SAX events for the data at the given data path in the current {@link DataHolder}
     * Do not generate any event if there is no values at the given path
     * @param contentHandler the {@link ContentHandler} that will receive the SAX events
     * @param dataPath the path of the data to SAX
     * @param context The context of the data to SAX
     * @throws SAXException if an error occurs during the SAX events generation
     */
    public void dataToSAX(ContentHandler contentHandler, String dataPath, DataContext context) throws SAXException;
    
    /**
     * Convert the data at the given path into a JSON object
     * @param dataPath the path of the data to convert
     * @return The value as JSON
     */
    public default Object dataToJSON(String dataPath)
    {
        return dataToJSON(dataPath, DataContext.newInstance());
    }
    
    /**
     * Convert the data at the given path into a JSON object
     * @param dataPath the path of the data to convert
     * @param context The context of the data to convert
     * @return The value as JSON
     */
    public Object dataToJSON(String dataPath, DataContext context);
    
    /**
     * Retrieves the repository data used by this {@link DataHolder}
     * @return the repository data used by this {@link DataHolder}
     */
    public RepositoryData getRepositoryData();
    
    /**
     * Retrieves the optional parent of the current {@link DataHolder}
     * There can be no parent if the current {@link DataHolder} is the root
     * @return the parent of the current {@link DataHolder}
     */
    public Optional<? extends DataHolder> getParentDataHolder();
    
    /**
     * Retrieves the {@link DataHolder} that is the root of the current one
     * @return the root {@link DataHolder}
     */
    public DataHolder getRootDataHolder();
}
