/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.data.holder;

import java.util.Collection;
import java.util.Map;
import java.util.Optional;

import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.plugins.repository.data.external.ExternalizableDataProvider.ExternalizableDataStatus;
import org.ametys.plugins.repository.data.holder.group.ModelAwareComposite;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.impl.DataHolderHelper;
import org.ametys.plugins.repository.data.holder.values.SynchronizationContext;
import org.ametys.plugins.repository.model.RepeaterDefinition;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.ModelHelper;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ModelItemContainer;
import org.ametys.runtime.model.ViewHelper;
import org.ametys.runtime.model.ViewItemAccessor;
import org.ametys.runtime.model.exception.BadDataPathCardinalityException;
import org.ametys.runtime.model.exception.BadItemTypeException;
import org.ametys.runtime.model.exception.UndefinedItemPathException;
import org.ametys.runtime.model.type.DataContext;
import org.ametys.runtime.model.type.ModelItemType;

/**
 * Interface for data containers with models
 */
public interface ModelAwareDataHolder extends DataHolder
{
    /** Suffix used for the alternative value */
    public static final String ALTERNATIVE_SUFFIX = "__alt";
    /** Suffix used for the status value */
    public static final String STATUS_SUFFIX = "__status";
    
    /**
     * {@inheritDoc}
     * @throws UndefinedItemPathException if the given composite path is not defined by the model
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    public ModelAwareComposite getComposite(String compositePath) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException;
    
    /**
     * Retrieves the local composite at the given path
     * @param compositePath path of the externalizable composite to retrieve
     * @return the composite or <code>null</code> if not exists or is empty
     * @throws IllegalArgumentException if the given composite path is null or empty
     * @throws BadItemTypeException if the stored value at the given path is not a composite
     * @throws UndefinedItemPathException if the given composite path is not defined by the model
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    public ModelAwareComposite getLocalComposite(String compositePath) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException;
    
    /**
     * Retrieves the external composite at the given path
     * @param compositePath path of the externalizable composite to retrieve
     * @return the composite or <code>null</code> if not exists or is empty
     * @throws IllegalArgumentException if the given composite path is null or empty
     * @throws BadItemTypeException if the stored value at the given path is not a composite
     * @throws UndefinedItemPathException if the given composite path is not defined by the model
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    public ModelAwareComposite getExternalComposite(String compositePath) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException;
    
    /**
     * Retrieves the repeater at the given path
     * @param repeaterPath path of the repeater to retrieve
     * @return the repeater or <code>null</code> if not exists or is empty
     * @throws IllegalArgumentException if the given repeater path is null or empty
     * @throws BadItemTypeException if the stored value at the given path is not a repeater
     * @throws UndefinedItemPathException if the given repeater path is not defined by the model
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    public ModelAwareRepeater getRepeater(String repeaterPath) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException;
    
    /**
     * Retrieves the local repeater at the given path
     * @param repeaterPath path of the externalizable repeater to retrieve
     * @return the repeater or <code>null</code> if not exists or is empty
     * @throws IllegalArgumentException if the given repeater path is null or empty
     * @throws BadItemTypeException if the stored value at the given path is not a repeater
     * @throws UndefinedItemPathException if the given repeater path is not defined by the model
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    public ModelAwareRepeater getLocalRepeater(String repeaterPath) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException;
    
    /**
     * Retrieves the external repeater at the given path
     * @param repeaterPath path of the externalizable repeater to retrieve
     * @return the repeater or <code>null</code> if not exists or is empty
     * @throws IllegalArgumentException if the given repeater path is null or empty
     * @throws BadItemTypeException if the stored value at the given path is not a repeater
     * @throws UndefinedItemPathException if the given repeater path is not defined by the model
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    public ModelAwareRepeater getExternalRepeater(String repeaterPath) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException;
    
    /**
     * {@inheritDoc}
     * @return <code>true</code> if the data at the given path is defined by the model, if there is a non empty value for the data and if the type of this value matches the type of the definition. <code>false</code> otherwise
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    @Override
    public boolean hasValue(String dataPath) throws IllegalArgumentException, BadDataPathCardinalityException;
    
    /**
     * Checks if there is a non empty local value for the data at the given path
     * @param dataPath path of the externalizable data
     * @return <code>true</code> if the data at the given path is defined by the model, if there is a non empty local value for the data and if the type of this value matches the type of the definition. <code>false</code> otherwise
     * @throws IllegalArgumentException if the given data path is null or empty
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    public boolean hasLocalValue(String dataPath) throws IllegalArgumentException, BadDataPathCardinalityException;
    
    /**
     * Checks if there is a non empty external value for the data at the given path
     * @param dataPath path of the externalizable data
     * @return <code>true</code> if the data at the given path is defined by the model, if there is a non empty external value for the data and if the type of this value matches the type of the definition. <code>false</code> otherwise
     * @throws IllegalArgumentException if the given data path is null or empty
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    public boolean hasExternalValue(String dataPath) throws IllegalArgumentException, BadDataPathCardinalityException;
    
    /**
     * {@inheritDoc}
     * @return <code>true</code> if the data at the given path is defined by the model, if there is a value for the data, even empty, and if the type of this value matches the type of the definition. <code>false</code> otherwise
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    @Override
    public boolean hasValueOrEmpty(String dataPath) throws IllegalArgumentException, BadDataPathCardinalityException;
    
    /**
     * Checks if there is a local value for the data at the given path
     * @param dataPath path of the externalizable data
     * @return <code>true</code> if the data at the given path is defined by the model, if there is a local value for the data, even empty, and if the type of this value matches the type of the definition. <code>false</code> otherwise
     * @throws IllegalArgumentException if the given data path is null or empty
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    public boolean hasLocalValueOrEmpty(String dataPath) throws IllegalArgumentException, BadDataPathCardinalityException;
    
    /**
     * Checks if there is an external value for the data at the given path
     * @param dataPath path of the externalizable data
     * @return <code>true</code> if the data at the given path is defined by the model, if there is an external value for the data, even empty, and if the type of this value matches the type of the definition. <code>false</code> otherwise
     * @throws IllegalArgumentException if the given data path is null or empty
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    public boolean hasExternalValueOrEmpty(String dataPath) throws IllegalArgumentException, BadDataPathCardinalityException;
    
    /**
     * {@inheritDoc}
     * @return the names of the data contained by this data holder and that are defined by the model
     */
    @Override
    public Collection<String> getDataNames();
    
    /**
     * Retrieves the value of the data at the given path
     * @param <T> type of the value to retrieve
     * @param dataPath path of the data
     * @return the value of the data or <code>null</code> if not exists or is empty. The object returned may be of a generic class defined by the storage (if the model is unknown). For example, an url may be returned as a String.
     * @throws IllegalArgumentException if the given data path is null or empty
     * @throws UndefinedItemPathException if the given data path is not defined by the model
     * @throws BadItemTypeException if the type defined by the model doesn't match the type of the stored value
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    public default <T extends Object> T getValue(String dataPath) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException
    {
        return getValue(dataPath, false);
    }
    
    /**
     * Retrieves the value of the data at the given path
     * @param <T> type of the value to retrieve
     * @param dataPath path of the data
     * @param allowMultiValuedPathSegments <code>true</code> to allow multi-valued segments in the path (not necessarily at the last segment), <code>false</code> otherwise.
     *      If <code>true</code>, if there is no indicated entry for a repeater, the values of all the entries are retrieved
     *      If <code>true</code> and if there are multiple values, all data are retrieved in one array
     * @return the value of the data or <code>null</code> if allowMultiValuedPathSegments is <code>false</code> and there is no non empty value. The object returned may be of a generic class defined by the storage (if the model is unknown). For example, an url may be returned as a String.
     * @throws IllegalArgumentException if the given data path is null or empty
     * @throws UndefinedItemPathException if the given data path is not defined by the model
     * @throws BadItemTypeException if the type defined by the model doesn't match the type of the stored value
     * @throws BadDataPathCardinalityException if the managesMultiples boolean is <code>false</code> and the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    public <T extends Object> T getValue(String dataPath, boolean allowMultiValuedPathSegments) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException;
    
    /**
     * Retrieves the value of the data at the given path, or the default value
     * The returned value is one of those ones, in the order:
     * <ol>
     * <li>The value of the data if exists and is not empty</li>
     * <li>The default value from the model if useDefaultFromModel is <code>true</code> and there is a default value defined by the model</li>
     * <li>The given default value</li>
     * </ol>
     * @param <T> type of the value to retrieve
     * @param dataPath path of the data
     * @param useDefaultFromModel true to use the default value from the model, false to use the given default value
     * @param defaultValue default value used if value is null and useDefaultFromModel is false, or if there is no default value on model
     * @return the value of the data at the given path
     * @throws IllegalArgumentException if the given data path is null or empty
     * @throws UndefinedItemPathException if the given data path is not defined by the model
     * @throws BadItemTypeException if the type defined by the model doesn't match the type of the stored value
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    public <T extends Object> T getValue(String dataPath, boolean useDefaultFromModel, T defaultValue) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException;
    
    /**
     * Retrieves the local value of the data at the given path
     * @param <T> type of the value to retrieve
     * @param dataPath path of the externalizable data
     * @return the local value of the data or <code>null</code> if not exists or is empty. The object returned may be of a generic class defined by the storage (if the model is unknown). For example, an url may be returned as a String.
     * @throws IllegalArgumentException if the given data path is null or empty
     * @throws UndefinedItemPathException if the given data path is not defined by the model
     * @throws BadItemTypeException if the type defined by the model doesn't match the type of the stored value
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    public <T extends Object> T getLocalValue(String dataPath) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException;
    
    /**
     * Retrieves the external value of the data at the given path
     * @param <T> type of the value to retrieve
     * @param dataPath path of the externalizable data
     * @return the external value of the data or <code>null</code> if not exists or is empty. The object returned may be of a generic class defined by the storage (if the model is unknown). For example, an url may be returned as a String.
     * @throws IllegalArgumentException if the given data path is null or empty
     * @throws UndefinedItemPathException if the given data path is not defined by the model
     * @throws BadItemTypeException if the type defined by the model doesn't match the type of the stored value
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    public <T extends Object> T getExternalValue(String dataPath) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException;
    
    /**
     * Retrieves the status of the externalizable data at the given path
     * Warning: This method won't check that your data is externalizable. But there is no sense to call it with a non externalizable data
     * @param dataPath path of the externalizable data
     * @return the status of the externalizable data at the given path
     * @throws IllegalArgumentException if the given data path is null or empty
     * @throws UndefinedItemPathException if the given data path is not defined by the model
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    public ExternalizableDataStatus getStatus(String dataPath) throws IllegalArgumentException, UndefinedItemPathException, BadDataPathCardinalityException;
    
    /**
     * Checks if the definition of the element at the given path is multiple
     * @param path path of the element. No matter if it is a definition or data path (with repeater entry positions)
     * @return <code>true</code> if the element is multiple, <code>false</code> otherwise
     * @throws IllegalArgumentException if the given path is null or empty
     * @throws UndefinedItemPathException if the given path is not defined by the model
     */
    public default boolean isMultiple(String path) throws IllegalArgumentException, UndefinedItemPathException
    {
        ModelItem item = getDefinition(path);
        if (item instanceof ElementDefinition)
        {
            return ((ElementDefinition) item).isMultiple();
        }
        else if (item instanceof RepeaterDefinition)
        {
            // If the given path represents a repeater , but with no specified entry, consider the data as multiple
            return !DataHolderHelper.isRepeaterEntryPath(path);
        }
        else
        {
            // Composites are not multiples
            return false;
        }
    }
    
    /**
     * Retrieves the type of the data at the given path
     * @param <X> type of the item type
     * @param path path of the data. No matter if it is a definition or data path (with repeater entry positions)
     * @return the type of the data
     * @throws IllegalArgumentException if the given data path is null or empty
     * @throws UndefinedItemPathException if the given data path is not defined by the model
     */
    @SuppressWarnings("unchecked")
    public default <X extends ModelItemType> X getType(String path) throws IllegalArgumentException, UndefinedItemPathException
    {
        return (X) getDefinition(path).getType();
    }
    
    /**
     * Retrieves the data holder's model
     * @return the data holder's model
     */
    public Collection<? extends ModelItemContainer> getModel();
    
    /**
     * Retrieves the definition of the data at the given path
     * @param path path of the data. No matter if it is a definition or data path (with repeater entry positions)
     * @return the definition of the data
     * @throws IllegalArgumentException if the given path is null or empty
     * @throws UndefinedItemPathException if the given path is not defined by the model
     */
    public default ModelItem getDefinition(String path) throws IllegalArgumentException, UndefinedItemPathException
    {
        return ModelHelper.getModelItem(path, getModel());
    }
    
    /**
     * Checks if there is a definition at the given path
     * @param path path of the data. No matter if it is a definition or data path (with repeater entry positions)
     * @return <code>true</code> if there is definition at the given path, <code>false</code> otherwise
     * @throws IllegalArgumentException if the given path is null or empty
     */
    public default boolean hasDefinition(String path) throws IllegalArgumentException
    {
        try
        {
            getDefinition(path);
            return true;
        }
        catch (UndefinedItemPathException e)
        {
            return false;
        }
    }
    
    /**
     * Generates SAX events for the data in the model of the current {@link DataHolder}
     * @param contentHandler the {@link ContentHandler} that will receive the SAX events
     * @throws SAXException if an error occurs during the SAX events generation
     * @throws BadItemTypeException if the saxed value's type does not matches the stored data
     */
    public default void dataToSAX(ContentHandler contentHandler) throws SAXException, BadItemTypeException
    {
        dataToSAX(contentHandler, DataContext.newInstance());
    }
    
    /**
     * Generates SAX events for the data in the model of the current {@link DataHolder}
     * @param contentHandler the {@link ContentHandler} that will receive the SAX events
     * @param context The context of the data to SAX
     * @throws SAXException if an error occurs during the SAX events generation
     * @throws BadItemTypeException if the saxed value's type does not matches the stored data
     */
    public default void dataToSAX(ContentHandler contentHandler, DataContext context) throws SAXException, BadItemTypeException
    {
        ViewItemAccessor viewItemAccessor = ViewHelper.createViewItemAccessor(getModel());
        dataToSAX(contentHandler, viewItemAccessor, context);
    }
    
    /**
     * Generates SAX events for the data in the given view in the current {@link DataHolder}
     * @param contentHandler the {@link ContentHandler} that will receive the SAX events
     * @param viewItemAccessor the {@link ViewItemAccessor} referencing the items for which generate SAX events
     * @throws SAXException if an error occurs during the SAX events generation
     * @throws BadItemTypeException if the saxed value's type does not matches the stored data
     */
    public default void dataToSAX(ContentHandler contentHandler, ViewItemAccessor viewItemAccessor) throws SAXException, BadItemTypeException
    {
        dataToSAX(contentHandler, viewItemAccessor, DataContext.newInstance());
    }
    
    /**
     * Generates SAX events for the data in the given view in the current {@link DataHolder}
     * @param contentHandler the {@link ContentHandler} that will receive the SAX events
     * @param viewItemAccessor the {@link ViewItemAccessor} referencing the items for which generate SAX events
     * @param context The context of the data to SAX
     * @throws SAXException if an error occurs during the SAX events generation
     * @throws BadItemTypeException if the saxed value's type does not matches the stored data
     */
    public void dataToSAX(ContentHandler contentHandler, ViewItemAccessor viewItemAccessor, DataContext context) throws SAXException, BadItemTypeException;
    
    /**
     * Generates SAX events for the data in the given view in edition mode in the current {@link DataHolder}
     * @param contentHandler the {@link ContentHandler} that will receive the SAX events
     * @param viewItemAccessor the {@link ViewItemAccessor} referencing the items for which generate SAX events
     * @param context The context of the data to SAX
     * @throws SAXException if an error occurs during the SAX events generation
     * @throws BadItemTypeException if the saxed value's type does not matches the stored data
     */
    public void dataToSAXForEdition(ContentHandler contentHandler, ViewItemAccessor viewItemAccessor, DataContext context) throws SAXException, BadItemTypeException;
    
    /**
     * Convert the data in the model of the current {@link DataHolder}
     * @return The data of the current {@link DataHolder} as JSON
     * @throws BadItemTypeException if the value's type does not matches the stored data
     */
    public default Map<String, Object> dataToJSON() throws BadItemTypeException
    {
        return dataToJSON(DataContext.newInstance());
    }
    
    /**
     * Convert the data in the model of the current {@link DataHolder}
     * @param context The context of the data to convert
     * @return The data of the current {@link DataHolder} as JSON
     * @throws BadItemTypeException if the value's type does not matches the stored data
     */
    public default Map<String, Object> dataToJSON(DataContext context) throws BadItemTypeException
    {
        ViewItemAccessor viewItemAccessor = ViewHelper.createViewItemAccessor(getModel());
        return dataToJSON(viewItemAccessor, context);
    }
    
    /**
     * Convert the data in the given view of the current {@link DataHolder}
     * @param viewItemAccessor the {@link ViewItemAccessor} referencing the items to convert
     * @return The data of the given view as JSON
     * @throws BadItemTypeException if the value's type does not matches the stored data
     */
    public default Map<String, Object> dataToJSON(ViewItemAccessor viewItemAccessor) throws BadItemTypeException
    {
        return dataToJSON(viewItemAccessor, DataContext.newInstance());
    }
    
    /**
     * Convert the data in the given view of the current {@link DataHolder}
     * @param viewItemAccessor the {@link ViewItemAccessor} referencing the items to convert
     * @param context The context of the data to convert
     * @return The data of the given view as JSON
     * @throws BadItemTypeException if the value's type does not matches the stored data
     */
    public Map<String, Object> dataToJSON(ViewItemAccessor viewItemAccessor, DataContext context) throws BadItemTypeException;
    
    /**
     * Convert the data in the given view in edition mode in the current {@link DataHolder}
     * @param viewItemAccessor the {@link ViewItemAccessor} referencing the items to convert
     * @param context The context of the data to convert
     * @return The data of the given view as JSON
     * @throws BadItemTypeException if the value's type does not matches the stored data
     */
    public Map<String, Object> dataToJSONForEdition(ViewItemAccessor viewItemAccessor, DataContext context) throws BadItemTypeException;
    
    /**
     * Retrieves all data of this DataHolder as a typed-values Map.
     * @return a Map containing all data.
     */
    public default Map<String, Object> dataToMap()
    {
        return dataToMap(DataContext.newInstance());
    }
    
    /**
     * Retrieves all data of this DataHolder as a typed-values Map.
     * @param context The context of the data
     * @return  a Map containing all data.
     */
    public default Map<String, Object> dataToMap(DataContext context)
    {
        ViewItemAccessor viewItemAccessor = ViewHelper.createViewItemAccessor(getModel());
        return dataToMap(viewItemAccessor, context);
    }
    
    /**
     * Retrieves data of this DataHolder as a typed-values Map.
     * @param viewItemAccessor the {@link ViewItemAccessor} referencing the items to include in the resulting Map
     * @return a Map containing all data.
     */
    public default Map<String, Object> dataToMap(ViewItemAccessor viewItemAccessor)
    {
        return dataToMap(viewItemAccessor, DataContext.newInstance());
    }
    
    /**
     * Retrieves data of this DataHolder as a typed-values Map.
     * @param viewItemAccessor the {@link ViewItemAccessor} referencing the items to include in the resulting Map
     * @param context The context of the data
     * @return a Map containing all data.
     */
    public Map<String, Object> dataToMap(ViewItemAccessor viewItemAccessor, DataContext context);
    
    /**
     * Check if there are differences between the given values and the current ones
     * @param viewItemAccessor The {@link ViewItemAccessor} for all items to check
     * @param values the values to check
     * @return <code>true</code> if there are differences, <code>false</code> otherwise
     * @throws UndefinedItemPathException if a key in the given Map refers to a data that is not defined by the model
     * @throws BadItemTypeException if the type defined by the model of one of the Map's key doesn't match the corresponding value
     */
    public boolean hasDifferences(ViewItemAccessor viewItemAccessor, Map<String, Object> values) throws UndefinedItemPathException, BadItemTypeException;
    
    /**
     * Check if there are differences between the given values and the current ones
     * @param viewItemAccessor The {@link ViewItemAccessor} for all items to check
     * @param values the values to check
     * @param context the context of the synchronization
     * @return <code>true</code> if there are differences, <code>false</code> otherwise
     * @throws UndefinedItemPathException if a key in the given Map refers to a data that is not defined by the model
     * @throws BadItemTypeException if the type defined by the model of one of the Map's key doesn't match the corresponding value
     */
    public boolean hasDifferences(ViewItemAccessor viewItemAccessor, Map<String, Object> values, SynchronizationContext context) throws UndefinedItemPathException, BadItemTypeException;

    /**
     * Get the collection of model items where there are differences between the given values and the current ones
     * @param viewItemAccessor The {@link ViewItemAccessor} for all items to check
     * @param values the values to check
     * @return a collection of model items with differences
     * @throws UndefinedItemPathException if a key in the given Map refers to a data that is not defined by the model
     * @throws BadItemTypeException if the type defined by the model of one of the Map's key doesn't match the corresponding value
     */
    public Collection<ModelItem> getDifferences(ViewItemAccessor viewItemAccessor, Map<String, Object> values) throws UndefinedItemPathException, BadItemTypeException;

    /**
     * Get the collection of model items where there are differences between the given values and the current ones
     * @param viewItemAccessor The {@link ViewItemAccessor} for all items to check
     * @param values the values to check
     * @param context the context of the synchronization
     * @return a collection of model items with differences
     * @throws UndefinedItemPathException if a key in the given Map refers to a data that is not defined by the model
     * @throws BadItemTypeException if the type defined by the model of one of the Map's key doesn't match the corresponding value
     */
    public Collection<ModelItem> getDifferences(ViewItemAccessor viewItemAccessor, Map<String, Object> values, SynchronizationContext context) throws UndefinedItemPathException, BadItemTypeException;
    
    @Override
    public Optional<? extends ModelAwareDataHolder> getParentDataHolder();
    
    @Override
    public ModelAwareDataHolder getRootDataHolder();
}
