/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.data.holder;

import java.util.Map;
import java.util.Optional;

import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.plugins.repository.data.UnknownDataException;
import org.ametys.plugins.repository.data.holder.group.ModelLessComposite;
import org.ametys.plugins.repository.data.type.ModelItemTypeExtensionPoint;
import org.ametys.plugins.repository.data.type.RepositoryModelItemType;
import org.ametys.runtime.model.exception.BadDataPathCardinalityException;
import org.ametys.runtime.model.exception.BadItemTypeException;
import org.ametys.runtime.model.exception.NotUniqueTypeException;
import org.ametys.runtime.model.exception.UnknownTypeException;
import org.ametys.runtime.model.type.DataContext;

/**
 * Interface for data containers without models
 */
public interface ModelLessDataHolder extends DataHolder
{
    /**
     * {@inheritDoc}
     * @throws BadDataPathCardinalityException if the value of a part of the data path is multiple. Only the last part can be multiple
     */
    @Override
    public ModelLessComposite getComposite(String compositePath) throws IllegalArgumentException, BadItemTypeException, BadDataPathCardinalityException;
    
    /**
     * {@inheritDoc}
     * @throws BadDataPathCardinalityException if the value of a part of the data path is multiple. Only the last part can be multiple
     */
    @Override
    public boolean hasValue(String dataPath) throws IllegalArgumentException, BadDataPathCardinalityException;
    
    /**
     * Checks if there is a non empty value for the data at the given path
     * @param dataPath path of the data
     * @param dataTypeId type identifier of the data
     * @return <code>true</code> if there is a non empty value for the data, <code>false</code> otherwise
     * @throws IllegalArgumentException if the given data path is null or empty
     * @throws UnknownTypeException if the given type isn't available for this data holder's type extension point
     * @throws BadDataPathCardinalityException if the value of a part of the data path is multiple. Only the last part can be multiple
     */
    public boolean hasValue(String dataPath, String dataTypeId) throws IllegalArgumentException, UnknownTypeException, BadDataPathCardinalityException;
    
    /**
     * {@inheritDoc}
     * @throws BadDataPathCardinalityException if the value of a part of the data path is multiple. Only the last part can be multiple
     */
    @Override
    public boolean hasValueOrEmpty(String dataPath) throws IllegalArgumentException, BadDataPathCardinalityException;
    
    /**
     * Retrieves the value of the data at the given path
     * The type of the value will be deduced from the stored data.
     * In some cases, the type can be wrong. For example, it is impossible to know if a stored date is a date or a date time
     * @param <T> type of the value to retrieve. Should match the given data type
     * @param dataPath path of the data
     * @return the value of the data or <code>null</code> if not exists or is empty. The object returned may be of a generic class defined by the storage. For example, an url may be returned as a String. Use the 2 arguments version to ensure to get the right kind of Object.
     * @throws IllegalArgumentException if the given data path is null or empty
     * @throws UnknownTypeException if there is no compatible type with the data at the given data path
     * @throws NotUniqueTypeException if there are many compatible types (there is no way to determine which type is the good one)
     * @throws BadItemTypeException if value at the sub data path of the given one is not a composite
     * @throws BadDataPathCardinalityException if the value of a part of the data path is multiple. Only the last part can be multiple
     */
    public <T> T getValue(String dataPath) throws IllegalArgumentException, UnknownTypeException, NotUniqueTypeException, BadItemTypeException, BadDataPathCardinalityException;
    
    /**
     * Retrieves the value of the data at the given path if exists and is not empty, or the default value
     * @param <T> type of the value to retrieve. Should match the given data type
     * @param dataPath path of the data
     * @param defaultValue default value
     * @return the value of the data, <code>null</code> if the data exists but is empty, or the given default value
     * @throws IllegalArgumentException if the given data path is null or empty
     * @throws UnknownTypeException if there is no compatible type with the data at the given data path
     * @throws NotUniqueTypeException if there are many compatible types (there is no way to determine which type is the good one)
     * @throws BadItemTypeException if value at the sub data path of the given one is not a composite
     * @throws BadDataPathCardinalityException if the value of a part of the data path is multiple. Only the last part can be multiple
     */
    public <T> T getValue(String dataPath, T defaultValue) throws IllegalArgumentException, UnknownTypeException, NotUniqueTypeException, BadItemTypeException, BadDataPathCardinalityException;
    
    /**
     * Retrieves the value of the data at the given path
     * @param <T> type of the value to retrieve. Should match the given data type
     * @param dataPath path of the data
     * @param dataTypeId type identifier of the data
     * @return the value of the data or <code>null</code> if not exists or is empty
     * @throws IllegalArgumentException if the given data path is null or empty
     * @throws UnknownTypeException if the given type isn't available for this data holder's type extension point
     * @throws BadItemTypeException if the given type doesn't match the type of the stored value at the given path
     * @throws BadDataPathCardinalityException if the value of a part of the data path is multiple. Only the last part can be multiple
     */
    public <T> T getValueOfType(String dataPath, String dataTypeId) throws IllegalArgumentException, UnknownTypeException, BadItemTypeException, BadDataPathCardinalityException;
    
    /**
     * Retrieves the value of the data at the given path if exists and is not empty, or the default value
     * @param <T> type of the value to retrieve. Should match the given data type
     * @param dataPath path of the data
     * @param dataTypeId type identifier of the data
     * @param defaultValue default value
     * @return the value of the data, <code>null</code> if the data exists but is empty, or the given default value
     * @throws IllegalArgumentException if the given data path is null or empty
     * @throws UnknownTypeException if the given type isn't available for this data holder's type extension point
     * @throws BadItemTypeException if the given type doesn't match the type of the stored value at the given path
     * @throws BadDataPathCardinalityException if the value of a part of the data path is multiple. Only the last part can be multiple
     */
    public <T> T getValueOfType(String dataPath, String dataTypeId, T defaultValue) throws IllegalArgumentException, UnknownTypeException, BadItemTypeException, BadDataPathCardinalityException;
    
    /**
     * Checks if the value of the data at the given path is multiple
     * @param dataPath path of the data to check
     * @return <code>true</code> if the value of the data is multiple, <code>false</code> otherwise
     * @throws IllegalArgumentException if the given data path is null or empty
     * @throws UnknownDataException the data at the given path does not exist
     * @throws UnknownTypeException if there is no compatible type with the data at the given data path
     * @throws NotUniqueTypeException if there are many compatible types (there is no way to determine which type is the good one)
     * @throws BadItemTypeException if value at the sub data path of the given one is not a composite
     * @throws BadDataPathCardinalityException if the value of a part of the data path is multiple. Only the last part can be multiple
     */
    public boolean isMultiple(String dataPath) throws IllegalArgumentException, UnknownDataException, UnknownTypeException, NotUniqueTypeException, BadItemTypeException, BadDataPathCardinalityException;
    
    /**
     * Checks if the value of the data at the given path is multiple
     * @param dataPath path of the data to check
     * @param dataTypeId type identifier of the data
     * @return <code>true</code> if the value of the data is multiple, <code>false</code> otherwise
     * @throws IllegalArgumentException if the given data path is null or empty
     * @throws UnknownDataException the data at the given path does not exist
     * @throws UnknownTypeException if there is no compatible type with the data at the given data path
     * @throws BadItemTypeException if value at the sub data path of the given one is not a composite
     * @throws BadDataPathCardinalityException if the value of a part of the data path is multiple. Only the last part can be multiple
     */
    public boolean isMultiple(String dataPath, String dataTypeId) throws IllegalArgumentException, UnknownDataException, UnknownTypeException, BadItemTypeException, BadDataPathCardinalityException;
    
    /**
     * Retrieves the type of the data at the given path
     * @param dataPath path of the data
     * @return the type of the data
     * @throws IllegalArgumentException if the given data path is null or empty
     * @throws UnknownDataException if there is no data stored at the given path
     * @throws UnknownTypeException if there is no compatible type with the data at the given data path or if the data is a repeater entry but the composite type is not available
     * @throws NotUniqueTypeException if there are many compatible types (there is no way to determine which type is the good one)
     * @throws BadDataPathCardinalityException if the value of a part of the data path is multiple. Only the last part can be multiple
     */
    public RepositoryModelItemType getType(String dataPath) throws IllegalArgumentException, UnknownDataException, UnknownTypeException, NotUniqueTypeException, BadDataPathCardinalityException;
    
    /**
     * Generates SAX events for data contained in this {@link DataHolder}
     * @param contentHandler the {@link ContentHandler} that will receive the SAX events
     * @throws SAXException if an error occurs during the SAX events generation
     * @throws UnknownTypeException if there is no compatible type with the saxed value
     * @throws NotUniqueTypeException if there are many compatible types (there is no way to determine which type is the good one) with the saxed value
     */
    public default void dataToSAX(ContentHandler contentHandler) throws SAXException, UnknownTypeException, NotUniqueTypeException
    {
        dataToSAX(contentHandler, DataContext.newInstance());
    }
    
    /**
     * Generates SAX events for data contained in this {@link DataHolder}
     * @param contentHandler the {@link ContentHandler} that will receive the SAX events
     * @param context The context of the data to SAX
     * @throws SAXException if an error occurs during the SAX events generation
     * @throws UnknownTypeException if there is no compatible type with the saxed value
     * @throws NotUniqueTypeException if there are many compatible types (there is no way to determine which type is the good one) with the saxed value
     */
    public void dataToSAX(ContentHandler contentHandler, DataContext context) throws SAXException, UnknownTypeException, NotUniqueTypeException;
    
    /**
     * Convert the data contained in this {@link DataHolder}
     * @return The data of the current {@link DataHolder} as JSON
     * @throws UnknownTypeException if there is no compatible type with the value to convert
     * @throws NotUniqueTypeException if there are many compatible types (there is no way to determine which type is the good one) with the value to convert
     */
    public default Map<String, Object> dataToJSON() throws UnknownTypeException, NotUniqueTypeException
    {
        return dataToJSON(DataContext.newInstance());
    }
    
    /**
     * Convert the data contained in this {@link DataHolder}
     * @param context The context of the data to convert
     * @return The data of the current {@link DataHolder} as JSON
     * @throws UnknownTypeException if there is no compatible type with the value to convert
     * @throws NotUniqueTypeException if there are many compatible types (there is no way to determine which type is the good one) with the value to convert
     */
    public Map<String, Object> dataToJSON(DataContext context) throws UnknownTypeException, NotUniqueTypeException;
    
    /**
     * Check if there are differences between the given values and the current ones
     * @param values the values to check
     * @return <code>true</code> if there are differences, <code>false</code> otherwise
     * @throws UnknownTypeException if there is no available type compatible with a given value for this data holder's type extension point
     * @throws NotUniqueTypeException if there is more than one available types compatibles with the a value for this data holder's type extension point
     */
    public boolean hasDifferences(Map<String, Object> values) throws UnknownTypeException, NotUniqueTypeException;
    
    @Override
    public Optional<? extends ModelLessDataHolder> getParentDataHolder();
    
    @Override
    public ModelLessDataHolder getRootDataHolder();
    
    /**
     * Provide the ModelItemTypeExtensionPoint linked with the DataHolder
     * @return the extension point
     */
    public ModelItemTypeExtensionPoint getModelItemTypeExtensionPoint();
}
