/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.data.holder;

import java.util.Map;
import java.util.Optional;

import org.ametys.plugins.repository.data.external.ExternalizableDataProvider.ExternalizableDataStatus;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareComposite;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.values.SynchronizationContext;
import org.ametys.plugins.repository.data.holder.values.SynchronizationResult;
import org.ametys.runtime.model.ViewItemAccessor;
import org.ametys.runtime.model.exception.BadDataPathCardinalityException;
import org.ametys.runtime.model.exception.BadItemTypeException;
import org.ametys.runtime.model.exception.UndefinedItemPathException;

/**
 * Interface for modifiable data containers with models
 */
public interface ModifiableModelAwareDataHolder extends ModifiableDataHolder, ModelAwareDataHolder
{
    @Override
    public ModifiableModelAwareComposite getComposite(String compositePath) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException;
    
    @Override
    public ModifiableModelAwareComposite getLocalComposite(String compositePath) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException;
    
    @Override
    public ModifiableModelAwareComposite getExternalComposite(String compositePath) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException;
    
    @Override
    public ModifiableModelAwareRepeater getRepeater(String repeaterPath) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException;
    
    @Override
    public ModifiableModelAwareRepeater getLocalRepeater(String repeaterPath) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException;
    
    @Override
    public ModifiableModelAwareRepeater getExternalRepeater(String repeaterPath) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException;
    
    /**
     * {@inheritDoc}
     * @throws UndefinedItemPathException if the given composite path is not defined by the model
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    @Override
    public ModifiableModelAwareComposite getComposite(String compositePath, boolean createNew) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException;
    
    /**
     * Retrieves the local composite at the given path
     * @param compositePath path of the externalizable composite to retrieve
     * @param createNew <code>true</code> to create the composite if it does not exist, <code>false</code> otherwise
     * @return the composite or <code>null</code> if createNew is <code>false</code> and value not exists or is empty
     * @throws IllegalArgumentException if the given composite path is null or empty
     * @throws BadItemTypeException if the stored value at the given path is not a composite
     * @throws UndefinedItemPathException if the given composite path is not defined by the model
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    public ModifiableModelAwareComposite getLocalComposite(String compositePath, boolean createNew) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException;
    
    /**
     * Retrieves the external composite at the given path
     * @param compositePath path of the externalizable composite to retrieve
     * @param createNew <code>true</code> to create the composite if it does not exist, <code>false</code> otherwise
     * @return the composite or <code>null</code> if createNew is <code>false</code> and value not exists or is empty
     * @throws IllegalArgumentException if the given composite path is null or empty
     * @throws BadItemTypeException if the stored value at the given path is not a composite
     * @throws UndefinedItemPathException if the given composite path is not defined by the model
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    public ModifiableModelAwareComposite getExternalComposite(String compositePath, boolean createNew) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException;
    
    /**
     * Retrieves the repeater at the given path
     * @param repeaterPath path of the repeater to retrieve
     * @param createNew <code>true</code> to create the repeater if it does not exist, <code>false</code> otherwise
     * @return the repeater or <code>null</code> if createNew is <code>false</code> and value not exists or is empty
     * @throws IllegalArgumentException if the given repeater path is null or empty
     * @throws BadItemTypeException if the stored value at the given path is not a repeater
     * @throws UndefinedItemPathException if the given composite path is not defined by the model
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    public ModifiableModelAwareRepeater getRepeater(String repeaterPath, boolean createNew) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException;
    
    /**
     * Retrieves the local repeater at the given path
     * @param repeaterPath path of the externalizable repeater to retrieve
     * @param createNew <code>true</code> to create the repeater if it does not exist, <code>false</code> otherwise
     * @return the repeater or <code>null</code> if createNew is <code>false</code> and value not exists or is empty
     * @throws IllegalArgumentException if the given repeater path is null or empty
     * @throws BadItemTypeException if the stored value at the given path is not a repeater
     * @throws UndefinedItemPathException if the given composite path is not defined by the model
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    public ModifiableModelAwareRepeater getLocalRepeater(String repeaterPath, boolean createNew) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException;
    
    /**
     * Retrieves the external repeater at the given path
     * @param repeaterPath path of the externalizable repeater to retrieve
     * @param createNew <code>true</code> to create the repeater if it does not exist, <code>false</code> otherwise
     * @return the repeater or <code>null</code> if createNew is <code>false</code> and value not exists or is empty
     * @throws IllegalArgumentException if the given repeater path is null or empty
     * @throws BadItemTypeException if the stored value at the given path is not a repeater
     * @throws UndefinedItemPathException if the given composite path is not defined by the model
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    public ModifiableModelAwareRepeater getExternalRepeater(String repeaterPath, boolean createNew) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException;
    
    /**
     * Synchronizes the given values with the current ones
     * @param <T> the type of the {@link SynchronizationResult}
     * @param values the values to synchronize
     * @return the {@link SynchronizationResult}
     * @throws UndefinedItemPathException if a key in the given Map refers to a data that is not defined by the model
     * @throws BadItemTypeException if the type defined by the model of one of the Map's key doesn't match the corresponding value
     */
    public <T extends SynchronizationResult> T synchronizeValues(Map<String, Object> values) throws UndefinedItemPathException, BadItemTypeException;
    
    /**
     * Synchronizes the given values with the current ones
     * @param <T> the type of the {@link SynchronizationResult}
     * @param values the values to synchronize
     * @param context the context of the synchronization
     * @return the {@link SynchronizationResult}
     * @throws UndefinedItemPathException if a key in the given Map refers to a data that is not defined by the model
     * @throws BadItemTypeException if the type defined by the model of one of the Map's key doesn't match the corresponding value
     */
    public <T extends SynchronizationResult> T synchronizeValues(Map<String, Object> values, SynchronizationContext context) throws UndefinedItemPathException, BadItemTypeException;
    
    /**
     * Synchronizes the given values with the current ones
     * @param <T> the type of the {@link SynchronizationResult}
     * @param viewItemAccessor The {@link ViewItemAccessor} for all items to synchronize
     * @param values the values to synchronize
     * @return the {@link SynchronizationResult}
     * @throws UndefinedItemPathException if a key in the given Map refers to a data that is not defined by the model
     * @throws BadItemTypeException if the type defined by the model of one of the Map's key doesn't match the corresponding value
     */
    public <T extends SynchronizationResult> T synchronizeValues(ViewItemAccessor viewItemAccessor, Map<String, Object> values) throws UndefinedItemPathException, BadItemTypeException;
    
    /**
     * Synchronizes the given values with the current {@link ModifiableModelAwareDataHolder}'s ones
     * @param <T> the type of the {@link SynchronizationResult}
     * @param viewItemAccessor The {@link ViewItemAccessor} for all items to synchronize
     * @param values the values to synchronize
     * @param context the context of the synchronization
     * @return the {@link SynchronizationResult}
     * @throws UndefinedItemPathException if a key in the given Map refers to a data that is not defined by the model
     * @throws BadItemTypeException if the type defined by the model of one of the Map's key doesn't match the corresponding value
     */
    public <T extends SynchronizationResult> T synchronizeValues(ViewItemAccessor viewItemAccessor, Map<String, Object> values, SynchronizationContext context) throws UndefinedItemPathException, BadItemTypeException;
    
    /**
     * Sets the value of the data at the given path
     * @param dataPath path of the data
     * @param value the value to set. Give <code>null</code> to empty the value.
     * @throws IllegalArgumentException if the given data path is null or empty
     * @throws UndefinedItemPathException if the given data path is not defined by the model
     * @throws BadItemTypeException if the type defined by the model doesn't match the given value to set
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    public void setValue(String dataPath, Object value) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException;
    
    /**
     * Sets the local value of the data at the given path
     * @param dataPath path of the externalizable data
     * @param localValue the local value to set. Give <code>null</code> to empty the value.
     * @throws IllegalArgumentException if the given data path is null or empty
     * @throws UndefinedItemPathException if the given data path is not defined by the model
     * @throws BadItemTypeException if the type defined by the model doesn't match the given value to set
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    public void setLocalValue(String dataPath, Object localValue) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException;
    
    /**
     * Sets the external value of the data at the given path
     * @param dataPath path of the externalizable data
     * @param externalValue the external value to set. Give <code>null</code> to empty the value.
     * @throws IllegalArgumentException if the given data path is null or empty
     * @throws UndefinedItemPathException if the given data path is not defined by the model
     * @throws BadItemTypeException if the type defined by the model doesn't match the given value to set
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    public void setExternalValue(String dataPath, Object externalValue) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException;

    /**
     * Set the status of the data at the given path
     * @param dataPath path of the externalizable data
     * @param status the new status
     * @throws IllegalArgumentException if the given data path is null or empty
     * @throws UndefinedItemPathException if the given data path is not defined by the model
     * @throws BadItemTypeException if there is a type issue while getting the parent (part of the dataPath without the last segment)
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    public void setStatus(String dataPath, ExternalizableDataStatus status) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException;
    
    /**
     * {@inheritDoc}
     * @throws UndefinedItemPathException if the given data path is not defined by the model
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    @Override
    public void removeValue(String dataPath) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException;
    
    /**
     * Removes the local value of the data at the given path
     * @param dataPath path of the externalizable data
     * @throws IllegalArgumentException if the given data path is null or empty
     * @throws UndefinedItemPathException if the given data path is not defined by the model
     * @throws BadItemTypeException if the value of the parent of the given path is not an item container
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    public void removeLocalValue(String dataPath) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException;
    
    /**
     * Removes the external value of the data at the given path
     * @param dataPath path of the externalizable data
     * @throws IllegalArgumentException if the given data path is null or empty
     * @throws UndefinedItemPathException if the given data path is not defined by the model
     * @throws BadItemTypeException if the value of the parent of the given path is not an item container
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    public void removeExternalValue(String dataPath) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException;
    
    /**
     * Removes the stored metadata (status and alternative value) for externalizable (or ex-externalizable) data
     * @param dataPath path of the data
     * @throws IllegalArgumentException if the given data path is null or empty
     * @throws BadItemTypeException if the value of the parent of the given path is not an item container
     * @throws UndefinedItemPathException if the given data path is not defined by the model
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    public void removeExternalizableMetadataIfExists(String dataPath) throws IllegalArgumentException, BadItemTypeException, UndefinedItemPathException, BadDataPathCardinalityException;
    
    @Override
    public Optional<? extends ModifiableModelAwareDataHolder> getParentDataHolder();
    
    @Override
    public ModifiableModelAwareDataHolder getRootDataHolder();
}
