/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.data.holder;

import java.util.Map;
import java.util.Optional;

import org.ametys.plugins.repository.data.UnknownDataException;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelLessComposite;
import org.ametys.runtime.model.exception.BadDataPathCardinalityException;
import org.ametys.runtime.model.exception.BadItemTypeException;
import org.ametys.runtime.model.exception.NotUniqueTypeException;
import org.ametys.runtime.model.exception.UnknownTypeException;

/**
 * Interface for modifiable data containers without models
 */
public interface ModifiableModelLessDataHolder extends ModifiableDataHolder, ModelLessDataHolder
{
    @Override
    public ModifiableModelLessComposite getComposite(String compositePath) throws IllegalArgumentException, BadItemTypeException, BadDataPathCardinalityException;
    
    /**
     * {@inheritDoc}
     * @throws BadDataPathCardinalityException if the value of a part of the data path is multiple. Only the last part can be multiple
     */
    @Override
    public ModifiableModelLessComposite getComposite(String compositePath, boolean createNew) throws IllegalArgumentException, BadItemTypeException, BadDataPathCardinalityException;
    
    /**
     * Synchronizes the given values with the current ones
     * @param values the values to synchronize
     * @return <code>true</code> if some values have changed, <code>false</code> otherwise
     * @throws UnknownTypeException if there is no available type compatible with a given value for this data holder's type extension point
     * @throws NotUniqueTypeException if there is more than one available types compatibles with the a value for this data holder's type extension point
     */
    public boolean synchronizeValues(Map<String, Object> values) throws UnknownTypeException, NotUniqueTypeException;
    
    /**
     * Sets the value of the data at the given path
     * @param dataPath path of the data
     * @param value the value to set. To empty a value, use {@link #setValue(String, Object, String)} with a <code>null</code> value
     * @throws IllegalArgumentException if the given data path is null or empty
     * @throws UnknownTypeException if there is no available type compatible with the given value for this data holder's type extension point
     * @throws NotUniqueTypeException if there is more than one available types compatibles with the given value for this data holder's type extension point
     * @throws UnknownDataException if given the data path is composed of an inexisting group
     * @throws BadDataPathCardinalityException if the value of a part of the data path is multiple. Only the last part can be multiple
     */
    public void setValue(String dataPath, Object value) throws IllegalArgumentException, UnknownTypeException, NotUniqueTypeException, UnknownDataException, BadDataPathCardinalityException;
    
    /**
     * Sets the value of the data at the given path
     * @param dataPath path of the data
     * @param value the value to set. Give <code>null</code> to empty the value.  
     * @param dataTypeId type identifier of the data
     * @throws IllegalArgumentException if the given data path is null or empty
     * @throws UnknownTypeException if the given type is not available for this data holder's type extension point
     * @throws BadItemTypeException if the given type doesn't match the given value to set
     * @throws UnknownDataException if the given data path is composed of an inexisting group
     * @throws BadDataPathCardinalityException if the value of a part of the data path is multiple. Only the last part can be multiple
     */
    public void setValue(String dataPath, Object value, String dataTypeId) throws IllegalArgumentException, UnknownTypeException, BadItemTypeException, UnknownDataException, BadDataPathCardinalityException;

    /**
     * {@inheritDoc}
     * @throws BadDataPathCardinalityException if the value of a part of the data path is multiple. Only the last part can be multiple
     */
    @Override
    public void removeValue(String dataPath) throws IllegalArgumentException, UnknownDataException, BadItemTypeException, BadDataPathCardinalityException;
    
    @Override
    public Optional<? extends ModifiableModelLessDataHolder> getParentDataHolder();
    
    @Override
    public ModifiableModelLessDataHolder getRootDataHolder();
}
