/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.data.holder.group;

import java.util.Collection;
import java.util.List;
import java.util.Map;

import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.plugins.repository.data.holder.DataHolder;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.plugins.repository.data.holder.values.SynchronizableRepeater;
import org.ametys.plugins.repository.data.holder.values.SynchronizationContext;
import org.ametys.plugins.repository.model.RepeaterDefinition;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ViewItemAccessor;
import org.ametys.runtime.model.exception.BadItemTypeException;
import org.ametys.runtime.model.exception.UndefinedItemPathException;
import org.ametys.runtime.model.type.DataContext;

/**
 * Class for model aware repeaters
 * 
 * Here is the format of generated SAX events for data in a repeater:
 * &lt;entry name="&lt;entry position&gt;"&gt;
 *     &lt;sax events for data1 in the current entry&gt;
 *     &lt;sax events for data2 in the current entry&gt;
 *     ...
 * &lt;\entry&gt;
 * ...
 * 
 * Here is the format of JSON object for data in a repeater:
 * {
 *     "entryCount" : &lt;size&gt;,
 *     "entries" : [
 *         {
 *             "position" : &lt;entry position&gt;,
 *             "values" : {
 *                 "&lt;data1&gt;" : &lt;value of data1 in current entry&gt;
 *                 "&lt;data2&gt;" : &lt;value of data2 in current entry&gt;
 *                 ...
 *             }
 *         },
 *         ...
 *     ]
 * }
 */
public interface ModelAwareRepeater extends Repeater
{
    @Override
    public List<? extends ModelAwareRepeaterEntry> getEntries();

    @Override
    public ModelAwareRepeaterEntry getEntry(int position);
    
    /**
     * Generates SAX events for the data in the model of the current {@link DataHolder}
     * @param contentHandler the {@link ContentHandler} that will receive the SAX events
     * @throws SAXException if an error occurs during the SAX events generation
     * @throws BadItemTypeException if the saxed value's type does not matches the stored data
     */
    public default void dataToSAX(ContentHandler contentHandler) throws SAXException, BadItemTypeException
    {
        dataToSAX(contentHandler, DataContext.newInstance());
    }
    
    /**
     * Generates SAX events for the data in the model of the current {@link DataHolder}
     * @param contentHandler the {@link ContentHandler} that will receive the SAX events
     * @param context The context of the data to SAX
     * @throws SAXException if an error occurs during the SAX events generation
     * @throws BadItemTypeException if the saxed value's type does not matches the stored data
     */
    public void dataToSAX(ContentHandler contentHandler, DataContext context) throws SAXException, BadItemTypeException;
    
    /**
     * Generates SAX events for the data in the given view in the current {@link Repeater}
     * @param contentHandler the {@link ContentHandler} that will receive the SAX events
     * @param viewItemAccessor the {@link ViewItemAccessor} referencing the items for which generate SAX events
     * @throws SAXException if an error occurs during the SAX events generation
     * @throws BadItemTypeException if the saxed value's type does not matches the stored data
     */
    public default void dataToSAX(ContentHandler contentHandler, ViewItemAccessor viewItemAccessor) throws SAXException, BadItemTypeException
    {
        dataToSAX(contentHandler, viewItemAccessor, DataContext.newInstance());
    }
    
    /**
     * Generates SAX events for the data in the given view in the current {@link Repeater}
     * @param contentHandler the {@link ContentHandler} that will receive the SAX events
     * @param viewItemAccessor the {@link ViewItemAccessor} referencing the items for which generate SAX events
     * @param context The context of the data to SAX
     * @throws SAXException if an error occurs during the SAX events generation
     * @throws BadItemTypeException if the saxed value's type does not matches the stored data
     */
    public void dataToSAX(ContentHandler contentHandler, ViewItemAccessor viewItemAccessor, DataContext context) throws SAXException, BadItemTypeException;
    
    /**
     * Generates SAX events for the data in the given view in edition mode in the current {@link Repeater}
     * @param contentHandler the {@link ContentHandler} that will receive the SAX events
     * @param viewItemAccessor the {@link ViewItemAccessor} referencing the items for which generate SAX events
     * @param context The context of the data to SAX
     * @throws SAXException if an error occurs during the SAX events generation
     * @throws BadItemTypeException if the saxed value's type does not matches the stored data
     */
    public void dataToSAXForEdition(ContentHandler contentHandler, ViewItemAccessor viewItemAccessor, DataContext context) throws SAXException, BadItemTypeException;
    
    /**
     * Convert into a JSON object the data in the model of the current {@link Repeater}
     * @return The data of the current {@link DataHolder} as JSON
     * @throws BadItemTypeException if the value's type does not matches the stored data
     */
    public default Map<String, Object> dataToJSON() throws BadItemTypeException
    {
        return dataToJSON(DataContext.newInstance());
    }
    
    /**
     * Convert into a JSON object the data in the model of the current {@link Repeater}
     * @param context The context of the data to convert
     * @return The data of the current {@link DataHolder} as JSON
     * @throws BadItemTypeException if the value's type does not matches the stored data
     */
    public Map<String, Object> dataToJSON(DataContext context) throws BadItemTypeException;
    
    /**
     * Convert into a JSON object the data in the given view of the current {@link Repeater}
     * @param viewItemAccessor the {@link ViewItemAccessor} referencing the items to convert
     * @return The data of the given view as JSON
     * @throws BadItemTypeException if the value's type does not matches the stored data
     */
    public default Map<String, Object> dataToJSON(ViewItemAccessor viewItemAccessor) throws BadItemTypeException
    {
        return dataToJSON(viewItemAccessor, DataContext.newInstance());
    }
    
    /**
     * Convert into a JSON object the data in the given view of the current {@link Repeater}
     * @param viewItemAccessor the {@link ViewItemAccessor} referencing the items to convert
     * @param context The context of the data to convert
     * @return The data of the given view as JSON
     * @throws BadItemTypeException if the value's type does not matches the stored data
     */
    public Map<String, Object> dataToJSON(ViewItemAccessor viewItemAccessor, DataContext context) throws BadItemTypeException;
    
    /**
     * Convert into a JSON object the data in the given view in edition mode of the current {@link Repeater}
     * @param viewItemAccessor the {@link ViewItemAccessor} referencing the items to convert
     * @param context The context of the data to convert
     * @return The data of the given view as JSON
     * @throws BadItemTypeException if the value's type does not matches the stored data
     */
    public Map<String, Object> dataToJSONForEdition(ViewItemAccessor viewItemAccessor, DataContext context) throws BadItemTypeException;
    
    /**
     * Check if there are differences between the given values and the repeater's entries
     * @param viewItemAccessor The {@link ViewItemAccessor} for all items to check
     * @param repeaterValues the values of the repeater to check
     * @return <code>true</code> if there are differences, <code>false</code> otherwise
     * @throws UndefinedItemPathException if an entry's key refers to a data that is not defined by the model
     * @throws BadItemTypeException if the type defined by the model of one of the entry's key doesn't match the corresponding value
     */
    public default boolean hasDifferences(ViewItemAccessor viewItemAccessor, SynchronizableRepeater repeaterValues) throws UndefinedItemPathException, BadItemTypeException
    {
        return hasDifferences(viewItemAccessor, repeaterValues, SynchronizationContext.newInstance());
    }
    
    /**
     * Check if there are differences between the given values and the repeater's entries
     * @param viewItemAccessor The {@link ViewItemAccessor} for all items to check
     * @param repeaterValues the values of the repeater to check
     * @param context the context of the synchronization
     * @return <code>true</code> if there are differences, <code>false</code> otherwise
     * @throws UndefinedItemPathException if an entry's key refers to a data that is not defined by the model
     * @throws BadItemTypeException if the type defined by the model of one of the entry's key doesn't match the corresponding value
     */
    public boolean hasDifferences(ViewItemAccessor viewItemAccessor, SynchronizableRepeater repeaterValues, SynchronizationContext context) throws UndefinedItemPathException, BadItemTypeException;
    
    /**
     * Get the collection of model items where there are differences between the given values and the repeater's entries
     * @param viewItemAccessor The {@link ViewItemAccessor} for all items to check
     * @param repeaterValues the values of the repeater to check
     * @return a collection of model items with differences
     * @throws UndefinedItemPathException if an entry's key refers to a data that is not defined by the model
     * @throws BadItemTypeException if the type defined by the model of one of the entry's key doesn't match the corresponding value
     */
    public default Collection<ModelItem> getDifferences(ViewItemAccessor viewItemAccessor, SynchronizableRepeater repeaterValues) throws UndefinedItemPathException, BadItemTypeException
    {
        return getDifferences(viewItemAccessor, repeaterValues, SynchronizationContext.newInstance());
    }

    /**
     * Get the collection of model items where there are differences between the given values and the repeater's entries
     * @param viewItemAccessor The {@link ViewItemAccessor} for all items to check
     * @param repeaterValues the values of the repeater to check
     * @param context the context of the synchronization
     * @return a collection of model items with differences
     * @throws UndefinedItemPathException if a key in the given Map refers to a data that is not defined by the model
     * @throws BadItemTypeException if the type defined by the model of one of the Map's key doesn't match the corresponding value
     */
    public Collection<ModelItem> getDifferences(ViewItemAccessor viewItemAccessor, SynchronizableRepeater repeaterValues, SynchronizationContext context) throws UndefinedItemPathException, BadItemTypeException;
    
    @Override
    public ModelAwareDataHolder getParentDataHolder();
    
    @Override
    public ModelAwareDataHolder getRootDataHolder();

    /**
     * Retrieves the repeater's model
     * @return the repeater's model
     */
    public RepeaterDefinition getModel();
}
