/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.data.holder.group;

import java.util.Map;
import java.util.Set;

import org.ametys.plugins.repository.data.UnknownDataException;
import org.ametys.plugins.repository.data.holder.ModifiableDataHolder;
import org.ametys.plugins.repository.data.repositorydata.ModifiableRepositoryData;

/**
 * Interface for modifiable repeaters
 */
public interface ModifiableRepeater extends Repeater
{
    @Override
    public ModifiableRepeaterEntry getEntry(int position);
    
    /**
     * Creates a Repeater entry at the last position.
     * @return the created entry
     */
    public ModifiableRepeaterEntry addEntry();
    
    /**
     * Creates a Repeater entry at the given position. The position starts at index 1.
     * The position can be an integer between 1 and the repeater size + 1 to insert an entry from the beginning
     * Or the position can an integer between 0 and - the repeater size to insert an entry from the end (0 means at the end, -1 means before the last one and so on)
     * @param position The position of the new entry
     * @return the created entry
     * @throws IllegalArgumentException if the position is not between the negative and positive repeater size
     */
    public ModifiableRepeaterEntry addEntry(int position) throws IllegalArgumentException;
    
    /**
     * Moves the entries according to the given position mapping.
     * The given position mapping's key is the previous position and the value is the new one
     * The existent entries with no corresponding entry in the position mapping will be removed
     * the given size is taken into account to create new empty entries if needed
     * @param positionsMapping the position mapping
     * @param targetSize the target size of the repeater
     * @return <code>true</code> if some entries have moved, <code>false</code> otherwise
     */
    public boolean moveEntries(Map<Integer, Integer> positionsMapping, int targetSize);
    
    /**
     * Removes the repeater entries at the given positions. The position starts at index 1.
     * The positions can be integers between 1 and the repeater size to remove an entry from the beginning
     * Or the positions can integers between 0 and - the repeater size to remove an entry from the end (0 means the last entry, -1 means before the last one and so on)
     * @param positions The positions of the entries to remove
     * @throws UnknownDataException if there is no entry for one of the given positions
     */
    public void removeEntries(Set<Integer> positions) throws UnknownDataException;
    
    /**
     * Removes the repeater entry at the given position. The position starts at index 1.
     * The position can be an integer between 1 and the repeater size to remove an entry from the beginning
     * Or the position can an integer between 0 and - the repeater size to remove an entry from the end (0 means the last entry, -1 means before the last one and so on)
     * @param position The position of the entry to remove
     * @throws UnknownDataException if there is no entry at the given position
     */
    public void removeEntry(int position) throws UnknownDataException;
    
    @Override
    public ModifiableRepositoryData getRepositoryData();
    
    @Override
    public ModifiableDataHolder getParentDataHolder();
    
    @Override
    public ModifiableDataHolder getRootDataHolder();
}
