/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.data.holder.group;

import java.io.IOException;
import java.util.List;
import java.util.Map;

import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.plugins.repository.data.holder.DataHolder;
import org.ametys.plugins.repository.data.repositorydata.RepositoryData;
import org.ametys.runtime.model.exception.BadItemTypeException;
import org.ametys.runtime.model.exception.UndefinedItemPathException;
import org.ametys.runtime.model.type.DataContext;

/**
 * Interface for repeaters
 * 
 * Here is the format of generated SAX events for a given data in a repeater:
 * &lt;entry name="&lt;entry position&gt;"&gt;
 *     &lt;sax events for the given data in the current entry&gt;
 * &lt;\entry&gt;
 * ...
 * 
 * Here is the format of JSON object for a given data in a repeater:
 * {
 *     "entryCount" : &lt;size&gt;,
 *     "entries" : [
 *         {
 *             "position" : &lt;entry position&gt;,
 *             "value" : &lt;value of data at given path in current entry&gt;
 *         },
 *         ...
 *     ]
 * }
 */
public interface Repeater
{
    /**
     * Retrieves the repeater entries, sorted by position
     * @return the repeater entries
     */
    public List<? extends RepeaterEntry> getEntries();
    
    /**
     * Retrieves the repeater entry at the given position. The position starts at index 1.
     * The position can be an integer between 1 and the repeater size to get an entry from the beginning
     * Or the position can an integer between 0 and - the repeater size to get an entry from the end (0 means at the end, -1 means before the last one and so on)
     * @param position the position of the entry to retrieve
     * @return the repeater entry, <code>null</code> if there is no entry at this position
     */
    public RepeaterEntry getEntry(int position);
    
    /**
     * Retrieves the size of the repeater
     * @return the size of the repeater
     */
    public int getSize();
    
    /**
     * Determines if the repeater has an entry at the given position. The position starts at index 1.
     * The position can be an integer between 1 and the repeater size to get an entry from the beginning
     * Or the position can an integer between 0 and - the repeater size to get an entry from the end (0 means at the end, -1 means before the last one and so on)
     * @param position the position
     * @return <code>true</code> if the repeater has an entry at the given position, <code>false</code> otherwise
     */
    public boolean hasEntry(int position);
    
    /**
     * Copies the current {@link Repeater} to the given {@link ModifiableRepeater}.
     * @param repeater The destination repeater. Can not be null.
     * @throws UndefinedItemPathException if one of the copied data is not defined by the model of the destination {@link ModifiableModelAwareRepeater}
     * @throws BadItemTypeException if the type defined by the model of the destination {@link ModifiableModelAwareRepeater} doesn't match the copied value
     */
    public default void copyTo(ModifiableRepeater repeater) throws UndefinedItemPathException, BadItemTypeException
    {
        copyTo(repeater, DataContext.newInstance());
    }
    
    /**
     * Copies the current {@link Repeater} to the given {@link ModifiableRepeater}.
     * @param repeater The destination repeater. Can not be null.
     * @param context The context of the data to copy
     * @throws UndefinedItemPathException if one of the copied data is not defined by the model of the destination {@link ModifiableModelAwareRepeater}
     * @throws BadItemTypeException if the type defined by the model of the destination {@link ModifiableModelAwareRepeater} doesn't match the copied value
     */
    public void copyTo(ModifiableRepeater repeater, DataContext context) throws UndefinedItemPathException, BadItemTypeException;
    
    /**
     * Generates SAX events for the data at the given data path in the current {@link DataHolder}
     * Do not generate any event if there is no values at the given path
     * @param contentHandler the {@link ContentHandler} that will receive the SAX events
     * @param dataPath the path of the data to SAX
     * @throws SAXException if an error occurs during the SAX events generation
     * @throws IOException if an error occurs while reading a value using the I/O API
     */
    public default void dataToSAX(ContentHandler contentHandler, String dataPath) throws SAXException, IOException
    {
        dataToSAX(contentHandler, dataPath, DataContext.newInstance());
    }
    
    /**
     * Generates SAX events for the data at the given data path in the current {@link DataHolder}
     * Do not generate any event if there is no values at the given path
     * @param contentHandler the {@link ContentHandler} that will receive the SAX events
     * @param dataPath the path of the data to SAX
     * @param context The context of the data to SAX
     * @throws SAXException if an error occurs during the SAX events generation
     * @throws IOException if an error occurs while reading a value using the I/O API
     */
    public void dataToSAX(ContentHandler contentHandler, String dataPath, DataContext context) throws SAXException, IOException;
    
    /**
     * Convert the data at the given path into a JSON object
     * @param dataPath the path of the data to convert
     * @return The value as JSON
     * @throws IOException if an error occurs while reading a value using the I/O API
     */
    public default Map<String, Object> dataToJSON(String dataPath) throws IOException
    {
        return dataToJSON(dataPath, DataContext.newInstance());
    }
    
    /**
     * Convert the data at the given path into a JSON object
     * @param dataPath the path of the data to convert
     * @param context The context of the data to convert
     * @return The value as JSON
     * @throws IOException if an error occurs while reading a value using the I/O API
     */
    public Map<String, Object> dataToJSON(String dataPath, DataContext context) throws IOException;
    
    /**
     * Check if has to be moved according to the given position mapping.
     * The given position mapping's key is the previous position and the value is the new one
     * The existent entries with no corresponding entry in the position mapping should be removed
     * the given size is taken into account to create new empty entries if needed
     * @param positionsMapping the position mapping
     * @param targetSize the target size of the repeater
     * @return <code>true</code> if some entries have moved, <code>false</code> otherwise
     */
    public boolean hasToMoveEntries(Map<Integer, Integer> positionsMapping, int targetSize);
    
    /**
     * Retrieves the repository data used by this {@link Repeater}
     * @return the repository data used by this {@link Repeater}
     */
    public RepositoryData getRepositoryData();
    
    /**
     * Retrieves the parent of the current {@link Repeater}
     * @return the parent of the current {@link Repeater}
     */
    public DataHolder getParentDataHolder();
    
    /**
     * Retrieves the root {@link DataHolder} of the current repeater
     * @return the root {@link DataHolder}
     */
    public DataHolder getRootDataHolder();
}
