/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.data.holder.impl;

import java.lang.reflect.Array;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.BiFunction;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Stream;

import org.apache.avalon.framework.activity.Disposable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Strings;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.core.ui.Callable;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.data.UnknownDataException;
import org.ametys.plugins.repository.data.ametysobject.ModelAwareDataAwareAmetysObject;
import org.ametys.plugins.repository.data.ametysobject.ModelLessDataAwareAmetysObject;
import org.ametys.plugins.repository.data.external.ExternalizableDataProvider.ExternalizableDataStatus;
import org.ametys.plugins.repository.data.external.ExternalizableDataProviderExtensionPoint;
import org.ametys.plugins.repository.data.holder.DataHolder;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.plugins.repository.data.holder.ModelLessDataHolder;
import org.ametys.plugins.repository.data.holder.ModifiableDataHolder;
import org.ametys.plugins.repository.data.holder.ModifiableModelAwareDataHolder;
import org.ametys.plugins.repository.data.holder.ModifiableModelLessDataHolder;
import org.ametys.plugins.repository.data.holder.group.Composite;
import org.ametys.plugins.repository.data.holder.group.ModelAwareComposite;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModelAwareRepeaterEntry;
import org.ametys.plugins.repository.data.holder.group.ModifiableComposite;
import org.ametys.plugins.repository.data.holder.group.ModifiableRepeater;
import org.ametys.plugins.repository.data.holder.group.Repeater;
import org.ametys.plugins.repository.data.holder.group.RepeaterEntry;
import org.ametys.plugins.repository.data.holder.values.SynchronizableRepeater;
import org.ametys.plugins.repository.data.holder.values.SynchronizableValue;
import org.ametys.plugins.repository.data.holder.values.SynchronizationContext;
import org.ametys.plugins.repository.data.holder.values.UntouchedValue;
import org.ametys.plugins.repository.data.holder.values.ValueContext;
import org.ametys.plugins.repository.data.type.ModelItemTypeConstants;
import org.ametys.plugins.repository.model.CompositeDefinition;
import org.ametys.plugins.repository.model.RepeaterDefinition;
import org.ametys.plugins.repository.model.RepositoryDataContext;
import org.ametys.plugins.repository.model.ViewHelper;
import org.ametys.plugins.repository.rights.ACLAmetysObjectRightAssignmentContext;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.ModelHelper;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.ModelItemAccessor;
import org.ametys.runtime.model.ModelViewItemGroup;
import org.ametys.runtime.model.ViewItemAccessor;
import org.ametys.runtime.model.exception.BadDataPathCardinalityException;
import org.ametys.runtime.model.exception.BadItemTypeException;
import org.ametys.runtime.model.exception.NotUniqueTypeException;
import org.ametys.runtime.model.exception.UndefinedItemPathException;
import org.ametys.runtime.model.exception.UnknownTypeException;
import org.ametys.runtime.model.type.DataContext;
import org.ametys.runtime.model.type.ElementType;
import org.ametys.runtime.model.type.ModelItemType;

/**
 * Helper for implementations of data holder
 */
public final class DataHolderHelper implements Component, Serviceable, Disposable
{
    /** Pattern for repeater entry : entryName[i] */
    public static final Pattern REPEATER_ENTRY_PATTERN = Pattern.compile("(.*)\\[(\\d+)\\]$");
    
    private static final Logger __LOGGER = LoggerFactory.getLogger(DataHolderHelper.class);

    private static ExternalizableDataProviderExtensionPoint _externalizableDataProviderEP;
    private static AmetysObjectResolver _resolver;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _externalizableDataProviderEP = (ExternalizableDataProviderExtensionPoint) manager.lookup(ExternalizableDataProviderExtensionPoint.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    public void dispose()
    {
        _externalizableDataProviderEP = null;
        _resolver = null;
    }
    
    /**
     * Checks if there is a repeater entry at the given position
     * @param dataHolder data holder that contains the repeater entry. The data holder must be the direct parent of the repeater
     * @param repeaterName the name of the repeater
     * @param entryPosition the position of the entry
     * @return <code>true</code> if there is an entry at the given position, <code>false</code> otherwise
     * @throws BadItemTypeException if the value stored in the repository for the given repeater name is not a repeater
     */
    public static boolean hasRepeaterEntry(ModelAwareDataHolder dataHolder, String repeaterName, int entryPosition) throws BadItemTypeException
    {
        if (dataHolder.getRepositoryData().hasValue(repeaterName))
        {
            Repeater repeater = dataHolder.getRepeater(repeaterName);
            return repeater.hasEntry(entryPosition);
        }

        return false;
    }
    
    /**
     * Checks if there is a non empty repeater entry at the given position
     * @param dataHolder data holder that contains the repeater entry. The data holder must be the direct parent of the repeater
     * @param repeaterName the name of the repeater
     * @param entryPosition the position of the entry
     * @return <code>true</code> if there is a non empty entry at the given position, <code>false</code> otherwise
     * @throws BadItemTypeException if the value stored in the repository for the given repeater name is not a repeater
     */
    public static boolean hasNonEmptyRepeaterEntry(ModelAwareDataHolder dataHolder, String repeaterName, int entryPosition) throws BadItemTypeException
    {
        if (dataHolder.getRepositoryData().hasValue(repeaterName))
        {
            Repeater repeater = dataHolder.getRepeater(repeaterName);
            if (repeater.hasEntry(entryPosition))
            {
                RepeaterEntry entry = repeater.getEntry(entryPosition);
                return !entry.getDataNames().isEmpty();
            }
        }
            
        return false;
    }
    
    /**
     * Retrieves the repeater entry at the given position
     * @param dataHolder data holder that contains the repeater entry. The data holder must be the direct parent of the repeater
     * @param repeaterName the name of the repeater
     * @param entryPosition the position of the entry
     * @return the repeater entry
     * @throws BadItemTypeException if the value stored in the repository for the given repeater name is not a repeater
     */
    public static RepeaterEntry getRepeaterEntry(ModelAwareDataHolder dataHolder, String repeaterName, int entryPosition) throws BadItemTypeException
    {
        Repeater repeater = dataHolder.getRepeater(repeaterName);
        
        if (repeater == null)
        {
            return null;
        }
        
        if (repeater.hasEntry(entryPosition))
        {
            return repeater.getEntry(entryPosition);
        }
        else
        {
            return null;
        }
    }
    
    /**
     * Test if the path is a repeater entry path (for example entries[1])
     * @param path the path representing the repeater entry
     * @return true if the pathSegment is a repeater entry path
     */
    public static boolean isRepeaterEntryPath(String path)
    {
        return REPEATER_ENTRY_PATTERN.matcher(path).matches();
    }
    
    /**
     * Retrieves the pair of repeater name and entry position of the given path segment
     * Return <code>null</code> if the given path does not represent a repeater entry
     * @param pathSegment the path segment representing the repeater entry
     * @return the pair of repeater name and entry position
     */
    public static Pair<String, Integer> getRepeaterNameAndEntryPosition(String pathSegment)
    {
        Matcher matcher = REPEATER_ENTRY_PATTERN.matcher(pathSegment);
        if (matcher.matches())
        {
            String repeaterName = matcher.group(1);
            String entryPositionAsString = matcher.group(2);
            return new ImmutablePair<>(repeaterName, Integer.parseInt(entryPositionAsString));
        }
        else
        {
            return null;
        }
    }
    
    /**
     * Checks if there is a non empty value, for the data at the given path
     * @param dataHolder the data holder
     * @param dataPath path of the data
     * @param context the context of the value to check
     * @return <code>true</code> if the data at the given path is defined by the model, if there is a non empty value for the data, and if the type of this value matches the type of the definition. <code>false</code> otherwise
     * @throws IllegalArgumentException if the given data path is null or empty
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    public static boolean hasValue(ModelAwareDataHolder dataHolder, String dataPath, ValueContext context) throws IllegalArgumentException, BadDataPathCardinalityException
    {
        if (context.getStatus().isPresent())
        {
            ExternalizableDataStatus status = context.getStatus().get();
            if (ExternalizableDataStatus.LOCAL.equals(status))
            {
                return dataHolder.hasLocalValue(dataPath);
            }
            else
            {
                return dataHolder.hasExternalValue(dataPath);
            }
        }
        else
        {
            return dataHolder.hasValue(dataPath);
        }
    }
    
    /**
     * Checks if there is value, even empty, for the data at the given path
     * @param dataHolder the data holder
     * @param dataPath path of the data
     * @param context the context of the value to check
     * @return <code>true</code> if the data at the given path is defined by the model, if there is a value, even empty, for the data, and if the type of this value matches the type of the definition. <code>false</code> otherwise
     * @throws IllegalArgumentException if the given data path is null or empty
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    public static boolean hasValueOrEmpty(ModelAwareDataHolder dataHolder, String dataPath, ValueContext context) throws IllegalArgumentException, BadDataPathCardinalityException
    {
        if (context.getStatus().isPresent())
        {
            ExternalizableDataStatus status = context.getStatus().get();
            if (ExternalizableDataStatus.LOCAL.equals(status))
            {
                return dataHolder.hasLocalValueOrEmpty(dataPath);
            }
            else
            {
                return dataHolder.hasExternalValueOrEmpty(dataPath);
            }
        }
        else
        {
            return dataHolder.hasValueOrEmpty(dataPath);
        }
    }
    
    /**
     * Retrieves the value at the given path for the data aware ametys object with given id
     * @param <T> type of the value to retrieve
     * @param ametysObjectId identifier of the data aware ametys object
     * @param dataPath path of the data
     * @return the value of the data or <code>null</code> if not exists or is empty.
     * @throws IllegalArgumentException if the given data path is null or empty
     * @throws UndefinedItemPathException if the given data path is not defined by the model
     * @throws BadItemTypeException if the type defined by the model doesn't match the type of the stored value
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    @Callable (rights = Callable.READ_ACCESS, rightContext = ACLAmetysObjectRightAssignmentContext.ID, paramIndex = 0)
    public static <T> T getValue(String ametysObjectId, String dataPath) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException
    {
        AmetysObject ametysObject = _resolver.resolveById(ametysObjectId);
        return getValue(ametysObject, dataPath);
    }
    
    /**
     * Retrieves the value at the given path for the data aware ametys object with given id
     * @param <T> type of the value to retrieve
     * @param ametysObject the data aware ametys object
     * @param dataPath path of the data
     * @return the value of the data or <code>null</code> if not exists or is empty.
     * @throws IllegalArgumentException if the given data path is null or empty
     * @throws UndefinedItemPathException if the given data path is not defined by the model
     * @throws BadItemTypeException if the type defined by the model doesn't match the type of the stored value
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    public static <T> T getValue(AmetysObject ametysObject, String dataPath)
    {
        if (ametysObject instanceof ModelAwareDataAwareAmetysObject)
        {
            return ((ModelAwareDataAwareAmetysObject) ametysObject).getValue(dataPath);
        }
        else if (ametysObject instanceof ModelLessDataAwareAmetysObject)
        {
            return ((ModelLessDataAwareAmetysObject) ametysObject).getValue(dataPath);
        }
        else
        {
            String message = String.format("Unable to retrieve the value at path '%s' from the ametys object '%s': this ametys object is not data aware.", dataPath, ametysObject.getId());
            throw new IllegalArgumentException(message);
        }
    }
    
    /**
     * Retrieves the value of the data at the given path
     * @param <T> type of the value to retrieve
     * @param dataHolder the data holder
     * @param dataPath path of the data
     * @param context the context of the value to retrieve
     * @return the value of the data or <code>null</code> if not exists or is empty.
     * @throws IllegalArgumentException if the given data path is null or empty
     * @throws UndefinedItemPathException if the given data path is not defined by the model
     * @throws BadItemTypeException if the type defined by the model doesn't match the type of the stored value
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    public static <T> T getValue(ModelAwareDataHolder dataHolder, String dataPath, ValueContext context) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException
    {
        if (context.getStatus().isPresent())
        {
            ExternalizableDataStatus status = context.getStatus().get();
            if (ExternalizableDataStatus.LOCAL.equals(status))
            {
                return dataHolder.getLocalValue(dataPath);
            }
            else
            {
                return dataHolder.getExternalValue(dataPath);
            }
        }
        else
        {
            return dataHolder.getValue(dataPath);
        }
    }
    
    /**
     * Sets the value of the data at the given path
     * @param dataHolder the data holder
     * @param dataPath path of the data
     * @param value the value to set. Give <code>null</code> to empty the value.
     * @param context context of the data to set
     * @throws IllegalArgumentException if the given data path is null or empty
     * @throws UndefinedItemPathException if the given data path is not defined by the model
     * @throws BadItemTypeException if the type defined by the model doesn't match the given value to set
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    public static void setValue(ModifiableModelAwareDataHolder dataHolder, String dataPath, Object value, ValueContext context) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException
    {
        setValue(dataHolder, dataPath, value, context, false);
    }
    
    /**
     * Sets the value of the data at the given path
     * @param dataHolder the data holder
     * @param dataPath path of the data
     * @param value the value to set. Give <code>null</code> to empty the value.
     * @param context context of the data to set
     * @param forceStatus <code>true</code> to force the status to the given status context if defined
     * @throws IllegalArgumentException if the given data path is null or empty
     * @throws UndefinedItemPathException if the given data path is not defined by the model
     * @throws BadItemTypeException if the type defined by the model doesn't match the given value to set
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    public static void setValue(ModifiableModelAwareDataHolder dataHolder, String dataPath, Object value, ValueContext context, boolean forceStatus) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, BadDataPathCardinalityException
    {
        if (context.getStatus().isPresent())
        {
            ExternalizableDataStatus status = context.getStatus().get();
            if (ExternalizableDataStatus.LOCAL.equals(status))
            {
                dataHolder.setLocalValue(dataPath, value);
            }
            else
            {
                dataHolder.setExternalValue(dataPath, value);
            }
            
            if (forceStatus)
            {
                dataHolder.setStatus(dataPath, status);
            }
        }
        else
        {
            dataHolder.setValue(dataPath, value);
        }
    }
    
    /**
     * Removes the stored value of the data at the given path
     * @param dataHolder the data holder
     * @param dataPath path of the data
     * @param context context of the data to remove
     * @throws IllegalArgumentException if the given data path is null or empty
     * @throws UnknownDataException if the value at the given data path does not exist
     * @throws BadItemTypeException if the value of the parent of the given path is not an item container
     * @throws UndefinedItemPathException if the given data path is not defined by the model
     * @throws BadDataPathCardinalityException if the definition of a part of the data path is multiple. Only the last part can be multiple
     */
    public static void removeValue(ModifiableModelAwareDataHolder dataHolder, String dataPath, ValueContext context) throws IllegalArgumentException, UndefinedItemPathException, BadItemTypeException, UnknownDataException, BadDataPathCardinalityException
    {
        if (context.getStatus().isPresent())
        {
            ExternalizableDataStatus status = context.getStatus().get();
            if (ExternalizableDataStatus.LOCAL.equals(status))
            {
                dataHolder.removeLocalValue(dataPath);
            }
            else
            {
                dataHolder.removeExternalValue(dataPath);
            }
        }
        else
        {
            dataHolder.removeValue(dataPath);
        }
    }
    
    /**
     * Copies the source {@link DataHolder} to the given {@link ModifiableDataHolder} destination.
     * @param source the source {@link DataHolder}
     * @param destination the {@link ModifiableDataHolder} destination
     * @param context The context of the data to copy
     */
    public static void copyTo(DataHolder source, ModifiableDataHolder destination, DataContext context)
    {
        for (String name : source.getDataNames())
        {
            DataContext newContext = context.cloneContext().addSegmentToDataPath(name);
            copyTo(source, destination, name, newContext);
        }
    }
    
    /**
     * Copy the data name of the source {@link DataHolder} to the given {@link ModifiableDataHolder} destination.
     * @param source the source {@link DataHolder}
     * @param destination the {@link ModifiableDataHolder} destination
     * @param name the name of the data
     * @param context The context of the data to copy
     */
    public static void copyTo(DataHolder source, ModifiableDataHolder destination, String name, DataContext context)
    {
        ModelItemType type = source instanceof ModelAwareDataHolder
                ? ((ModelAwareDataHolder) source).getType(name)
                : ((ModelLessDataHolder) source).getType(name);
        
        if (ModelItemTypeConstants.COMPOSITE_TYPE_ID.equals(type.getId()))
        {
            Composite composite = source.getComposite(name);
            ModifiableComposite compositeDestination = destination.getComposite(name, true);
            composite.copyTo(compositeDestination, context);
        }
        else if (source instanceof ModelAwareDataHolder modelAwareSource && destination instanceof ModifiableModelAwareDataHolder modelAwareDestination)
        {
            if (ModelItemTypeConstants.REPEATER_TYPE_ID.equals(type.getId()))
            {
                ModelAwareRepeater repeater = modelAwareSource.getRepeater(name);
                ModifiableRepeater repeaterDestination = modelAwareDestination.getRepeater(name, true);
                repeater.copyTo(repeaterDestination, context);
            }
            else
            {
                ModelItem modelItem = modelAwareDestination.getDefinition(name);
                if (modelItem instanceof ElementDefinition definition && definition.isEditable())
                {
                    if (context instanceof RepositoryDataContext repositoryDataContext && repositoryDataContext.copyExternalMetadata())
                    {
                        Object localValue = modelAwareSource.getLocalValue(name);
                        modelAwareDestination.setLocalValue(name, localValue);
                        Object externalValue = modelAwareSource.getExternalValue(name);
                        modelAwareDestination.setExternalValue(name, externalValue);
                        ExternalizableDataStatus status = modelAwareSource.getStatus(name);
                        modelAwareDestination.setStatus(name, status);
                    }
                    else
                    {
                        Object value = modelAwareSource.getValue(name);
                        modelAwareDestination.setValue(name, value);
                    }
                }
            }
        }
        else if (source instanceof ModelLessDataHolder modelLessSource && destination instanceof ModifiableModelLessDataHolder modelLessDestination)
        {
            Object value = modelLessSource.getValue(name);
            modelLessDestination.setValue(name, value);
        }
    }
    
    /**
     * Generates SAX events for data contained in this {@link DataHolder}
     * @param dataHolder the {@link ModelLessDataHolder} to SAX
     * @param contentHandler the {@link ContentHandler} that will receive the SAX events
     * @param context The context of the data to SAX
     * @throws SAXException if an error occurs during the SAX events generation
     * @throws UnknownTypeException if there is no compatible type with the saxed value
     * @throws NotUniqueTypeException if there are many compatible types (there is no way to determine which type is the good one) with the saxed value
     */
    public static void dataToSAX(ModelLessDataHolder dataHolder, ContentHandler contentHandler, DataContext context) throws SAXException, UnknownTypeException, NotUniqueTypeException
    {
        for (String dataName : dataHolder.getDataNames())
        {
            DataContext newContext = context.cloneContext().addSegmentToDataPath(dataName);
            dataToSAX(dataHolder, contentHandler, dataName, newContext);
        }
    }
    
    /**
     * Generates SAX events for data at the given relative path in this {@link DataHolder}
     * @param dataHolder the {@link ModelLessDataHolder} to SAX
     * @param contentHandler the {@link ContentHandler} that will receive the SAX events
     * @param relativeDataPath the path of the data to SAX, relative to the given data holder
     * @param context The context of the data to SAX
     * @throws SAXException if an error occurs during the SAX events generation
     * @throws UnknownTypeException if there is no compatible type with the saxed value
     * @throws NotUniqueTypeException if there are many compatible types (there is no way to determine which type is the good one) with the saxed value
     */
    public static void dataToSAX(ModelLessDataHolder dataHolder, ContentHandler contentHandler, String relativeDataPath, DataContext context) throws SAXException, UnknownTypeException, NotUniqueTypeException
    {
        String dataName = relativeDataPath;
        if (relativeDataPath.contains(ModelItem.ITEM_PATH_SEPARATOR))
        {
            dataName = StringUtils.substringAfterLast(relativeDataPath, ModelItem.ITEM_PATH_SEPARATOR);
        }

        if (dataHolder.hasValue(relativeDataPath)
                || context.renderEmptyValues() && dataHolder.hasValueOrEmpty(relativeDataPath))
        {
            ModelItemType type = dataHolder.getType(relativeDataPath);
            Object value = dataHolder.getValueOfType(relativeDataPath, type.getId());
            type.valueToSAX(contentHandler, dataName, value, context);
        }
    }
    
    /**
     * Convert the data contained in the given {@link DataHolder}
     * @param dataHolder the {@link ModelLessDataHolder}
     * @param context The context of the data to convert
     * @return The data of the given {@link DataHolder} as JSON
     * @throws UnknownTypeException if there is no compatible type with the value to convert
     * @throws NotUniqueTypeException if there are many compatible types (there is no way to determine which type is the good one) with the value to convert
     */
    public static Map<String, Object> dataToJSON(ModelLessDataHolder dataHolder, DataContext context) throws UnknownTypeException, NotUniqueTypeException
    {
        Map<String, Object> result = new HashMap<>();
        for (String dataName : dataHolder.getDataNames())
        {
            if (context.renderEmptyValues() || dataHolder.hasValue(dataName))
            {
                DataContext newContext = context.cloneContext().addSegmentToDataPath(dataName);
                result.put(dataName, dataToJSON(dataHolder, dataName, newContext));
            }
        }
        
        return result;
    }
    
    /**
     * Convert the data at the given relative path into a JSON object
     * @param dataHolder the {@link ModelLessDataHolder}
     * @param relativeDataPath the path of the data to convert, relative to the given data holder
     * @param context The context of the data to convert
     * @return The value as JSON
     * @throws UnknownTypeException if there is no compatible type with the value to convert
     * @throws NotUniqueTypeException if there are many compatible types (there is no way to determine which type is the good one) with the value to convert
     */
    public static Object dataToJSON(ModelLessDataHolder dataHolder, String relativeDataPath, DataContext context) throws UnknownTypeException, NotUniqueTypeException
    {
        ModelItemType type = dataHolder.getType(relativeDataPath);
        Object value = dataHolder.getValueOfType(relativeDataPath, type.getId());
        return type.valueToJSONForClient(value, context);
    }
    
    /**
     * Find all modifiable data of the given type on a {@link ModelAwareDataHolder}, including in composite or repeater entries.
     * Properties are not modifiables, so they are excluded from the result
     * @param <T> the actual type of the requested data.
     * @param dataHolder the data holder.
     * @param type the type identifier.
     * @return a Map with all data paths as keys and corresponding data as values.
     */
    public static <T extends Object> Map<String, T> findEditableItemsByType(ModelAwareDataHolder dataHolder, String type)
    {
        ViewItemAccessor viewItemAccessor = org.ametys.runtime.model.ViewHelper.createViewItemAccessor(dataHolder.getModel());
        return _findItemsByType(dataHolder, viewItemAccessor, type, false, StringUtils.EMPTY);
    }
    
    /**
     * Find all data of the given type on a {@link ModelAwareDataHolder}, including in composite or repeater entries.
     * @param <T> the actual type of the requested data.
     * @param dataHolder the data holder.
     * @param type the type identifier.
     * @return a Map with all data paths as keys and corresponding data as values.
     */
    public static <T extends Object> Map<String, T> findItemsByType(ModelAwareDataHolder dataHolder, String type)
    {
        ViewItemAccessor viewItemAccessor = org.ametys.runtime.model.ViewHelper.createViewItemAccessor(dataHolder.getModel());
        return _findItemsByType(dataHolder, viewItemAccessor, type, true, StringUtils.EMPTY);
    }
    
    /**
     * Find all data of the given type on a {@link ModelAwareDataHolder}, including in composite or repeater entries.
     * @param <T> the actual type of the requested data.
     * @param dataHolder the data holder.
     * @param viewItemAccessor the view items to restrict to
     * @param type the type identifier.
     * @return a Map with all data paths as keys and corresponding data as values.
     */
    public static <T extends Object> Map<String, T> findItemsByType(ModelAwareDataHolder dataHolder, ViewItemAccessor viewItemAccessor, String type)
    {
        return _findItemsByType(dataHolder, viewItemAccessor, type, true, StringUtils.EMPTY);
    }
    
    private static <T extends Object> Map<String, T> _findItemsByType(ModelAwareDataHolder dataHolder, ViewItemAccessor viewItemAccessor, String type, boolean includeNotEditableItems, String prefix)
    {
        Map<String, T> attributes = new HashMap<>();
        
        ViewHelper.visitView(viewItemAccessor,
            (element, definition) -> {
                // simple element
                if ((includeNotEditableItems || definition.isEditable()) && definition.getType().getId().equals(type))
                {
                    String name = definition.getName();
                    attributes.put(prefix + name, dataHolder.getValue(name));
                }
            },
            (group, definition) -> {
                // composite
                String name = definition.getName();
                ModelAwareComposite composite = dataHolder.getComposite(name);
                if (composite != null)
                {
                    attributes.putAll(_findItemsByType(composite, group, type, includeNotEditableItems, prefix + name + "/"));
                }
            },
            (group, definition) -> {
                // repeater
                String name = definition.getName();
                ModelAwareRepeater repeater = dataHolder.getRepeater(name);
                if (repeater != null)
                {
                    for (ModelAwareRepeaterEntry entry : repeater.getEntries())
                    {
                        attributes.putAll(_findItemsByType(entry, group, type, includeNotEditableItems, prefix + name + "[" + entry.getPosition() + "]/"));
                    }
                }
            },
            group -> attributes.putAll(_findItemsByType(dataHolder, group, type, includeNotEditableItems, prefix)));
        
        return attributes;
    }
    
    /**
     * Retrieve the {@link ExternalizableDataProviderExtensionPoint}
     * @return the {@link ExternalizableDataProviderExtensionPoint}
     */
    public static ExternalizableDataProviderExtensionPoint getExternalizableDataProviderExtensionPoint()
    {
        return _externalizableDataProviderEP;
    }
    
    /**
     * Creates a value context from the given {@link SynchronizationContext}
     * @param dataHolder the data holder
     * @param dataPath the path of the value needing context
     * @param synchronizationContext the {@link SynchronizationContext}
     * @return the created {@link ValueContext}
     */
    public static ValueContext createValueContextFromSynchronizationContext(ModelAwareDataHolder dataHolder, String dataPath, SynchronizationContext synchronizationContext)
    {
        ModelItem modelItem = dataHolder.getDefinition(dataPath);
        ValueContext valueContext = ValueContext.newInstance();
        
        // If the data is not externalizable at all, there is no status to set.
        // This can have an impact during values synchronization (ex: metatada for externalizable data status would be removed)
        boolean isDataExternalizableInAnyContext = getExternalizableDataProviderExtensionPoint().isDataExternalizable(dataHolder.getRootDataHolder(), modelItem);
        if (isDataExternalizableInAnyContext)
        {
            // If data is externalizable (not necessary in the current context), the value context has to be set, taking current context into account
            boolean isDataExternalizableInCurrentContext = getExternalizableDataProviderExtensionPoint().isDataExternalizable(dataHolder.getRootDataHolder(), modelItem, synchronizationContext.getExternalizableDataContext());
            if (ExternalizableDataStatus.EXTERNAL.equals(synchronizationContext.getStatusToSynchronize()) && isDataExternalizableInCurrentContext)
            {
                valueContext.withStatus(ExternalizableDataStatus.EXTERNAL);
            }
            else
            {
                valueContext.withStatus(ExternalizableDataStatus.LOCAL);
            }
        }
        
        
        return valueContext;
    }
    
    /**
     * Returns the value of the synchronized value corresponding of the future status
     * If the value is not a {@link SynchronizableValue}, the value itself is returned
     * The future status is determined from the synchronizable value itself or from the context if needed.
     * As the returned value is extracted from the synchronizable value, it can be an {@link UntouchedValue}
     * @param value the synchronizable value
     * @param dataHolder the data holder concerned by the value
     * @param modelItem the model item corresponding to the value
     * @param dataPath the current data path of the value. Can be empty if the data is in a repeater, in a not yet existing entry
     * @param context the synchronization context, used to compute the future status of the value
     * @return the value of the synchronized value corresponding of the future status
     */
    public static Object getValueFromSynchronizableValue(Object value, ModelAwareDataHolder dataHolder, ModelItem modelItem, Optional<String> dataPath, SynchronizationContext context)
    {
        if (value instanceof SynchronizableValue synchronizableValue)
        {
            Optional<ExternalizableDataStatus> valueStatus = Optional.empty();
            if (!getExternalizableDataProviderExtensionPoint().isDataExternalizable(dataHolder, modelItem, context.getExternalizableDataContext()))
            {
                // If the data is not externalizable, use the local value
                valueStatus = Optional.of(ExternalizableDataStatus.LOCAL);
            }
            else if (synchronizableValue.getExternalizableStatus() != null)
            {
                // If the given synchronizable value specifies the future status of the data,
                // the value to validate is the corresponding one
                valueStatus = Optional.of(synchronizableValue.getExternalizableStatus());
            }
            else if (dataPath.isPresent())
            {
                // Check if the status of the data is already present in the repository
                String[] pathSegments = StringUtils.split(dataPath.get(), ModelItem.ITEM_PATH_SEPARATOR);
                ModelAwareDataHolder parentDataHolder;
                String dataName;
                if (pathSegments.length == 1)
                {
                    parentDataHolder = dataHolder;
                    dataName = pathSegments[0];
                }
                else
                {
                    String parentPath = StringUtils.join(pathSegments, ModelItem.ITEM_PATH_SEPARATOR, 0, pathSegments.length - 1);
                    parentDataHolder = dataHolder.getValue(parentPath);
                    dataName = pathSegments[pathSegments.length - 1];
                }
                
                if (parentDataHolder != null && parentDataHolder.getRepositoryData().hasValue(dataName + ModelAwareDataHolder.STATUS_SUFFIX))
                {
                    // Use the current status if it is present
                    valueStatus = Optional.of(parentDataHolder.getStatus(dataName));
                }
                else if (context.forceStatusIfNotPresent())
                {
                    valueStatus = Optional.of(context.getStatusToSynchronize());
                }
            }
            else if (context.forceStatusIfNotPresent())
            {
                // The data does not exist yet, get the status from the context
                valueStatus = Optional.of(context.getStatusToSynchronize());
            }
                
            return synchronizableValue.getValue(valueStatus);
        }
        else
        {
            return value;
        }
    }
    
    /**
     * Retrieves an array of values from the given {@link SynchronizableValue}.
     * If the value is not an array, create one with a single entry
     * @param synchronizableValue the {@link SynchronizableValue}
     * @param valueContext the {@link ValueContext} to choose the value to retrieve (local or external)
     * @return an array containing the values of the {@link SynchronizableValue}
     */
    public static Object getArrayValuesFromSynchronizableValue(SynchronizableValue synchronizableValue, ValueContext valueContext)
    {
        Object values = synchronizableValue.getValue(valueContext.getStatus());
        
        if (values != null && !values.getClass().isArray())
        {
            // Create an array to put the single given value
            Object singleValue = values;
            values = Array.newInstance(singleValue.getClass(), 1);
            Array.set(values, 0, singleValue);
        }
        
        return values;
    }
    
    /**
     * Check if at least one element in the attribute path is multiple from the given parent accessor.
     * @param parent The parent accessor
     * @param attributePath The attribute path to explore
     * @return <code>false</code> if all elements in the path are single, otherwise <code>true</code> (can be a repeater or a multiple attribute)
     */
    public static boolean isMultiple(ModelItemAccessor parent, String attributePath)
    {
        String firstSegment = StringUtils.substringBefore(attributePath, ModelItem.ITEM_PATH_SEPARATOR);
        
        if (!parent.hasModelItem(firstSegment))
        {
            throw new UndefinedItemPathException("The item at path '" + firstSegment + "' is not defined in the model item accessor " + parent + ".");
        }
        
        ModelItem modelItem = parent.getModelItem(firstSegment);
        
        if (modelItem instanceof ElementDefinition definition && definition.isMultiple())
        {
            return true;
        }
        // If it is a repeater
        else if (modelItem instanceof RepeaterDefinition)
        {
            return true;
        }
        
        if (!Strings.CS.equals(firstSegment, attributePath) && modelItem instanceof ModelItemAccessor accessor)
        {
            return isMultiple(accessor, StringUtils.substringAfter(attributePath, ModelItem.ITEM_PATH_SEPARATOR));
        }
        
        return false;
    }
    
    /**
     * Converts the given values according to the definitions in the given {@link ViewItemAccessor}
     * @param viewItemAccessor the view item accessor
     * @param values the values to convert
     * @return the converted values
     */
    public static Map<String, Object> convertValues(ViewItemAccessor viewItemAccessor, Map<String, Object> values)
    {
        return convertValues(viewItemAccessor, values, DataHolderHelper::convertValue);
    }
    
    /**
     * Converts the given values according to the definitions in the given {@link ViewItemAccessor}
     * @param viewItemAccessor the view item accessor
     * @param values the values to convert
     * @param convertValueFunction The function to use to convert single values
     * @return the converted values
     */
    public static Map<String, Object> convertValues(ViewItemAccessor viewItemAccessor, Map<String, Object> values, BiFunction<ElementDefinition, Object, Object> convertValueFunction)
    {
        return _convertValues(viewItemAccessor, values, convertValueFunction, false, Optional.empty());
    }
    
    /**
     * Converts the given values according to the definitions in the given {@link ViewItemAccessor}
     * @param viewItemAccessor the view item accessor
     * @param values the values to convert
     * @param convertValueFunction The function to use to convert single values
     * @param synchronizableValuesStatusInfos informations needed to choose status of created {@link SynchronizableValue}. If empty, the status is set to {@link ExternalizableDataStatus#LOCAL}
     * @return the converted values
     */
    public static Map<String, Object> convertValuesWithSynchronizableValues(ViewItemAccessor viewItemAccessor, Map<String, Object> values, BiFunction<ElementDefinition, Object, Object> convertValueFunction, Optional<SynchronizableValuesStatusInfos> synchronizableValuesStatusInfos)
    {
        return _convertValues(viewItemAccessor, values, convertValueFunction, true, synchronizableValuesStatusInfos);
    }
    
    @SuppressWarnings("unchecked")
    private static Map<String, Object> _convertValues(ViewItemAccessor viewItemAccessor, Map<String, Object> values, BiFunction<ElementDefinition, Object, Object> convertValueFunction, boolean withSynchronizableValues, Optional<SynchronizableValuesStatusInfos> synchronizableValuesStatusInfos)
    {
        if (values == null)
        {
            return null;
        }
        
        Map<String, Object> result = new HashMap<>();
        
        ViewHelper.visitView(viewItemAccessor,
            (element, definition) -> {
                // simple element
                String name = definition.getName();
                
                if (values.containsKey(name))
                {
                    Object value = values.get(name);
                    if (value instanceof SynchronizableValue)
                    {
                        SynchronizableValue syncValue = (SynchronizableValue) value;
                        syncValue.setLocalValue(convertValueFunction.apply(definition, syncValue.getLocalValue()));
                        syncValue.setExternalValue(convertValueFunction.apply(definition, syncValue.getExternalValue()));
                        result.put(name, syncValue);
                    }
                    else
                    {
                        value = convertValueFunction.apply(definition, value);
                        if (withSynchronizableValues)
                        {
                            ExternalizableDataStatus status = ExternalizableDataStatus.LOCAL;
                            if (synchronizableValuesStatusInfos.isPresent() && getExternalizableDataProviderExtensionPoint().isDataExternalizable(synchronizableValuesStatusInfos.get().dataHolder(), definition, synchronizableValuesStatusInfos.get().externalizableContext))
                            {
                                status = ExternalizableDataStatus.EXTERNAL;
                            }
                            value = new SynchronizableValue(value, status);
                        }
                        result.put(name, value);
                    }
                }
            },
            (group, definition) -> {
                // composite
                String name = definition.getName();
                if (values.containsKey(name))
                {
                    result.put(name, _convertValues(group, (Map<String, Object>) values.get(name), convertValueFunction, withSynchronizableValues, synchronizableValuesStatusInfos));
                }
            },
            (group, definition) -> {
                // repeater
                String name = definition.getName();
                if (values.containsKey(name))
                {
                    Object value = _convertRepeaterValues(group, values.get(name), convertValueFunction, withSynchronizableValues, synchronizableValuesStatusInfos);
                    result.put(name, value);
                }
            },
            group -> result.putAll(_convertValues(group, values, convertValueFunction, withSynchronizableValues, synchronizableValuesStatusInfos)));
        
        return result;
    }
    
    private static Object _convertRepeaterValues(ModelViewItemGroup<RepeaterDefinition> group, Object value, BiFunction<ElementDefinition, Object, Object> convertValueFunction, boolean withSynchronizableValues, Optional<SynchronizableValuesStatusInfos> synchronizableValuesStatusInfos)
    {
        SynchronizableRepeater syncRepeater = value instanceof SynchronizableRepeater ? (SynchronizableRepeater) value : null;
        @SuppressWarnings("unchecked")
        List<Map<String, Object>> entries = value == null ? null : value instanceof List ? (List<Map<String, Object>>) value : ((SynchronizableRepeater) value).getEntries();
        
        Object newValue = null;
        if (entries != null)
        {
            List<Map<String, Object>> newEntries = new ArrayList<>();
            
            for (int i = 0; i < entries.size(); i++)
            {
                newEntries.add(_convertValues(group, entries.get(i), convertValueFunction, withSynchronizableValues, synchronizableValuesStatusInfos));
            }
            
            newValue = syncRepeater != null
                    ? SynchronizableRepeater.copy(syncRepeater, newEntries)
                    : withSynchronizableValues
                            ? SynchronizableRepeater.replaceAll(newEntries, null)
                            : newEntries;
        }
        
        return newValue;
    }
    
    /**
     * Converts the given value according to the given definition
     * If the value is multiple, an array is retrieved with each value converted to the right type
     * If the value is not compatible whit the data type, no exception is thrown and the value is converted to an {@link UntouchedValue} in order to be ignored
     * @param definition the definition
     * @param value the value to convert
     * @return the converted value
     */
    public static Object convertValueIgnoringIncompatibleOnes(ElementDefinition definition, Object value)
    {
        try
        {
            return convertValue(definition, value);
        }
        catch (BadItemTypeException e)
        {
            __LOGGER.error("Unable to convert the item '{}', the value '{}' is compatible with the data type ('{}'). This attribute will be ignored", definition.getPath(), value, definition.getType().getId(), e);
            return new UntouchedValue();
        }
    }
    
    /**
     * Converts the given value according to the given definition
     * If the value is multiple, an array is retrieved with each value converted to the right type
     * @param definition the definition
     * @param value the value to convert
     * @return the converted value
     * @throws BadItemTypeException if the value to convert is not compatible with the data type
     */
    public static Object convertValue(ElementDefinition definition, Object value) throws BadItemTypeException
    {
        if (value == null)
        {
            return null;
        }
        
        if (value instanceof UntouchedValue)
        {
            return value;
        }
        
        if (definition.isMultiple())
        {
            if (value instanceof Collection)
            {
                return ((Collection) value).stream()
                                           .map(v -> definition.getType().castValue(v))
                                           .toArray(i -> Array.newInstance(definition.getType().getManagedClass(), i));
            }
            else if (value.getClass().isArray())
            {
                Class<?> valueType = value.getClass().getComponentType();
                Stream<Object> valueStream;
                if (!valueType.isPrimitive())
                {
                    valueStream = Arrays.stream((Object[]) value);
                }
                else if (valueType.equals(Boolean.TYPE))
                {
                    valueStream = Arrays.stream(ArrayUtils.toObject((boolean[]) value));
                }
                else if (valueType.equals(Byte.TYPE))
                {
                    valueStream = Arrays.stream(ArrayUtils.toObject((byte[]) value));
                }
                else if (valueType.equals(Character.TYPE))
                {
                    valueStream = Arrays.stream(ArrayUtils.toObject((char[]) value));
                }
                else if (valueType.equals(Short.TYPE))
                {
                    valueStream = Arrays.stream(ArrayUtils.toObject((short[]) value));
                }
                else if (valueType.equals(Integer.TYPE))
                {
                    valueStream = Arrays.stream(ArrayUtils.toObject((int[]) value));
                }
                else if (valueType.equals(Long.TYPE))
                {
                    valueStream = Arrays.stream(ArrayUtils.toObject((long[]) value));
                }
                else if (valueType.equals(Double.TYPE))
                {
                    valueStream = Arrays.stream(ArrayUtils.toObject((double[]) value));
                }
                else if (valueType.equals(Float.TYPE))
                {
                    valueStream = Arrays.stream(ArrayUtils.toObject((float[]) value));
                }
                else
                {
                    throw new IllegalArgumentException(value + " cannot be converted to array");
                }
                
                return valueStream.map(v -> definition.getType().castValue(v)).toArray(i -> (Object[]) Array.newInstance(definition.getType().getManagedClass(), i));
            }
            
            throw new IllegalArgumentException(value + " cannot be converted to array");
        }
        else
        {
            return definition.getType().castValue(value);
        }
    }
    
    /**
     * Create a {@link ViewItemAccessor} for the items corresponding to the given values
     * @param modelItemAccessors the model containing the items corresponding to the values
     * @param values the values
     * @return the created {@link ViewItemAccessor}
     */
    public static ViewItemAccessor createViewItemAccessorFromValues(Collection<? extends ModelItemAccessor> modelItemAccessors, Map<String, Object> values)
    {
        ViewItemAccessor viewItemAccessor = org.ametys.runtime.model.ViewHelper.createEmptyViewItemAccessor(modelItemAccessors);
        _fillViewItemContainerFromValues(modelItemAccessors, viewItemAccessor, values);
        
        return viewItemAccessor;
    }
    
    @SuppressWarnings("unchecked")
    private static void _fillViewItemContainerFromValues(Collection<? extends ModelItemAccessor> modelItemAccessors, ViewItemAccessor viewItemAccessor, Map<String, Object> values)
    {
        for (String dataName : values.keySet())
        {
            ModelItem modelItem = ModelHelper.getModelItem(dataName, modelItemAccessors);
            
            if (modelItem instanceof RepeaterDefinition repeaterDefinition)
            {
                ModelViewItemGroup modelViewItemGroup = (ModelViewItemGroup) viewItemAccessor.getModelViewItem(dataName);
                if (modelViewItemGroup == null)
                {
                    modelViewItemGroup = new ModelViewItemGroup();
                    modelViewItemGroup.setDefinition(repeaterDefinition);
                    viewItemAccessor.addViewItem(modelViewItemGroup);
                }
                
                List<Map<String, Object>> entries = _getRepeaterEntriesFromValues(dataName, values);
                for (Map<String, Object> entry : entries)
                {
                    _fillViewItemContainerFromValues(List.of(repeaterDefinition), modelViewItemGroup, entry);
                }
                
            }
            else if (modelItem instanceof CompositeDefinition compositeDefinition)
            {
                ModelViewItemGroup modelViewItemGroup = (ModelViewItemGroup) viewItemAccessor.getModelViewItem(dataName);
                if (modelViewItemGroup == null)
                {
                    modelViewItemGroup = new ModelViewItemGroup();
                    modelViewItemGroup.setDefinition(compositeDefinition);
                    viewItemAccessor.addViewItem(modelViewItemGroup);
                }
                
                Map<String, Object> composite = _getCompositeFromValues(dataName, values);
                _fillViewItemContainerFromValues(List.of(compositeDefinition), modelViewItemGroup, composite);
            }
            else if (!viewItemAccessor.hasModelViewItem(dataName))
            {
                org.ametys.runtime.model.ViewHelper.addViewItem(dataName, viewItemAccessor, modelItemAccessors.toArray(new ModelItemAccessor[modelItemAccessors.size()]));
            }
        }
    }
    
    @SuppressWarnings("unchecked")
    private static List<Map<String, Object>> _getRepeaterEntriesFromValues(String dataName, Map<String, Object> values)
    {
        Object value = values.get(dataName);
        
        if (value == null)
        {
            return new ArrayList<>();
        }
        
        if (value instanceof SynchronizableRepeater synchronizableRepeater)
        {
            return synchronizableRepeater.getEntries();
        }
        
        if (value instanceof List)
        {
            return (List<Map<String, Object>>) value;
        }

        throw new BadItemTypeException("Unable to synchronize the repeater named '" + dataName + "': the given value should be a list containing its entries");
    }
    
    @SuppressWarnings("unchecked")
    private static Map<String, Object> _getCompositeFromValues(String dataName, Map<String, Object> values)
    {
        Object value = values.get(dataName);
        
        if (value == null)
        {
            return  new HashMap<>();
        }
        
        if (value instanceof Map)
        {
            return (Map<String, Object>) value;
        }
           
        throw new BadItemTypeException("Unable to synchronize the composite named '" + dataName + "': the given value should be a map containing values of all of its items");
    }
    
    /**
     * Aggregates multiple values of the data holders at the given path
     * @param <T> type of the value to retrieve
     * @param dataHolders the data holding the values
     * @param dataPath the path of the data to retrieve
     * @param managedClass class of the retrieved values
     * @return aggregated multiple values
     */
    @SuppressWarnings("unchecked")
    public static <T> T aggregateMultipleValues(List<? extends ModelAwareDataHolder> dataHolders, String dataPath, Class managedClass)
    {
        // Put all the values in a list
        List<Object> allValuesAsList = new ArrayList<>();
        for (ModelAwareDataHolder dataHolder : dataHolders)
        {
            Object value = dataHolder.getValue(dataPath, true);
            if (value != null)
            {
                if (value.getClass().isArray())
                {
                    for (int i = 0; i < Array.getLength(value); i++)
                    {
                        allValuesAsList.add(Array.get(value, i));
                    }
                }
                else
                {
                    allValuesAsList.add(value);
                }
            }
        }
        
        Object[] array = (Object[]) Array.newInstance(managedClass, allValuesAsList.size());
        return (T) allValuesAsList.toArray(array);
    }
    
    /**
     * Removes some values from the given original ones
     * @param originalValues the original values
     * @param valuesToRemove the values to remove from the original ones
     * @param elementType the type of the element concerned by the values
     * @return an array containing the original values, less the ones to remove
     */
    public static Object removeValuesInArray(Object originalValues, Object valuesToRemove, ElementType elementType)
    {
        List<Object> valuesAsList = new ArrayList<>();

        for (int i = 0; i < Array.getLength(originalValues); i++)
        {
            // for each original value
            Object originalValue = Array.get(originalValues, i);

            boolean keepOldValue = true;
            for (int j = 0; j < Array.getLength(valuesToRemove); j++)
            {
                Object valueToRemove = Array.get(valuesToRemove, j);
                if (elementType.compareValues(valueToRemove, originalValue).count() == 0)
                {
                    // If the original value is corresponding to a value to remove, do not keep it in the final values
                    keepOldValue = false;
                    break;
                }
            }
            
            if (keepOldValue)
            {
                valuesAsList.add(originalValue);
            }
        }
        
        Object[] values = (Object[]) Array.newInstance(elementType.getManagedClass(), valuesAsList.size());
        return valuesAsList.toArray(values);
    }
    
    /**
     * Append some values to the given original ones
     * @param originalValues the original values
     * @param valuesToAppend the values to append to the original ones
     * @param elementType the type of the element concerned by the values
     * @return an array containing the original values, and the ones to append
     */
    public static Object appendValuesInArray(Object originalValues, Object valuesToAppend, ElementType elementType)
    {
        // Create a array with the original values with the final size
        Object[] values = Arrays.copyOf((Object[]) originalValues, Array.getLength(originalValues) + Array.getLength(valuesToAppend));
        
        // Append each value from the given array
        for (int i = 0; i < Array.getLength(valuesToAppend); i++)
        {
            Object valueToAppend = elementType.castValue(Array.get(valuesToAppend, i));
            values[i + Array.getLength(originalValues)] = valueToAppend;
        }
        
        return values;
    }
    
    /**
     * Stores informations needed to choose status of created {@link SynchronizableValue}
     * @param dataHolder the data holder
     * @param externalizableContext the context that can be used to determine if the data is externalizable
     */
    public record SynchronizableValuesStatusInfos(ModelAwareDataHolder dataHolder, Map<String, Object> externalizableContext) { /* Empty */ }
}
