/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.data.holder.impl;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.data.UnknownDataException;
import org.ametys.plugins.repository.data.holder.DataHolder;
import org.ametys.plugins.repository.data.holder.ModelLessDataHolder;
import org.ametys.plugins.repository.data.holder.group.ModelLessComposite;
import org.ametys.plugins.repository.data.holder.group.impl.DefaultModelLessComposite;
import org.ametys.plugins.repository.data.repositorydata.RepositoryData;
import org.ametys.plugins.repository.data.type.ModelItemTypeConstants;
import org.ametys.plugins.repository.data.type.ModelItemTypeExtensionPoint;
import org.ametys.plugins.repository.data.type.RepositoryElementType;
import org.ametys.plugins.repository.data.type.RepositoryModelItemGroupType;
import org.ametys.plugins.repository.data.type.RepositoryModelItemType;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.exception.BadDataPathCardinalityException;
import org.ametys.runtime.model.exception.BadItemTypeException;
import org.ametys.runtime.model.exception.NotUniqueTypeException;
import org.ametys.runtime.model.exception.UnknownTypeException;
import org.ametys.runtime.model.type.DataContext;
import org.ametys.runtime.model.type.ElementType;

/**
 * Default implementation for data holder without model
 */
public class DefaultModelLessDataHolder implements ModelLessDataHolder
{
    /** Extension point to use to get available element types */
    protected ModelItemTypeExtensionPoint _typeExtensionPoint;
    
    /** Repository data to use to store data in the repository */
    protected RepositoryData _repositoryData;
    
    /** Parent of the current {@link DataHolder} */
    protected Optional<? extends ModelLessDataHolder> _parent;
    
    /** Root {@link DataHolder} */
    protected ModelLessDataHolder _root;
    
    /**
     * Creates a default model free data holder
     * @param typeExtensionPoint the extension point to use to get available element types
     * @param repositoryData the repository data to use
     */
    public DefaultModelLessDataHolder(ModelItemTypeExtensionPoint typeExtensionPoint, RepositoryData repositoryData)
    {
        this(typeExtensionPoint, repositoryData, Optional.empty(), Optional.empty());
    }
    
    /**
     * Creates a default model free data holder
     * @param typeExtensionPoint the extension point to use to get available element types
     * @param parent the parent of the created {@link DataHolder}, empty if the created {@link DataHolder} is the root {@link DataHolder}
     * @param root the root {@link DataHolder}
     * @param repositoryData the repository data to use
     */
    public DefaultModelLessDataHolder(ModelItemTypeExtensionPoint typeExtensionPoint, RepositoryData repositoryData, Optional<? extends ModelLessDataHolder> parent, Optional<? extends ModelLessDataHolder> root)
    {
        _typeExtensionPoint = typeExtensionPoint;
        _repositoryData = repositoryData;
        
        _parent = parent;
        _root = root.map(ModelLessDataHolder.class::cast)
                .or(() -> _parent.map(ModelLessDataHolder::getRootDataHolder)) // if no root is specified but a parent, the root is the parent's root
                .orElse(this); // if no root or parent is specified, the root is the current DataHolder
    }
    
    public ModelLessComposite getComposite(String compositePath) throws IllegalArgumentException, BadItemTypeException, BadDataPathCardinalityException
    {
        Object value = getValueOfType(compositePath, ModelItemTypeConstants.COMPOSITE_TYPE_ID);
        if (value == null)
        {
            return null;
        }
        else if (value instanceof ModelLessComposite)
        {
            return (ModelLessComposite) value;
        }
        else
        {
            throw new BadItemTypeException("The item at path '" + compositePath + "' is not a composite.");
        }
    }

    public <T> T getValue(String dataPath) throws IllegalArgumentException, UnknownTypeException, NotUniqueTypeException, BadItemTypeException, BadDataPathCardinalityException
    {
        String[] pathSegments = StringUtils.split(dataPath,  ModelItem.ITEM_PATH_SEPARATOR);
        
        if (pathSegments == null || pathSegments.length < 1)
        {
            throw new IllegalArgumentException("Unable to retrieve the data at the given path. This path is empty.");
        }
        else if (pathSegments.length == 1)
        {
            try
            {
                if (_repositoryData.hasValue(dataPath + RepositoryModelItemType.EMPTY_METADATA_SUFFIX, RepositoryConstants.NAMESPACE_PREFIX_INTERNAL))
                {
                    return null;
                }
                else
                {
                    return getValueOfType(dataPath, getType(dataPath).getId());
                }
            }
            catch (UnknownDataException e)
            {
                return null;
            }
        }
        else
        {
            // Path where current part is a data holder
            ModelLessDataHolder parent = _getParentValue(this, dataPath);
            if (parent == null)
            {
                return null;
            }
            else
            {
                String childName = dataPath.substring(dataPath.lastIndexOf(ModelItem.ITEM_PATH_SEPARATOR) + 1);
                return parent.getValue(childName);
            }
        }
    }
    
    public <T> T getValue(String dataPath, T defaultValue) throws IllegalArgumentException, UnknownTypeException, NotUniqueTypeException, BadItemTypeException, BadDataPathCardinalityException
    {
        String[] pathSegments = StringUtils.split(dataPath,  ModelItem.ITEM_PATH_SEPARATOR);
        
        if (pathSegments == null || pathSegments.length < 1)
        {
            throw new IllegalArgumentException("Unable to retrieve the data at the given path. This path is empty.");
        }
        else if (pathSegments.length == 1)
        {
            try
            {
                if (_repositoryData.hasValue(dataPath + RepositoryModelItemType.EMPTY_METADATA_SUFFIX, RepositoryConstants.NAMESPACE_PREFIX_INTERNAL))
                {
                    return defaultValue;
                }
                else
                {
                    return getValueOfType(dataPath, getType(dataPath).getId(), defaultValue);
                }
            }
            catch (UnknownDataException e)
            {
                return defaultValue;
            }
        }
        else
        {
            // Path where current part is a data holder
            ModelLessDataHolder parent = _getParentValue(this, dataPath);
            if (parent == null)
            {
                return defaultValue;
            }
            else
            {
                String childName = dataPath.substring(dataPath.lastIndexOf(ModelItem.ITEM_PATH_SEPARATOR) + 1);
                return parent.getValue(childName, defaultValue);
            }
        }
    }
    
    public <T> T getValueOfType(String dataPath, String dataTypeId, T defaultValue) throws IllegalArgumentException, UnknownTypeException, BadItemTypeException, BadDataPathCardinalityException
    {
        if (hasValue(dataPath, dataTypeId))
        {
            return getValueOfType(dataPath, dataTypeId);
        }
        
        return defaultValue;
    }
    
    @SuppressWarnings("unchecked")
    public <T> T getValueOfType(String dataPath, String dataTypeId) throws IllegalArgumentException, UnknownTypeException, BadItemTypeException, BadDataPathCardinalityException
    {
        if (!_typeExtensionPoint.hasExtension(dataTypeId))
        {
            String availableTypes = StringUtils.join(_typeExtensionPoint.getExtensionsIds(), ", ");
            throw new UnknownTypeException("The type '" + dataTypeId + "' is not available for the extension point '" + _typeExtensionPoint + "'. Available types are: '" + availableTypes + "'.");
        }
        
        String[] pathSegments = StringUtils.split(dataPath,  ModelItem.ITEM_PATH_SEPARATOR);
        
        if (pathSegments == null || pathSegments.length < 1)
        {
            throw new IllegalArgumentException("Unable to retrieve the data at the given path. This path is empty.");
        }
        else if (pathSegments.length == 1)
        {
            // Simple path => get the value
            RepositoryModelItemType type = (RepositoryModelItemType) _typeExtensionPoint.getExtension(dataTypeId);
            if (type instanceof RepositoryElementType)
            {
                return (T) ((RepositoryElementType) type).read(_repositoryData, dataPath);
            }
            else
            {
                return (T) _getComposite(dataPath);
            }
        }
        else
        {
            // Path where current part is a data holder
            ModelLessDataHolder parent = _getParentValue(this, dataPath);
            if (parent == null)
            {
                return null;
            }
            else
            {
                String childName = dataPath.substring(dataPath.lastIndexOf(ModelItem.ITEM_PATH_SEPARATOR) + 1);
                return parent.getValueOfType(childName, dataTypeId);
            }
        }
    }
    
    /**
     * Retrieves the composite with the given name
     * @param name name of the composite to retrieve
     * @return the composite
     * @throws BadItemTypeException if the value stored in the repository with the given name is not a composite
     */
    protected ModelLessComposite _getComposite(String name) throws BadItemTypeException
    {
        RepositoryModelItemGroupType type = (RepositoryModelItemGroupType) _typeExtensionPoint.getExtension(ModelItemTypeConstants.COMPOSITE_TYPE_ID);
        RepositoryData compositeRepositoryData = type.read(_repositoryData, name);
        
        if (compositeRepositoryData != null)
        {
            return new DefaultModelLessComposite(_typeExtensionPoint, compositeRepositoryData, this, _root);
        }
        else
        {
            return null;
        }
    }
    
    public boolean hasValue(String dataPath) throws IllegalArgumentException, BadDataPathCardinalityException
    {
        String[] pathSegments = StringUtils.split(dataPath, ModelItem.ITEM_PATH_SEPARATOR);
        
        if (pathSegments == null || pathSegments.length < 1)
        {
            throw new IllegalArgumentException("The data path is empty. It is not possible to determine if it has a value or not.");
        }
        else if (pathSegments.length == 1)
        {
            try
            {
                RepositoryModelItemType type = getType(dataPath);
                return hasValue(dataPath, type.getId());
            }
            catch (UnknownDataException | UnknownTypeException | NotUniqueTypeException e)
            {
                return false;
            }
        }
        else
        {
            try
            {
                // Path where current part is a data holder
                ModelLessDataHolder parent = _getParentValue(this, dataPath);
                if (parent == null)
                {
                    return false;
                }
                else
                {
                    String childName = StringUtils.substringAfterLast(dataPath, ModelItem.ITEM_PATH_SEPARATOR);
                    return parent.hasValue(childName);
                }
            }
            catch (BadItemTypeException e)
            {
                return false;
            }
        }
    }
    
    public boolean hasValue(String dataPath, String dataTypeId) throws IllegalArgumentException, UnknownTypeException, BadDataPathCardinalityException
    {
        if (!_typeExtensionPoint.hasExtension(dataTypeId))
        {
            String availableTypes = StringUtils.join(_typeExtensionPoint.getExtensionsIds(), ", ");
            throw new UnknownTypeException("The type '" + dataTypeId + "' is not available for the extension point '" + _typeExtensionPoint + "'. Available types are: '" + availableTypes + "'.");
        }
        
        String[] pathSegments = StringUtils.split(dataPath, ModelItem.ITEM_PATH_SEPARATOR);
        
        if (pathSegments == null || pathSegments.length < 1)
        {
            throw new IllegalArgumentException("The data path is empty. It is not possible to determine if it has a value or not.");
        }
        else if (pathSegments.length == 1)
        {
            try
            {
                RepositoryModelItemType type = (RepositoryModelItemType) _typeExtensionPoint.getExtension(dataTypeId);
                return type.hasNonEmptyValue(_repositoryData, dataPath);
            }
            catch (BadItemTypeException e)
            {
                return false;
            }
        }
        else
        {
            try
            {
                // Path where current part is a data holder
                ModelLessDataHolder parent = _getParentValue(this, dataPath);
                if (parent == null)
                {
                    return false;
                }
                else
                {
                    String childName = StringUtils.substringAfterLast(dataPath, ModelItem.ITEM_PATH_SEPARATOR);
                    return parent.hasValue(childName, dataTypeId);
                }
            }
            catch (BadItemTypeException e)
            {
                return false;
            }
        }
    }
    
    public boolean hasValueOrEmpty(String dataPath) throws IllegalArgumentException, BadDataPathCardinalityException
    {
        String[] pathSegments = StringUtils.split(dataPath, ModelItem.ITEM_PATH_SEPARATOR);
        
        if (pathSegments == null || pathSegments.length < 1)
        {
            throw new IllegalArgumentException("The data path is empty. It is not possible to determine if it has a value or not.");
        }
        else if (pathSegments.length == 1)
        {
            return _repositoryData.hasValue(dataPath) || _repositoryData.hasValue(dataPath + RepositoryModelItemType.EMPTY_METADATA_SUFFIX, RepositoryConstants.NAMESPACE_PREFIX_INTERNAL);
        }
        else
        {
            try
            {
                // Path where current part is a data holder
                ModelLessDataHolder parent = _getParentValue(this, dataPath);
                if (parent == null)
                {
                    return false;
                }
                else
                {
                    String childName = StringUtils.substringAfterLast(dataPath, ModelItem.ITEM_PATH_SEPARATOR);
                    return parent.hasValueOrEmpty(childName);
                }
            }
            catch (BadItemTypeException e)
            {
                return false;
            }
        }
    }
    
    public boolean isMultiple(String dataPath) throws IllegalArgumentException, UnknownDataException, UnknownTypeException, NotUniqueTypeException, BadItemTypeException, BadDataPathCardinalityException
    {
        String[] pathSegments = StringUtils.split(dataPath, ModelItem.ITEM_PATH_SEPARATOR);
        
        if (pathSegments == null || pathSegments.length < 1)
        {
            throw new IllegalArgumentException("The data path is empty. It is not possible to determine if it is multiple or not.");
        }
        else if (pathSegments.length == 1)
        {
            return isMultiple(dataPath, getType(dataPath).getId());
        }
        else
        {
            // Path where current part is a data holder
            ModelLessDataHolder parent = _getParentValue(this, dataPath);
            if (parent == null)
            {
                throw new UnknownDataException("The data at path '" + dataPath + "' does not exist. It is not possible to determine if it is multiple or not.");
            }
            else
            {
                String childName = dataPath.substring(dataPath.lastIndexOf(ModelItem.ITEM_PATH_SEPARATOR) + 1);
                return parent.isMultiple(childName);
            }
        }
    }
    
    public boolean isMultiple(String dataPath, String dataTypeId) throws IllegalArgumentException, UnknownDataException, UnknownTypeException, BadItemTypeException, BadDataPathCardinalityException
    {
        if (!_typeExtensionPoint.hasExtension(dataTypeId))
        {
            String availableTypes = StringUtils.join(_typeExtensionPoint.getExtensionsIds(), ", ");
            throw new UnknownTypeException("The type '" + dataTypeId + "' is not available for the extension point '" + _typeExtensionPoint + "'. Available types are: '" + availableTypes + "'.");
        }
        
        String[] pathSegments = StringUtils.split(dataPath, ModelItem.ITEM_PATH_SEPARATOR);
        
        if (pathSegments == null || pathSegments.length < 1)
        {
            throw new IllegalArgumentException("The data path is empty. It is not possible to determine if it is multiple or not.");
        }
        else if (pathSegments.length == 1)
        {
            RepositoryModelItemType type = (RepositoryModelItemType) _typeExtensionPoint.getExtension(dataTypeId);
            return type.isMultiple(_repositoryData, dataPath);
        }
        else
        {
            // Path where current part is a data holder
            ModelLessDataHolder parent = _getParentValue(this, dataPath);
            if (parent == null)
            {
                throw new UnknownDataException("The data at path '" + dataPath + "' does not exist. It is not possible to determine if it is multiple or not.");
            }
            else
            {
                String childName = dataPath.substring(dataPath.lastIndexOf(ModelItem.ITEM_PATH_SEPARATOR) + 1);
                return parent.isMultiple(childName, dataTypeId);
            }
        }
    }
    
    public RepositoryModelItemType getType(String dataPath) throws IllegalArgumentException, UnknownDataException, UnknownTypeException, NotUniqueTypeException, BadDataPathCardinalityException
    {
        String[] pathSegments = StringUtils.split(dataPath, ModelItem.ITEM_PATH_SEPARATOR);
        
        if (pathSegments == null || pathSegments.length < 1)
        {
            throw new IllegalArgumentException("The data path is empty. It is not possible to determine its type.");
        }
        else if (pathSegments.length == 1)
        {
            return _getType(dataPath);
        }
        else
        {
            // Path where current part is a data holder
            ModelLessDataHolder parent = _getParentValue(this, dataPath);
            if (parent == null)
            {
                throw new UnknownDataException("The data at path '" + dataPath + "' does not exist. It is not possible to determine its type.");
            }
            else
            {
                String childName = dataPath.substring(dataPath.lastIndexOf(ModelItem.ITEM_PATH_SEPARATOR) + 1);
                return parent.getType(childName);
            }
        }
    }
    
    private RepositoryModelItemType _getType(String dataName) throws UnknownDataException, UnknownTypeException, NotUniqueTypeException
    {
        List<RepositoryModelItemType> compatibleTypes = _typeExtensionPoint.getExtensionsIds().stream()
                .map(id -> _typeExtensionPoint.getExtension(id))
                .filter(RepositoryModelItemType.class::isInstance)
                .map(RepositoryModelItemType.class::cast)
                .filter(type -> type.isCompatible(_repositoryData, dataName))
                .collect(Collectors.toList());
        if (compatibleTypes.isEmpty())
        {
            // The type has not been found, thrown an UnknownTypeException
            throw new UnknownTypeException("Unable to retrieve the type of the data '" + dataName + "'. No compatible type have been found.");
        }
        else if (compatibleTypes.size() > 1)
        {
            // Many types have been found, thrown an UnknownTypeException
            List<String> compatibleTypesIds = compatibleTypes.stream().map(type -> type.getId()).collect(Collectors.toList());
            throw new NotUniqueTypeException("Unable to retrieve the type of the data '" + dataName + "'. Many compatible types have been found, there is no way to determine which one is the good one. Compatible types found are: " + StringUtils.join(compatibleTypesIds, ", "));
        }
        else
        {
            return compatibleTypes.get(0);
        }
    }
    
    /**
     * Retrieves the data holder, last parent segment of the given data path
     * Example : call this method with a path like 'my-composite1/my-composite2/my-data' will retrieve the composite 'my-composite2' in the composite 'my-composite1'
     * @param dataHolder the data holder
     * @param dataPath the data path
     * @return the parent data holder
     * @throws BadItemTypeException if the value at the given data path is not a composite
     * @throws BadDataPathCardinalityException if the value of a part of the data path is multiple. Only the last part can be multiple
     */
    protected static ModelLessDataHolder _getParentValue(ModelLessDataHolder dataHolder, String dataPath) throws BadItemTypeException, BadDataPathCardinalityException
    {
        String[] pathSegments = StringUtils.split(dataPath,  ModelItem.ITEM_PATH_SEPARATOR);
        String parentPath = StringUtils.join(pathSegments, ModelItem.ITEM_PATH_SEPARATOR, 0, pathSegments.length - 1);
        
        return dataHolder.getValueOfType(parentPath, ModelItemTypeConstants.COMPOSITE_TYPE_ID);
    }
    
    public Collection<String> getDataNames()
    {
        return _repositoryData.getDataNames();
    }

    public void dataToSAX(ContentHandler contentHandler, String dataPath, DataContext context) throws SAXException, UnknownTypeException, NotUniqueTypeException
    {
        DataHolderHelper.dataToSAX(this, contentHandler, dataPath, context.cloneContext().withDataPath(dataPath));
    }
    
    public void dataToSAX(ContentHandler contentHandler, DataContext context) throws SAXException, UnknownTypeException, NotUniqueTypeException
    {
        DataHolderHelper.dataToSAX(this, contentHandler, context);
    }
    
    public Object dataToJSON(String dataPath, DataContext context) throws UnknownTypeException, NotUniqueTypeException
    {
        return DataHolderHelper.dataToJSON(this, dataPath, context);
    }
    
    public Map<String, Object> dataToJSON(DataContext context) throws UnknownTypeException, NotUniqueTypeException
    {
        return DataHolderHelper.dataToJSON(this, context);
    }
    
    @SuppressWarnings("unchecked")
    public boolean hasDifferences(Map<String, Object> values) throws UnknownTypeException, NotUniqueTypeException
    {
        for (Map.Entry<String, Object> entry : values.entrySet())
        {
            String dataName = entry.getKey();
            Object value = entry.getValue();
            
            if (value instanceof Map)
            {
                ModelLessComposite composite = getComposite(dataName);
                if (composite.hasDifferences((Map<String, Object>) value))
                {
                    return true;
                }
            }
            else
            {
                if (_repositoryData.hasValue(dataName + RepositoryModelItemType.EMPTY_METADATA_SUFFIX, RepositoryConstants.NAMESPACE_PREFIX_INTERNAL))
                {
                    // The old value is set as empty via an internal boolean property
                    if (value != null)
                    {
                        // the new value is not empty
                        return true;
                    }
                }
                else if (hasValueOrEmpty(dataName))
                {
                    ElementType type = (ElementType) getType(dataName);
                    Object oldValue = getValue(dataName);
                    
                    if (type.compareValues(value, oldValue).count() > 0)
                    {
                        return true;
                    }
                }
                else
                {
                    // There is no old value. If the given value is null, there is no way to determine the type of the value to set
                    if (value != null)
                    {
                        return true;
                    }
                }
            }
        }
        
        return false;
    }
    
    public RepositoryData getRepositoryData()
    {
        return _repositoryData;
    }
    
    public Optional<? extends ModelLessDataHolder> getParentDataHolder()
    {
        return _parent;
    }
    
    public ModelLessDataHolder getRootDataHolder()
    {
        return _root;
    }
    
    public ModelItemTypeExtensionPoint getModelItemTypeExtensionPoint()
    {
        return _typeExtensionPoint;
    }
}
