/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.data.holder.impl;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;

import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.data.UnknownDataException;
import org.ametys.plugins.repository.data.holder.DataHolder;
import org.ametys.plugins.repository.data.holder.ModifiableModelLessDataHolder;
import org.ametys.plugins.repository.data.holder.group.ModelLessComposite;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelLessComposite;
import org.ametys.plugins.repository.data.holder.group.impl.DefaultModifiableModelLessComposite;
import org.ametys.plugins.repository.data.repositorydata.ModifiableRepositoryData;
import org.ametys.plugins.repository.data.repositorydata.RepositoryData;
import org.ametys.plugins.repository.data.type.ModelItemTypeConstants;
import org.ametys.plugins.repository.data.type.ModelItemTypeExtensionPoint;
import org.ametys.plugins.repository.data.type.RepositoryElementType;
import org.ametys.plugins.repository.data.type.RepositoryModelItemGroupType;
import org.ametys.plugins.repository.data.type.RepositoryModelItemType;
import org.ametys.plugins.repository.lock.LockableAmetysObject;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.exception.BadDataPathCardinalityException;
import org.ametys.runtime.model.exception.BadItemTypeException;
import org.ametys.runtime.model.exception.NotUniqueTypeException;
import org.ametys.runtime.model.exception.UnknownTypeException;
import org.ametys.runtime.model.type.ElementType;

/**
 * Default implementation for modifiable data holder without model
 */
public class DefaultModifiableModelLessDataHolder extends DefaultModelLessDataHolder implements ModifiableModelLessDataHolder
{
    /** Repository data to use to store data in the repository */
    protected ModifiableRepositoryData _modifiableRepositoryData;
    
    /** Ametys object that can be locked on data modification */
    protected Optional<LockableAmetysObject> _lockableAmetysObject;
    
    /** Parent of the current {@link DataHolder} */
    protected Optional<? extends ModifiableModelLessDataHolder> _modifiableParent;
    
    /** Root {@link DataHolder} */
    protected ModifiableModelLessDataHolder _modifiableRoot;
    
    /**
     * Creates a modifiable default model free data holder
     * @param typeExtensionPoint the extension point to use to get available element types
     * @param repositoryData the repository data to use
     */
    public DefaultModifiableModelLessDataHolder(ModelItemTypeExtensionPoint typeExtensionPoint, ModifiableRepositoryData repositoryData)
    {
        this(typeExtensionPoint, repositoryData, Optional.empty(), Optional.empty(), Optional.empty());
    }
    
    /**
     * Creates a modifiable default model free data holder
     * @param typeExtensionPoint the extension point to use to get available element types
     * @param repositoryData the repository data to use
     * @param lockableAmetysObject the ametys object that can be locked on data modification
     */
    public DefaultModifiableModelLessDataHolder(ModelItemTypeExtensionPoint typeExtensionPoint, ModifiableRepositoryData repositoryData, Optional<LockableAmetysObject> lockableAmetysObject)
    {
        this(typeExtensionPoint, repositoryData, lockableAmetysObject, Optional.empty(), Optional.empty());
    }
    
    /**
     * Creates a modifiable default model free data holder
     * @param typeExtensionPoint the extension point to use to get available element types
     * @param parent the parent of the created {@link DataHolder}, can be <code>null</code> if the created {@link DataHolder} is the root {@link DataHolder}
     * @param lockableAmetysObject the ametys object that can be locked on data modification
     * @param root the root {@link DataHolder}
     * @param repositoryData the repository data to use
     */
    public DefaultModifiableModelLessDataHolder(ModelItemTypeExtensionPoint typeExtensionPoint, ModifiableRepositoryData repositoryData, Optional<LockableAmetysObject> lockableAmetysObject, Optional<? extends ModifiableModelLessDataHolder> parent, Optional<? extends ModifiableModelLessDataHolder> root)
    {
        super(typeExtensionPoint, repositoryData, parent, root);
        _modifiableRepositoryData = repositoryData;
        _lockableAmetysObject = lockableAmetysObject;
        _modifiableParent = parent;
        _modifiableRoot = root.map(ModifiableModelLessDataHolder.class::cast)
                .or(() -> _modifiableParent.map(ModifiableModelLessDataHolder::getRootDataHolder)) // if no root is specified but a parent, the root is the parent's root
                .orElse(this); // if no root or parent is specified, the root is the current DataHolder
    }
    
    @Override
    public ModifiableModelLessComposite getComposite(String compositePath) throws IllegalArgumentException, BadItemTypeException
    {
        return (ModifiableModelLessComposite) super.getComposite(compositePath);
    }
    
    @Override
    protected ModelLessComposite _getComposite(String name) throws BadItemTypeException
    {
        return _getComposite(name, false);
    }

    public ModifiableModelLessComposite getComposite(String compositePath, boolean createNew) throws IllegalArgumentException, BadItemTypeException
    {
        if (!_typeExtensionPoint.hasExtension(ModelItemTypeConstants.COMPOSITE_TYPE_ID))
        {
            throw new UnknownTypeException("The composites are not available for the extension point '" + _typeExtensionPoint + "'");
        }
        
        String[] pathSegments = StringUtils.split(compositePath, ModelItem.ITEM_PATH_SEPARATOR);
        
        if (pathSegments == null || pathSegments.length < 1)
        {
            throw new IllegalArgumentException("Unable to retrieve the composite at the given path. This path is empty.");
        }
        else if (pathSegments.length == 1)
        {
            // Simple path => get the composite
            return _getComposite(compositePath, createNew);
        }
        else
        {
            // Path where current part is a data holder
            ModifiableModelLessDataHolder parent = _getParentValue(this, compositePath);
            if (parent == null)
            {
                return null;
            }
            else
            {
                String childName = compositePath.substring(compositePath.lastIndexOf(ModelItem.ITEM_PATH_SEPARATOR) + 1);
                return parent.getComposite(childName, createNew);
            }
        }
    }
    
    /**
     * Retrieves the composite with the given name
     * @param name name of the composite to retrieve
     * @param createNew <code>true</code> to create the composite if it does not exist, <code>false</code> otherwise
     * @return the composite
     * @throws BadItemTypeException if the value stored in the repository with the given name is not a composite
     */
    protected ModifiableModelLessComposite _getComposite(String name, boolean createNew) throws BadItemTypeException
    {
        RepositoryModelItemGroupType type = (RepositoryModelItemGroupType) _typeExtensionPoint.getExtension(ModelItemTypeConstants.COMPOSITE_TYPE_ID);
        RepositoryData compositeRepositoryData = type.read(_modifiableRepositoryData, name);
        
        if (compositeRepositoryData != null)
        {
            return new DefaultModifiableModelLessComposite(_typeExtensionPoint, (ModifiableRepositoryData) compositeRepositoryData, _lockableAmetysObject, this, _modifiableRoot);
        }
        else
        {
            if (createNew)
            {
                _lockableAmetysObject.ifPresent(lao -> lao.setLockInfoOnCurrentContext());
                ModifiableRepositoryData createdRepositoryData = type.add(_modifiableRepositoryData, name);
                return new DefaultModifiableModelLessComposite(_typeExtensionPoint, createdRepositoryData, _lockableAmetysObject, this, _modifiableRoot);
            }
            else
            {
                return null;
            }
        }
    }
    
    @SuppressWarnings("unchecked")
    public boolean synchronizeValues(Map<String, Object> values) throws UnknownTypeException, NotUniqueTypeException
    {
        boolean hasChanged = false;
        
        for (Map.Entry<String, Object> entry : values.entrySet())
        {
            String dataName = entry.getKey();
            Object value = entry.getValue();
            
            if (value instanceof Map)
            {
                ModifiableModelLessComposite composite = getComposite(dataName, true);
                hasChanged = composite.synchronizeValues((Map<String, Object>) value) || hasChanged;
            }
            else
            {
                if (_repositoryData.hasValue(dataName + RepositoryModelItemType.EMPTY_METADATA_SUFFIX, RepositoryConstants.NAMESPACE_PREFIX_INTERNAL))
                {
                    // The old value is set as empty via an internal boolean property
                    if (value != null)
                    {
                        // the new value is not empty
                        hasChanged = true;
                        setValue(dataName, value);
                    }
                }
                else if (hasValueOrEmpty(dataName))
                {
                    ElementType type = (ElementType) getType(dataName);
                    Object oldValue = getValue(dataName);
                    
                    if (type.compareValues(value, oldValue).count() > 0)
                    {
                        hasChanged = true;
                        setValue(dataName, value);
                    }
                }
                else
                {
                    // There is no old value. If the given value is null, there is no way to determine the type of the value to set
                    if (value != null)
                    {
                        hasChanged = true;
                        setValue(dataName, value);
                    }
                }
            }
        }
        
        return hasChanged;
    }

    public void setValue(String dataPath, Object value) throws IllegalArgumentException, UnknownTypeException, NotUniqueTypeException, UnknownDataException
    {
        List<RepositoryModelItemType> compatibleTypes = _typeExtensionPoint.getExtensionsIds().stream()
                .map(id -> _typeExtensionPoint.getExtension(id))
                .filter(RepositoryElementType.class::isInstance)
                .map(RepositoryElementType.class::cast)
                .filter(type -> type.isCompatible(value))
                .collect(Collectors.toList());
        
        if (compatibleTypes.isEmpty())
        {
            // The type has not been found, throw an UnknownTypeException
            String availableTypes = StringUtils.join(_typeExtensionPoint.getExtensionsIds(), ", ");
            throw new UnknownTypeException("Unable to retrieve the type of the data '" + dataPath + "'. No compatible type have been found. Available types are: '" + availableTypes + "'.");
        }
        else if (compatibleTypes.size() > 1)
        {
            // Many types have been found, throw an UnknownTypeException
            List<String> compatibleTypesIds = compatibleTypes.stream().map(type -> type.getId()).collect(Collectors.toList());
            throw new NotUniqueTypeException("Unable to retrieve the type of the data '" + dataPath + "'. Many compatible types have been found, there is no way to determine which one is the good one. Compatible types found are: " + StringUtils.join(compatibleTypesIds, ", "));
        }
        else
        {
            setValue(dataPath, value, compatibleTypes.get(0).getId());
        }
    }

    public void setValue(String dataPath, Object value, String dataTypeId) throws IllegalArgumentException, BadItemTypeException, UnknownDataException
    {
        if (!_typeExtensionPoint.hasExtension(dataTypeId))
        {
            String availableTypes = StringUtils.join(_typeExtensionPoint.getExtensionsIds(), ", ");
            throw new UnknownTypeException("The type '" + dataTypeId + "' is not available for the extension point '" + _typeExtensionPoint + "'. Available types are: '" + availableTypes + "'.");
        }
        
        String[] pathSegments = StringUtils.split(dataPath, ModelItem.ITEM_PATH_SEPARATOR);
        
        if (pathSegments == null || pathSegments.length < 1)
        {
            throw new IllegalArgumentException("Unable to set a value at the given path. This path is empty.");
        }
        else if (pathSegments.length == 1)
        {
            // Simple path => set the value
            RepositoryModelItemType type = (RepositoryModelItemType) _typeExtensionPoint.getExtension(dataTypeId);
            if (type instanceof RepositoryElementType)
            {
                _lockableAmetysObject.ifPresent(lao -> lao.setLockInfoOnCurrentContext());
                ((RepositoryElementType) type).write(_modifiableRepositoryData, dataPath, value);
            }
            else
            {
                throw new BadItemTypeException("Unable to set the value '" + value + "' on the data at path '" + dataPath + "' in the repository because it is a group item.");
            }
        }
        else
        {
            // Path where current part is a data holder
            ModifiableModelLessDataHolder parent = _getParentValue(this, dataPath);
            if (parent == null)
            {
                String parentPath = StringUtils.join(pathSegments, ModelItem.ITEM_PATH_SEPARATOR, 0, pathSegments.length - 1);
                throw new UnknownDataException("The data at path '" + parentPath + "' does not exist. It is not possible to get the data at path '" + dataPath + "'.");
            }
            else
            {
                String childName = dataPath.substring(dataPath.lastIndexOf(ModelItem.ITEM_PATH_SEPARATOR) + 1);
                parent.setValue(childName, value, dataTypeId);
            }
        }
    }

    public void removeValue(String dataPath) throws IllegalArgumentException, BadItemTypeException
    {
        String[] pathSegments = StringUtils.split(dataPath, ModelItem.ITEM_PATH_SEPARATOR);
        
        if (pathSegments == null || pathSegments.length < 1)
        {
            throw new IllegalArgumentException("Unable to remove the value at the given path. This path is empty.");
        }
        else if (pathSegments.length == 1)
        {
            if (_modifiableRepositoryData.hasValue(dataPath))
            {
                _lockableAmetysObject.ifPresent(lao -> lao.setLockInfoOnCurrentContext());
                _modifiableRepositoryData.removeValue(dataPath);
            }

            if (_modifiableRepositoryData.hasValue(dataPath + RepositoryModelItemType.EMPTY_METADATA_SUFFIX, RepositoryConstants.NAMESPACE_PREFIX_INTERNAL))
            {
                _lockableAmetysObject.ifPresent(lao -> lao.setLockInfoOnCurrentContext());
                _modifiableRepositoryData.removeValue(dataPath + RepositoryModelItemType.EMPTY_METADATA_SUFFIX, RepositoryConstants.NAMESPACE_PREFIX_INTERNAL);
            }

        }
        else
        {
            // Path where current part is a data holder
            ModifiableModelLessDataHolder parent = _getParentValue(this, dataPath);
            if (parent != null)
            {
                String childName = dataPath.substring(dataPath.lastIndexOf(ModelItem.ITEM_PATH_SEPARATOR) + 1);
                parent.removeValue(childName);
            }
        }
    }
    
    /**
     * Retrieves the modifiable data holder, last parent segment of the given data path
     * Example : call this method with a path like 'my-composite1/my-composite2/my-data' will retrieve the composite 'my-composite2' in the composite 'my-composite1'
     * @param modifiableDataHolder the modifiable data holder
     * @param dataPath the data path
     * @return the parent data holder
     * @throws BadDataPathCardinalityException if the value of a part of the data path is multiple. Only the last part can be multiple
     * @throws BadItemTypeException if the value at the given data path is not a composite
     */
    protected static ModifiableModelLessDataHolder _getParentValue(ModifiableModelLessDataHolder modifiableDataHolder, String dataPath) throws BadDataPathCardinalityException, BadItemTypeException
    {
        return (ModifiableModelLessDataHolder) DefaultModelLessDataHolder._getParentValue(modifiableDataHolder, dataPath);
    }
    
    @Override
    public ModifiableRepositoryData getRepositoryData()
    {
        return _modifiableRepositoryData;
    }
    
    @Override
    public Optional<? extends ModifiableModelLessDataHolder> getParentDataHolder()
    {
        return _modifiableParent;
    }
    
    @Override
    public ModifiableModelLessDataHolder getRootDataHolder()
    {
        return _modifiableRoot;
    }
}
