/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.data.holder.values;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

/**
 * Wrapper for a synchronizable repeater.<br>
 * Contains the repeater values and a mapping between stored entry positions and new ones.
 */
public final class SynchronizableRepeater
{
    private List<Map<String, Object>> _entries;
    private Map<Integer, Integer> _positionsMapping;
    private List<Integer> _replacePositions;
    private Set<Integer> _removedEntries;
    private Mode _mode;
    
    /**
     * The entries write mode.
     */
    public static enum Mode
    {
        /** The entries will replace all existing values */
        REPLACE_ALL,
        /** The entries will replace specific entries */
        REPLACE,
        /** The entries will be appended after existing entries */
        APPEND
    }
    
    private SynchronizableRepeater(List<Map<String, Object>> entries, Mode mode)
    {
        _entries = entries;
        _mode = mode;
    }
    
    private SynchronizableRepeater(List<Map<String, Object>> entries, Map<Integer, Integer> positionsMapping, List<Integer> replacePositions, Set<Integer> removedEntries, Mode mode)
    {
        _entries = entries;
        _positionsMapping = positionsMapping;
        _replacePositions = replacePositions;
        _removedEntries = removedEntries;
        _mode = mode;
    }
    
    /**
     * Replace all values in a repeater, optionally moving or removing entries. Entries not present in the mapping will be removed.
     * @param entries the new repeater values.
     * @param positionsMapping a mapping from stored positions to new ones. No mapping for an existing entry means removal. A null mapping means the identity mapping.
     * @return a {@link SynchronizableRepeater} in replace mode.
     */
    public static SynchronizableRepeater replaceAll(List<Map<String, Object>> entries, Map<Integer, Integer> positionsMapping)
    {
        SynchronizableRepeater repeater = new SynchronizableRepeater(entries, Mode.REPLACE_ALL);
        repeater._positionsMapping = positionsMapping != null ? positionsMapping : IntStream.rangeClosed(1, entries.size()).boxed().collect(Collectors.toMap(Function.identity(), Function.identity()));
        
        return repeater;
    }
    
    /**
     * Replace values in a repeater.
     * @param entries the new repeater values.
     * @param positions the positions of the provided entries in the repeater. A null List means replacing the first entries.
     * @return a {@link SynchronizableRepeater} in replace mode.
     */
    public static SynchronizableRepeater replace(List<Map<String, Object>> entries, List<Integer> positions)
    {
        assert positions == null || positions.size() == entries.size();
        SynchronizableRepeater repeater = new SynchronizableRepeater(entries, Mode.REPLACE);
        repeater._replacePositions = positions != null ? positions : IntStream.rangeClosed(1, entries.size()).boxed().collect(Collectors.toList());
        
        return repeater;
    }
    
    /**
     * Append and/or remove entries in a repeater.
     * @param newEntries the entries to be appended to the repeater.
     * @param removedEntries the entries to be removed from the repeater.
     * @return a {@link SynchronizableRepeater} in append mode.
     */
    public static SynchronizableRepeater appendOrRemove(List<Map<String, Object>> newEntries, Set<Integer> removedEntries)
    {
        SynchronizableRepeater repeater = new SynchronizableRepeater(newEntries, Mode.APPEND);
        repeater._removedEntries = removedEntries;
        
        return repeater;
    }
    
    /**
     * Copy the provided repeater, optionally replacing entries with given ones.
     * @param repeater the SynchronizableRepeater to copy
     * @param newEntries the new entries values
     * @return a new SynchronizableRepeater
     */
    public static SynchronizableRepeater copy(SynchronizableRepeater repeater, List<Map<String, Object>> newEntries)
    {
        return new SynchronizableRepeater(newEntries != null ? newEntries : repeater.getEntries(), repeater.getPositionsMapping(), repeater.getReplacePositions(), repeater.getRemovedEntries(), repeater.getMode());
    }
    
    /**
     * Returns the mapping from stored positions to new ones.
     * @return the mapping from stored positions to new ones.
     */
    public Map<Integer, Integer> getPositionsMapping()
    {
        return _positionsMapping;
    }
    
    /**
     * Returns the positions of replacing entries.
     * @return the positions of replacing entries.
     */
    public List<Integer> getReplacePositions()
    {
        return _replacePositions;
    }
    
    /**
     * Returns the new repeater values.
     * @return the new repeater values.
     */
    public List<Map<String, Object>> getEntries()
    {
        return _entries;
    }
    
    /**
     * Returns the entries to be removed. <br>
     * Only relevant for append mode.<br>
     * For replace mode, removed entries are carried trough the positions mapping.
     * @return the new repeater values.
     */
    public Set<Integer> getRemovedEntries()
    {
        return _removedEntries;
    }
    
    /**
     * Retrieves the previous position of the given one from the repeater mapping
     * @param newPosition the new entry position
     * @return the previous entry position
     */
    public Optional<Integer> getPreviousPosition(int newPosition)
    {
        Map<Integer, Integer> mapping = getPositionsMapping();
        if (mapping != null)
        {
            for (Map.Entry<Integer, Integer> mappingEntry : mapping.entrySet())
            {
                if (mappingEntry.getValue().equals(newPosition))
                {
                    return Optional.of(mappingEntry.getKey());
                }
            }
        }
        
        return Optional.empty();
    }
    
    /**
     * Returns the write mode for provided entries.
     * @return the write mode
     */
    public Mode getMode()
    {
        return _mode;
    }
}
