/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.data.holder.values;

import java.util.Optional;

import org.ametys.plugins.repository.data.external.ExternalizableDataProvider.ExternalizableDataStatus;

/**
 * Wrapper for a synchronizable value
 * Contains the value itself and indicates if it is the external value
 */
public class SynchronizableValue
{
    private Object _localValue;
    private Object _externalValue;
    private ExternalizableDataStatus _externalizableDataStatus;
    private Mode _mode = Mode.REPLACE;
    
    /**
     * The value write mode.
     */
    public static enum Mode
    {
        /** The value will replace existing values */
        REPLACE,
        /** The value will be appended to existing values */
        APPEND,
        /** The value will be removed if found among existing values */
        REMOVE
    }
    
    /**
     * Constructor.
     * @param localValue the actual local value.
     */
    public SynchronizableValue(Object localValue)
    {
        this(localValue, ExternalizableDataStatus.LOCAL);
    }

    /**
     * Constructor
     * @param value the actual value
     * @param valueStatus the status of the given value (not the status of the object itself)
     */
    public SynchronizableValue(Object value, ExternalizableDataStatus valueStatus)
    {
        if (ExternalizableDataStatus.EXTERNAL.equals(valueStatus))
        {
            _externalValue = value;
            _localValue = new UntouchedValue();
        }
        else
        {
            _localValue = value;
            _externalValue = new UntouchedValue();
        }
    }
    
    /**
     * Retrieves the synchronizable local value
     * @return the synchronizable local value
     */
    public Object getLocalValue()
    {
        return _localValue;
    }
    
    /**
     * Sets the synchronizable local value
     * @param localValue the synchronizable local value to set
     */
    public void setLocalValue(Object localValue)
    {
        _localValue = localValue;
    }
    
    /**
     * Retrieves the external value
     * @return the external value
     */
    public Object getExternalValue()
    {
        return _externalValue;
    }
    
    /**
     * Sets the external value
     * @param externalValue the external value to set
     */
    public void setExternalValue(Object externalValue)
    {
        _externalValue = externalValue;
    }
    
    /**
     * Retrieves the value of the given status
     * @param status the status
     * @return the value of the given status
     */
    public Object getValue(Optional<ExternalizableDataStatus> status)
    {
        if (status.isPresent() && ExternalizableDataStatus.EXTERNAL.equals(status.get()))
        {
            return getExternalValue();
        }
        else
        {
            return getLocalValue();
        }
    }
    
    /**
     * Retrieves the externalizable status of the value
     * If <code>null</code>, the status of the value won't be updated
     * @return the externalizable status of the value
     */
    public ExternalizableDataStatus getExternalizableStatus()
    {
        return _externalizableDataStatus;
    }
    
    /**
     * Sets the externalizable status of the value
     * If not set, the status of the value won't be updated
     * @param externalizableDataStatus the status to set
     */
    public void setExternalizableStatus(ExternalizableDataStatus externalizableDataStatus)
    {
        _externalizableDataStatus = externalizableDataStatus;
    }
    
    /**
     * Retrieves the write mode for the value
     * @return the write mode for the value
     */
    public Mode getMode()
    {
        return _mode;
    }
    
    /**
     * Set the write mode for the value
     * @param mode the write mode
     */
    public void setMode(Mode mode)
    {
        _mode = mode;
    }
}
