/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.data.holder.values;

import java.util.HashMap;
import java.util.Map;

import org.ametys.plugins.repository.data.external.ExternalizableDataProvider.ExternalizableDataStatus;
import org.ametys.runtime.model.exception.BadItemTypeException;

/**
 * Object that gives some context for values synchronization
 */
public class SynchronizationContext
{
    private boolean _useDefaultFromModel;
    private ExternalizableDataStatus _externalizableDataStatus = ExternalizableDataStatus.LOCAL;
    private Map<String, Object> _externalizableDataContext = new HashMap<>();
    private boolean _forceStatusIfNotPresent = true;
    private boolean _ignoreIncompatibleValues;

    /**
     * Creates a new instance of a {@link SynchronizationContext}
     */
    protected SynchronizationContext()
    {
        // Empty constructor
    }
    
    /**
     * Creates a new instance of a {@link SynchronizationContext}
     * @return the created instance
     */
    public static SynchronizationContext newInstance()
    {
        return new SynchronizationContext();
    }

    /**
     * Checks if synchronization has to use the default value from the model
     * @return <code>true</code> to use the default value from the model, <code>false</code> otherwise
     */
    public boolean useDefaultFromModel()
    {
        return _useDefaultFromModel;
    }
    
    /**
     * Set to <code>true</code> to use the default value from the model (default to <code>false</code>) 
     * @param useDefaultFromModel <code>true</code> to use the default value from the model, <code>false</code> otherwise
     * @return the current {@link SynchronizationContext}
     */
    public SynchronizationContext withDefaultFromModel(boolean useDefaultFromModel)
    {
        _useDefaultFromModel = useDefaultFromModel;
        return this;
    }

    /**
     * Determines which values (locals or externals) have to be synchronized
     * @return the status of the value to synchronize
     */
    public ExternalizableDataStatus getStatusToSynchronize()
    {
        return _externalizableDataStatus;
    }
    
    /**
     * Sets the status to determine which values (locals or externals) have to be synchronized (default to local)
     * @param externalizableDataStatus the status to determine which values have to be synchronized
     * @return the current {@link SynchronizationContext}
     */
    public SynchronizationContext withStatus(ExternalizableDataStatus externalizableDataStatus)
    {
        _externalizableDataStatus = externalizableDataStatus;
        return this;
    }
    
    /**
     * Retrieves the context {@link Map} that is used to determine if a data is externalizable
     * @return the context {@link Map}
     */
    public Map<String, Object> getExternalizableDataContext()
    {
        return _externalizableDataContext;
    }
    
    /**
     * Add an entry in the context {@link Map} that is used to determine if a data is externalizable
     * @param entryKey the key of the entry
     * @param entryValue the value of the entry
     * @return the current {@link SynchronizationContext}
     */
    public SynchronizationContext withExternalizableDataContextEntry(String entryKey, Object entryValue)
    {
        _externalizableDataContext.put(entryKey, entryValue);
        return this;
    }
    
    /**
     * Determines if the status has to be forced if there is no status yet
     * This case can happen if a data is newly externalizable
     * @return <code>true</code> if the status has to be forced, <code>false</code> otherwise
     */
    public boolean forceStatusIfNotPresent()
    {
        return _forceStatusIfNotPresent;
    }
    
    /**
     * Set to <code>false</code> if the status has not to be forced (default to <code>true</code>)
     * @param forceStatusIfNotPresent <code>true</code> to force the status if it is not yet present, <code>false</code> otherwise
     * @return the current {@link SynchronizationContext}
     */
    public SynchronizationContext withStatusForcedIfNotPresent(boolean forceStatusIfNotPresent)
    {
        _forceStatusIfNotPresent = forceStatusIfNotPresent;
        return this;
    }
    
    /**
     * Determines if values that are incompatible with their data type (defined in model) have to be ignored
     * If <code>false</code> (default behavior), a {@link BadItemTypeException} is thrown if a value to synchronize is incompatible with its type
     * If <code>true</code>, such values are ignored. No exception is thrown
     * @return <code>true</code> if incompatible values have to be ignored, <code>false</code> otherwise
     */
    public boolean ignoreIncompatibleValues()
    {
        return _ignoreIncompatibleValues;
    }
    
    /**
     * Set to <code>true</code> to ignore values that are incompatible with their data type (defined in model) (default to <code>false</code>
     * @param ignoreIncompatibleValues <code>true</code> to ignore incompatible values, <code>false</code> otherwise
     * @return the current {@link SynchronizationContext}
     */
    public SynchronizationContext withIncompatibleValuesIgnored(boolean ignoreIncompatibleValues)
    {
        _ignoreIncompatibleValues = ignoreIncompatibleValues;
        return this;
    }
}
