/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.data.repositorydata;

import java.io.InputStream;
import java.util.Calendar;
import java.util.regex.Pattern;

import org.ametys.plugins.repository.data.UnknownDataException;

/**
 * Interface for modifiable data values management in repository
 */
public interface ModifiableRepositoryData extends RepositoryData
{
    /** The data path pattern to test validity */
    public static final Pattern DATA_NAME_PATTERN = Pattern.compile("[a-z][a-z0-9-_]*", Pattern.CASE_INSENSITIVE);
    
    @Override
    public default ModifiableRepositoryData getRepositoryData(String name)
    {
        return getRepositoryData(name, getDefaultPrefix());
    }
    
    @Override
    public ModifiableRepositoryData getRepositoryData(String name, String prefix);
    
    /**
     * Creates a repository data and stores it in the repository with the given name
     * @param name name of the data
     * @param dataTypeName data type name to use on creation
     * @return the value of the data
     */
    public default ModifiableRepositoryData addRepositoryData(String name, String dataTypeName)
    {
        return addRepositoryData(name, dataTypeName, getDefaultPrefix());
    }
    
    /**
     * Creates a repository data and stores it in the repository with the given name
     * @param name name of the data
     * @param dataTypeName data type name to use on creation
     * @param prefix prefix of the data name, to use instead of the default one.
     * @return the value of the data
     */
    public ModifiableRepositoryData addRepositoryData(String name, String dataTypeName, String prefix);
    
    /**
     * Rename the current repository data
     * @param newName the new name
     */
    public default void rename(String newName)
    {
        rename(newName, getDefaultPrefix());
    }
    
    /**
     * Rename the current repository data
     * @param newName the new name
     * @param prefix prefix of the data name, to use instead of the default one.
     */
    public void rename(String newName, String prefix);
    
    /**
     * Sets the value of the string data stored in the repository with the given name
     * @param name name of the data
     * @param value the value to set
     */
    public default void setValue(String name, String value)
    {
        setValue(name, value, getDefaultPrefix());
    }
    
    /**
     * Sets the value of the string data stored in the repository with the given name
     * @param name name of the data
     * @param value the value to set
     * @param prefix prefix of the data name, to use instead of the default one.
     */
    public void setValue(String name, String value, String prefix);
    
    /**
     * Sets the values of the multiple string data stored in the repository with the given name
     * @param name name of the data
     * @param values the value to set
     */
    public default void setValues(String name, String[] values)
    {
        setValues(name, values, getDefaultPrefix());
    }
    
    /**
     * Sets the values of the multiple string data stored in the repository with the given name
     * @param name name of the data
     * @param values the value to set
     * @param prefix prefix of the data name, to use instead of the default one.
     */
    public void setValues(String name, String[] values, String prefix);
    
    /**
     * Sets the value of the date data stored in the repository with the given name
     * @param name name of the data
     * @param value the value to set
     */
    public default void setValue(String name, Calendar value)
    {
        setValue(name, value, getDefaultPrefix());
    }
    
    /**
     * Sets the value of the date data stored in the repository with the given name
     * @param name name of the data
     * @param value the value to set
     * @param prefix prefix of the data name, to use instead of the default one.
     */
    public void setValue(String name, Calendar value, String prefix);
    
    /**
     * Sets the values of the multiple date data stored in the repository with the given name
     * @param name name of the data
     * @param values the value to set
     */
    public default void setValues(String name, Calendar[] values)
    {
        setValues(name, values, getDefaultPrefix());
    }
    
    /**
     * Sets the values of the multiple date data stored in the repository with the given name
     * @param name name of the data
     * @param values the value to set
     * @param prefix prefix of the data name, to use instead of the default one.
     */
    public void setValues(String name, Calendar[] values, String prefix);
    
    /**
     * Sets the value of the long data stored in the repository with the given name
     * @param name name of the data
     * @param value the value to set
     */
    public default void setValue(String name, Long value)
    {
        setValue(name, value, getDefaultPrefix());
    }
    
    /**
     * Sets the value of the long data stored in the repository with the given name
     * @param name name of the data
     * @param value the value to set
     * @param prefix prefix of the data name, to use instead of the default one.
     */
    public void setValue(String name, Long value, String prefix);
    
    /**
     * Sets the values of the multiple long data stored in the repository with the given name
     * @param name name of the data
     * @param values the value to set
     */
    public default void setValues(String name, Long[] values)
    {
        setValues(name, values, getDefaultPrefix());
    }
    
    /**
     * Sets the values of the multiple long data stored in the repository with the given name
     * @param name name of the data
     * @param values the value to set
     * @param prefix prefix of the data name, to use instead of the default one.
     */
    public void setValues(String name, Long[] values, String prefix);
    
    /**
     * Sets the value of the double data stored in the repository with the given name
     * @param name name of the data
     * @param value the value to set
     */
    public default void setValue(String name, Double value)
    {
        setValue(name, value, getDefaultPrefix());
    }
    
    /**
     * Sets the value of the double data stored in the repository with the given name
     * @param name name of the data
     * @param value the value to set
     * @param prefix prefix of the data name, to use instead of the default one.
     */
    public void setValue(String name, Double value, String prefix);
    
    /**
     * Sets the values of the multiple double data stored in the repository with the given name
     * @param name name of the data
     * @param values the value to set
     */
    public default void setValues(String name, Double[] values)
    {
        setValues(name, values, getDefaultPrefix());
    }
    
    /**
     * Sets the values of the multiple double data stored in the repository with the given name
     * @param name name of the data
     * @param values the value to set
     * @param prefix prefix of the data name, to use instead of the default one.
     */
    public void setValues(String name, Double[] values, String prefix);
    
    /**
     * Sets the value of the boolean data stored in the repository with the given name
     * @param name name of the data
     * @param value the value to set
     */
    public default void setValue(String name, Boolean value)
    {
        setValue(name, value, getDefaultPrefix());
    }
    
    /**
     * Sets the value of the boolean data stored in the repository with the given name
     * @param name name of the data
     * @param value the value to set
     * @param prefix prefix of the data name, to use instead of the default one.
     */
    public void setValue(String name, Boolean value, String prefix);
    
    /**
     * Sets the values of the multiple boolean data stored in the repository with the given name
     * @param name name of the data
     * @param values the value to set
     */
    public default void setValues(String name, Boolean[] values)
    {
        setValues(name, values, getDefaultPrefix());
    }
    
    /**
     * Sets the values of the multiple boolean data stored in the repository with the given name
     * @param name name of the data
     * @param values the value to set
     * @param prefix prefix of the data name, to use instead of the default one.
     */
    public void setValues(String name, Boolean[] values, String prefix);
    
    /**
     * Sets the value of the input stream data stored in the repository with the given name
     * @param name name of the data
     * @param value the value to set
     */
    public default void setValue(String name, InputStream value)
    {
        setValue(name, value, getDefaultPrefix());
    }
    
    /**
     * Sets the value of the input stream data stored in the repository with the given name
     * @param name name of the data
     * @param value the value to set
     * @param prefix prefix of the data name, to use instead of the default one.
     */
    public void setValue(String name, InputStream value, String prefix);
    
    /**
     * Sets the values of the multiple input stream data stored in the repository with the given name
     * @param name name of the data
     * @param values the value to set
     */
    public default void setValues(String name, InputStream[] values)
    {
        setValues(name, values, getDefaultPrefix());
    }
    
    /**
     * Sets the values of the multiple input stream data stored in the repository with the given name
     * @param name name of the data
     * @param values the value to set
     * @param prefix prefix of the data name, to use instead of the default one.
     */
    public void setValues(String name, InputStream[] values, String prefix);
    
    /**
     * Removes the value of the data stored in the repository with the given name
     * @param name name of the data
     * @throws UnknownDataException if there is no data stored with the given name
     */
    public default void removeValue(String name) throws UnknownDataException
    {
        removeValue(name, getDefaultPrefix());
    }
    
    /**
     * Removes the value of the data stored in the repository with the given name
     * @param name name of the data
     * @param prefix prefix of the data name, to use instead of the default one.
     * @throws UnknownDataException if there is no data stored with the given name
     */
    public void removeValue(String name, String prefix) throws UnknownDataException;
}
