/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.data.repositorydata;

import java.io.InputStream;
import java.util.Calendar;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;

import org.ametys.plugins.repository.data.UnknownDataException;

/**
 * Interface for data values management in repository
 */
public interface RepositoryData
{
    /** Constant for type string */
    public static final String STRING_REPOSITORY_DATA_TYPE = "string";
        
    /** Constant for type calendar */
    public static final String CALENDAR_REPOSITORY_DATA_TYPE = "calendar";
    
    /** Constant for type long */
    public static final String LONG_REPOSITORY_DATA_TYPE = "long";
    
    /** Constant for type double */
    public static final String DOUBLE_REPOSITORY_DATA_TYPE = "double";
    
    /** Constant for type boolean */
    public static final String BOOLEAN_REPOSITORY_DATA_TYPE = "boolean";
    
    /** Constant for type stream */
    public static final String STREAM_REPOSITORY_DATA_TYPE = "stream";
    
    /**
     * Retrieves the value of the string data stored in the repository with the given name
     * @param name name of the data
     * @return the value of the data
     */
    public default String getString(String name)
    {
        return getString(name, getDefaultPrefix());
    }
    
    /**
     * Retrieves the value of the string data stored in the repository with the given name
     * @param name name of the data
     * @param prefix prefix of the data name, to use instead of the default one.
     * @return the value of the data
     */
    public String getString(String name, String prefix);
    
    /**
     * Retrieves the values of the multiple string data stored in the repository with the given name
     * @param name name of the data
     * @return the values of the data as a string array
     */
    public default String[] getStrings(String name)
    {
        return getStrings(name, getDefaultPrefix());
    }
    
    /**
     * Retrieves the values of the multiple string data stored in the repository with the given name
     * @param name name of the data
     * @param prefix prefix of the data name, to use instead of the default one.
     * @return the values of the data as a string array
     */
    public String[] getStrings(String name, String prefix);
    
    /**
     * Retrieves the value of the date data stored as calendar in the repository with the given name
     * @param name name of the data
     * @return the value of the data
     */
    public default Calendar getDate(String name)
    {
        return getDate(name, getDefaultPrefix());
    }
    
    /**
     * Retrieves the value of the date data stored as calendar in the repository with the given name
     * @param name name of the data
     * @param prefix prefix of the data name, to use instead of the default one.
     * @return the value of the data
     */
    public Calendar getDate(String name, String prefix);
    
    /**
     * Retrieves the values of the multiple date data as calendar stored in the repository with the given name
     * @param name name of the data
     * @return the values of the data as a date array
     */
    public default Calendar[] getDates(String name)
    {
        return getDates(name, getDefaultPrefix());
    }
    
    /**
     * Retrieves the values of the multiple date data as calendar stored in the repository with the given name
     * @param name name of the data
     * @param prefix prefix of the data name, to use instead of the default one.
     * @return the values of the data as a date array
     */
    public Calendar[] getDates(String name, String prefix);
    
    /**
     * Retrieves the value of the long data stored in the repository with the given name
     * @param name name of the data
     * @return the value of the data
     */
    public default Long getLong(String name)
    {
        return getLong(name, getDefaultPrefix());
    }
    
    /**
     * Retrieves the value of the long data stored in the repository with the given name
     * @param name name of the data
     * @param prefix prefix of the data name, to use instead of the default one.
     * @return the value of the data
     */
    public Long getLong(String name, String prefix);
    
    /**
     * Retrieves the values of the multiple long data stored in the repository with the given name
     * @param name name of the data
     * @return the values of the data as a long array
     */
    public default Long[] getLongs(String name)
    {
        return getLongs(name, getDefaultPrefix());
    }
    
    /**
     * Retrieves the values of the multiple long data stored in the repository with the given name
     * @param name name of the data
     * @param prefix prefix of the data name, to use instead of the default one.
     * @return the values of the data as a long array
     */
    public Long[] getLongs(String name, String prefix);
    
    /**
     * Retrieves the value of the double data stored in the repository with the given name
     * @param name name of the data
     * @return the value of the data
     */
    public default Double getDouble(String name)
    {
        return getDouble(name, getDefaultPrefix());
    }
    
    /**
     * Retrieves the value of the double data stored in the repository with the given name
     * @param name name of the data
     * @param prefix prefix of the data name, to use instead of the default one.
     * @return the value of the data
     */
    public Double getDouble(String name, String prefix);
    
    /**
     * Retrieves the values of the multiple double data stored in the repository with the given name
     * @param name name of the data
     * @return the values of the data as a double array
     */
    public default Double[] getDoubles(String name)
    {
        return getDoubles(name, getDefaultPrefix());
    }
    
    /**
     * Retrieves the values of the multiple double data stored in the repository with the given name
     * @param name name of the data
     * @param prefix prefix of the data name, to use instead of the default one.
     * @return the values of the data as a double array
     */
    public Double[] getDoubles(String name, String prefix);
    
    /**
     * Retrieves the value of the boolean data stored in the repository with the given name
     * @param name name of the data
     * @return the value of the data
     */
    public default Boolean getBoolean(String name)
    {
        return getBoolean(name, getDefaultPrefix());
    }
    
    /**
     * Retrieves the value of the boolean data stored in the repository with the given name
     * @param name name of the data
     * @param prefix prefix of the data name, to use instead of the default one.
     * @return the value of the data
     */
    public Boolean getBoolean(String name, String prefix);
    
    /**
     * Retrieves the values of the multiple boolean data stored in the repository with the given name
     * @param name name of the data
     * @return the values of the data as a boolean array
     */
    public default Boolean[] getBooleans(String name)
    {
        return getBooleans(name, getDefaultPrefix());
    }
    
    /**
     * Retrieves the values of the multiple boolean data stored in the repository with the given name
     * @param name name of the data
     * @param prefix prefix of the data name, to use instead of the default one.
     * @return the values of the data as a boolean array
     */
    public Boolean[] getBooleans(String name, String prefix);
    
    /**
     * Retrieves the value of the repository data stored in the repository with the given name
     * @param name name of the data
     * @return the value of the data
     */
    public default RepositoryData getRepositoryData(String name)
    {
        return getRepositoryData(name, getDefaultPrefix());
    }
    
    /**
     * Retrieves the value of the repository data stored in the repository with the given name
     * @param name name of the data
     * @param prefix prefix of the data name, to use instead of the default one.
     * @return the value of the data
     */
    public RepositoryData getRepositoryData(String name, String prefix);
    
    /**
     * Retrieves the values of all the repository data stored in the repository with the given name
     * @param name name of the data
     * @return the values of the data
     */
    public default RepositoryData[] getAllRepositoryData(String name)
    {
        return getAllRepositoryData(name, getDefaultPrefix());
    }
    
    /**
     * Retrieves the values of all the repository data stored in the repository with the given name
     * @param name name of the data
     * @param prefix prefix of the data name, to use instead of the default one.
     * @return the values of the data
     */
    public RepositoryData[] getAllRepositoryData(String name, String prefix);
    
    /**
     * Retrieves the value of the stream data stored in the repository with the given name
     * @param name name of the data
     * @return the value of the data
     */
    public default InputStream getStream(String name)
    {
        return getStream(name, getDefaultPrefix());
    }
    
    /**
     * Retrieves the value of the stream data stored in the repository with the given name
     * @param name name of the data
     * @param prefix prefix of the data name, to use instead of the default one.
     * @return the value of the data
     */
    public InputStream getStream(String name, String prefix);
    
    /**
     * Retrieves the length of the value of the stream data stored in the repository with the given name
     * @param name name of the data
     * @return the length of the value of the data
     */
    public default Long getStreamLength(String name)
    {
        return getStreamLength(name, getDefaultPrefix());
    }
    
    /**
     * Retrieves the length of the value of the stream data stored in the repository with the given name
     * @param name name of the data
     * @param prefix prefix of the data name, to use instead of the default one.
     * @return the length of the value of the data
     */
    public Long getStreamLength(String name, String prefix);
    
    /**
     * Retrieves the values of the multiple stream data stored in the repository with the given name
     * @param name name of the data
     * @return the values of the data as an input stream array
     */
    public default InputStream[] getStreams(String name)
    {
        return getStreams(name, getDefaultPrefix());
    }
    
    /**
     * Retrieves the values of the multiple stream data stored in the repository with the given name
     * @param name name of the data
     * @param prefix prefix of the data name, to use instead of the default one.
     * @return the values of the data as an input stream array
     */
    public InputStream[] getStreams(String name, String prefix);
    
    /**
     * Retrieves the names of all data in this repository data
     * @return the names of all data in this repository data
     */
    public default Set<String> getAllDataNames()
    {
        return getDataNames(StringUtils.EMPTY);
    }
    
    /**
     * Retrieves the names of data in this repository data with the default prefix
     * @return the names of data by this repository data
     */
    public default Set<String> getDataNames()
    {
        return getDataNames(getDefaultPrefix());
    }
    
    /**
     * Retrieves the names of data in this repository data with the given prefix
     * @param prefix prefix of the data names to retrieve. If <code>null</code>, retrieves all the data names of this repository data
     * @return the names of data by this repository data
     */
    public Set<String> getDataNames(String prefix);
    
    /**
     * Retrieves the name of the current repository data, excluding its prefix
     * @return the name of the current repository data
     */
    public String getName();
    
    /**
     * Checks if there is a value for the data stored in the repository with the given name
     * @param name name of the data
     * @return true if there is value for the data, false otherwise
     */
    public default boolean hasValue(String name)
    {
        return hasValue(name, getDefaultPrefix());
    }
    
    /**
     * Checks if there is a value for the data stored in the repository with the given name
     * @param name name of the data
     * @param prefix prefix of the data name, to use instead of the default one.
     * @return true if there is value for the data, false otherwise
     */
    public boolean hasValue(String name, String prefix);
    
    /**
     * Retrieves the type of the value for the data stored in the repository with the given name
     * @param name name of the data
     * @return the value's type of the data
     * @throws UnknownDataException if there is no data stored with the given name
     */
    public default String getType(String name) throws UnknownDataException
    {
        return getType(name, getDefaultPrefix());
    }
    
    /**
     * Retrieves the type of the value for the data stored in the repository with the given name
     * @param name name of the data
     * @param prefix prefix of the data name, to use instead of the default one.
     * @return the value's type of the data
     * @throws UnknownDataException if there is no data stored with the given name
     */
    public String getType(String name, String prefix) throws UnknownDataException;
    
    /**
     * Checks if the value for the data stored in the repository with the given name is multiple
     * @param name name of the data
     * @return true if the value for the data is multiple, false otherwise
     * @throws UnknownDataException if there is no data stored with the given name
     */
    public default boolean isMultiple(String name) throws UnknownDataException
    {
        return isMultiple(name, getDefaultPrefix());
    }
    
    /**
     * Checks if the value for the data stored in the repository with the given name is multiple
     * @param name name of the data
     * @param prefix prefix of the data name, to use instead of the default one.
     * @return true if the value for the data is multiple, false otherwise
     * @throws UnknownDataException if there is no data stored with the given name
     */
    public boolean isMultiple(String name, String prefix) throws UnknownDataException;
    
    /**
     * Retrieves the default prefix
     * @return the default prefix
     */
    public String getDefaultPrefix();
}
