/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.data.repositorydata.impl;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.jcr.Binary;
import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.PathNotFoundException;
import javax.jcr.Property;
import javax.jcr.PropertyIterator;
import javax.jcr.PropertyType;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.Value;

import org.apache.commons.lang3.StringUtils;

import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.data.UnknownDataException;
import org.ametys.plugins.repository.data.repositorydata.ModifiableRepositoryData;
import org.ametys.plugins.repository.data.repositorydata.RepositoryData;
import org.ametys.plugins.repository.jcr.NodeHelper;
import org.ametys.plugins.repository.jcr.NodeTypeHelper;

/**
 * Class for data values management in JCR repository
 */
public class JCRRepositoryData implements ModifiableRepositoryData
{
    private static final Map<Integer, String> __DATA_TYPES;
    
    static
    {
        __DATA_TYPES = new LinkedHashMap<>(6);
        __DATA_TYPES.put(PropertyType.STRING, RepositoryData.STRING_REPOSITORY_DATA_TYPE);
        __DATA_TYPES.put(PropertyType.LONG, RepositoryData.LONG_REPOSITORY_DATA_TYPE);
        __DATA_TYPES.put(PropertyType.DOUBLE, RepositoryData.DOUBLE_REPOSITORY_DATA_TYPE);
        __DATA_TYPES.put(PropertyType.BOOLEAN, RepositoryData.BOOLEAN_REPOSITORY_DATA_TYPE);
        __DATA_TYPES.put(PropertyType.DATE, RepositoryData.CALENDAR_REPOSITORY_DATA_TYPE);
        __DATA_TYPES.put(PropertyType.BINARY, RepositoryData.STREAM_REPOSITORY_DATA_TYPE);
    }
    
    private Node _node;
    private String _defaultPrefix;
    
    /**
     * Creates a JCR repository data from the given node
     * @param node the Node supporting this repository data
     */
    public JCRRepositoryData(Node node)
    {
        this(node, RepositoryConstants.NAMESPACE_PREFIX);
    }
    
    /**
     * Creates a JCR repository data from the given node
     * @param node the Node supporting this repository data
     * @param defaultPrefix prefix to use for properties and child nodes
     */
    public JCRRepositoryData(Node node, String defaultPrefix)
    {
        _node = node;
        _defaultPrefix = defaultPrefix;
    }

    public String getString(String name, String prefix)
    {
        try
        {
            Value value = _getValue(name, prefix);
            return value.getString();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get the value of string data '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.", e);
        }
    }
    
    public String[] getStrings(String name, String prefix)
    {
        try
        {
            Value[] values = _getValues(name, prefix);
            String[] results = new String[values.length];
            
            for (int i = 0; i < values.length; i++)
            {
                Value value = values[i];
                results[i] = value.getString();
            }
          
            return results;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get the values of multiple string data '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.", e);
        }
    }
    
    public Calendar getDate(String name, String prefix)
    {
        try
        {
            Value value = _getValue(name, prefix);
            return value.getDate();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get the value of date data '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.", e);
        }
    }

    public Calendar[] getDates(String name, String prefix)
    {
        try
        {
            Value[] values = _getValues(name, prefix);
            Calendar[] results = new Calendar[values.length];

            for (int i = 0; i < values.length; i++)
            {
                Value value = values[i];
                results[i] = value.getDate();
            }
          
            return results;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get the values of multiple date data '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.", e);
        }
    }
    
    public Long getLong(String name, String prefix)
    {
        try
        {
            Value value = _getValue(name, prefix);
            return value.getLong();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get the value on long data '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.", e);
        }
    }
    
    public Long[] getLongs(String name, String prefix)
    {
        try
        {
            Value[] values = _getValues(name, prefix);
            Long[] results = new Long[values.length];
            
            for (int i = 0; i < values.length; i++)
            {
                Value value = values[i];
                results[i] = value.getLong();
            }
          
            return results;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get the values of multiple long data '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.", e);
        }
    }
    
    public Double getDouble(String name, String prefix)
    {
        try
        {
            Value value = _getValue(name, prefix);
            return value.getDouble();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get the value of double data '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.", e);
        }
    }
    
    public Double[] getDoubles(String name, String prefix)
    {
        try
        {
            Value[] values = _getValues(name, prefix);
            Double[] results = new Double[values.length];
            
            for (int i = 0; i < values.length; i++)
            {
                Value value = values[i];
                results[i] = value.getDouble();
            }
          
            return results;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get the values of multiple double data '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.", e);
        }
    }
    
    public Boolean getBoolean(String name, String prefix)
    {
        try
        {
            Value value = _getValue(name, prefix);
            return value.getBoolean();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get the value of boolean data '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.", e);
        }
    }
    
    public Boolean[] getBooleans(String name, String prefix)
    {
        try
        {
            Value[] values = _getValues(name, prefix);
            Boolean[] results = new Boolean[values.length];
            
            for (int i = 0; i < values.length; i++)
            {
                Value value = values[i];
                results[i] = value.getBoolean();
            }
          
            return results;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get the values of multiple boolean data '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.", e);
        }
    }
    
    public InputStream[] getStreams(String name, String prefix)
    {
        try
        {
            Value[] values = _getValues(name, prefix);
            InputStream[] results = new InputStream[values.length];
            
            for (int i = 0; i < values.length; i++)
            {
                Value value = values[i];
                results[i] = value.getBinary().getStream();
            }
          
            return results;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get the values of multiple stream data '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.", e);
        }
    }
    
    public InputStream getStream(String name, String prefix)
    {
        try
        {
            Value value = _getValue(name, prefix);
            return value.getBinary().getStream();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get the value of stream data '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.", e);
        }
    }
    
    public Long getStreamLength(String name, String prefix)
    {
        try
        {
            return _node.getProperty(_getFullName(name, prefix)).getLength();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get the length of the value of stream data '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.", e);
        }
    }
    
    private Value _getValue(String name, String prefix)
    {
        try
        {
            Property property = _node.getProperty(_getFullName(name, prefix));

            if (property.getDefinition().isMultiple())
            {
                Value[] values = property.getValues();

                if (values.length > 0)
                {
                    return values[0];
                }
                else
                {
                    throw new AmetysRepositoryException("Cannot get the value of data '" + _getFullName(name, prefix) + "' from an empty array in repository data at path '" + this + "'.");
                }
            }

            return property.getValue();
        }
        catch (PathNotFoundException e)
        {
            throw new UnknownDataException("Unknown data '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.", e);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get the value of data '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.", e);
        }
    }
    
    private Value[] _getValues(String name, String prefix)
    {
        try
        {
            Property property = _node.getProperty(_getFullName(name, prefix));

            if (property.getDefinition().isMultiple())
            {
                return property.getValues();
            }

            Value value = property.getValue();
            return new Value[] {value};
        }
        catch (PathNotFoundException e)
        {
            throw new UnknownDataException("Unknown data '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.", e);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get the multiple value of data '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.", e);
        }
    }
    
    public ModifiableRepositoryData getRepositoryData(String name, String prefix)
    {
        try
        {
            if (_node.hasNode(_getFullName(name, prefix)))
            {
                Node node = _node.getNode(_getFullName(name, prefix));
                return new JCRRepositoryData(node);
            }
            else
            {
                throw new UnknownDataException("Unknown repository data '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.");
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get the repository data '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.", e);
        }
    }
    
    public RepositoryData[] getAllRepositoryData(String name, String prefix)
    {
        try
        {
            if (_node.hasNode(_getFullName(name, prefix)))
            {
                List<RepositoryData> results = new ArrayList<>();
                NodeIterator nodeIterator = _node.getNodes(_getFullName(name, prefix));
                
                while (nodeIterator.hasNext())
                {
                    results.add(new JCRRepositoryData((Node) nodeIterator.next()));
                }
                
                return results.toArray(new RepositoryData[results.size()]);
            }
            else
            {
                throw new UnknownDataException("Unknown repository data '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.");
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get the repository data '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.", e);
        }
    }

    public Set<String> getDataNames(String prefix) throws AmetysRepositoryException
    {
        final String finalPrefix;
        if (prefix == null)
        {
            finalPrefix = "*:";
        }
        else if (StringUtils.isEmpty(prefix))
        {
            finalPrefix = prefix;
        }
        else
        {
            finalPrefix = prefix + ":";
        }
        
        try
        {
            PropertyIterator propertiesIterator = _node.getProperties(finalPrefix + "*");
            NodeIterator itNode = _node.getNodes(finalPrefix + "*");

            Set<String> names = new LinkedHashSet<>();

            while (propertiesIterator.hasNext())
            {
                Property property = propertiesIterator.nextProperty();
                String propertyName = property.getName();
                
                if (!StringUtils.isEmpty(finalPrefix) || !propertyName.contains(":"))
                {
                    names.add(propertyName.substring(finalPrefix.length()));
                }
            }

            while (itNode.hasNext())
            {
                Node node = itNode.nextNode();
                
                // Filtering out AmetysObjects: they're not data (except users).
                if (!NodeTypeHelper.isNodeType(node, AmetysObjectResolver.OBJECT_TYPE) || NodeTypeHelper.isNodeType(node, RepositoryConstants.USER_NODETYPE))
                {
                    names.add(node.getName().substring(finalPrefix.length()));
                }
            }

            return names;
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get the data names of the repository data at path '" + this + "'.", e);
        }
    }
    
    public String getName()
    {
        try
        {
            return getName(_node);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get the name of the repository data at path '" + this + "'.", e);
        }
    }
    
    /**
     * Retrieves the name of the given node, excluding its prefix
     * @param node the node
     * @return the name of the given node
     * @throws RepositoryException if an error occurs
     */
    protected String getName(Node node) throws RepositoryException
    {
        String fullName = node.getName();
        int prefixLength = fullName.indexOf(":") + 1;
        return prefixLength > 0 ? fullName.substring(prefixLength) : fullName;
    }
    
    public boolean hasValue(String name, String prefix)
    {
        try
        {
            return _node.hasProperty(_getFullName(name, prefix)) || _node.hasNode(_getFullName(name, prefix));
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to determine if there is a value for data '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.", e);
        }
    }
    
    public String getType(String name, String prefix) throws UnknownDataException
    {
        try
        {
            String dataName = _getFullName(name, prefix);
            
            if (_node.hasProperty(dataName))
            {
                Property property = _node.getProperty(dataName);
                int jcrType = property.getType();
                
                String dataType = __DATA_TYPES.get(jcrType);
                if (dataType == null)
                {
                    throw new AmetysRepositoryException("Unable to get the type of data '" + dataName + "' in repository data at path '" + this + "'.");
                }
                
                return dataType;
            }
            
            if (_node.hasNode(dataName))
            {
                return NodeTypeHelper.getNodeTypeName(_node.getNode(dataName));
            }
            
            throw new UnknownDataException("No data found for '" + dataName + "' in repository data at path '" + this + "'.");
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get the type of data '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.", e);
        }
    }

    public boolean isMultiple(String name, String prefix) throws UnknownDataException
    {
        try
        {
            String dataName = _getFullName(name, prefix);
            
            if (_node.hasProperty(dataName))
            {
                Property property = _node.getProperty(dataName);
    
                if (property.getDefinition().isMultiple())
                {
                    return true;
                }
    
                return false;
            }
            
            if (_node.hasNode(dataName))
            {
                String nodeTypeName = NodeTypeHelper.getNodeTypeName(_node.getNode(dataName));
                return RepositoryConstants.MULTIPLE_ITEM_NODETYPE.equals(nodeTypeName);
            }
            
            throw new UnknownDataException("No data found for '" + dataName + "' in repository data at path '" + this + "'.");
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to determine if data '" + _getFullName(name, prefix) + "' is multiple in repository data at path '" + this + "'.", e);
        }
    }
    
    public void rename(String newName, String prefix)
    {
        NodeHelper.rename(_node, prefix + ":" + newName);
    }
    
    public void setValue(String name, String value, String prefix)
    {
        if (value == null)
        {
            throw new NullPointerException("Unable to set a 'null' data for '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.");
        }

        _checkDataName(name, prefix);

        try
        {
            _node.setProperty(_getFullName(name, prefix), value);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set the value '" + value + "' for data '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.", e);
        }
    }
    
    public void setValues(String name, String[] values, String prefix)
    {
        if (values == null)
        {
            throw new NullPointerException("Unable to set a 'null' data for '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.");
        }

        _checkDataName(name, prefix);

        try
        {
            _node.setProperty(_getFullName(name, prefix), values);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set the values '" + values + "' for data '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.", e);
        }
    }
    
    public void setValue(String name, Calendar value, String prefix)
    {
        if (value == null)
        {
            throw new NullPointerException("Unable to set a 'null' data for '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.");
        }

        _checkDataName(name, prefix);

        try
        {
            _node.setProperty(_getFullName(name, prefix), value);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set the value '" + value + "' for data '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.", e);
        }
    }
    
    public void setValues(String name, Calendar[] values, String prefix)
    {
        if (values == null)
        {
            throw new NullPointerException("Unable to set a 'null' data for '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.");
        }

        _checkDataName(name, prefix);

        try
        {
            Value[] jcrValues = new Value[values.length];
            for (int i = 0; i < values.length; i++)
            {
                Calendar value = values[i];
                jcrValues[i] = _node.getSession().getValueFactory().createValue(value);
            }
            
            _node.setProperty(_getFullName(name, prefix), jcrValues, PropertyType.DATE);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set the values '" + values + "' for data '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.", e);
        }
    }
    
    public void setValue(String name, Long value, String prefix)
    {
        if (value == null)
        {
            throw new NullPointerException("Unable to set a 'null' data for '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.");
        }

        _checkDataName(name, prefix);

        try
        {
            _node.setProperty(_getFullName(name, prefix), value);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set the value '" + value + "' for data '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.", e);
        }
    }
    
    public void setValues(String name, Long[] values, String prefix)
    {
        if (values == null)
        {
            throw new NullPointerException("Unable to set a 'null' data for '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.");
        }

        _checkDataName(name, prefix);

        try
        {
            Value[] jcrValues = new Value[values.length];
            for (int i = 0; i < values.length; i++)
            {
                long value = values[i];
                jcrValues[i] = _node.getSession().getValueFactory().createValue(value);
            }
            
            _node.setProperty(_getFullName(name, prefix), jcrValues, PropertyType.LONG);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set the values '" + values + "' for data '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.", e);
        }
    }
    
    public void setValue(String name, Double value, String prefix)
    {
        if (value == null)
        {
            throw new NullPointerException("Unable to set a 'null' data for '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.");
        }

        _checkDataName(name, prefix);

        try
        {
            _node.setProperty(_getFullName(name, prefix), value);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set the value '" + value + "' for data '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.", e);
        }
    }
    
    public void setValues(String name, Double[] values, String prefix)
    {
        if (values == null)
        {
            throw new NullPointerException("Unable to set a 'null' data for '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.");
        }

        _checkDataName(name, prefix);

        try
        {
            Value[] jcrValues = new Value[values.length];
            for (int i = 0; i < values.length; i++)
            {
                double value = values[i];
                jcrValues[i] = _node.getSession().getValueFactory().createValue(value);
            }
            
            _node.setProperty(_getFullName(name, prefix), jcrValues, PropertyType.DOUBLE);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set the values '" + values + "' for data '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.", e);
        }
    }
    
    public void setValue(String name, Boolean value, String prefix)
    {
        if (value == null)
        {
            throw new NullPointerException("Unable to set a 'null' data for '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.");
        }

        _checkDataName(name, prefix);

        try
        {
            _node.setProperty(_getFullName(name, prefix), value);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set the value '" + value + "' for data '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.", e);
        }
    }
    
    public void setValues(String name, Boolean[] values, String prefix)
    {
        if (values == null)
        {
            throw new NullPointerException("Unable to set a 'null' data for '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.");
        }

        _checkDataName(name, prefix);

        try
        {
            Value[] jcrValues = new Value[values.length];
            for (int i = 0; i < values.length; i++)
            {
                boolean value = values[i];
                jcrValues[i] = _node.getSession().getValueFactory().createValue(value);
            }
            
            _node.setProperty(_getFullName(name, prefix), jcrValues, PropertyType.BOOLEAN);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set the values '" + values + "' for data '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.", e);
        }
    }
    
    public void setValue(String name, InputStream value, String prefix)
    {
        if (value == null)
        {
            throw new NullPointerException("Unable to set a 'null' data for '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.");
        }

        _checkDataName(name, prefix);

        try
        {
            Binary binary = _node.getSession().getValueFactory().createBinary(value);
            _node.setProperty(_getFullName(name, prefix), binary);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set the value '" + value + "' for data '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.", e);
        }
    }
    
    public void setValues(String name, InputStream[] values, String prefix)
    {
        if (values == null)
        {
            throw new NullPointerException("Unable to set a 'null' data for '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.");
        }

        _checkDataName(name, prefix);

        try
        {
            Value[] jcrValues = new Value[values.length];
            for (int i = 0; i < values.length; i++)
            {
                InputStream value = values[i];
                Binary binary = _node.getSession().getValueFactory().createBinary(value);
                jcrValues[i] = _node.getSession().getValueFactory().createValue(binary);
            }
            
            _node.setProperty(_getFullName(name, prefix), jcrValues, PropertyType.BINARY);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to set the values '" + values + "' for data '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.", e);
        }
    }
    
    public ModifiableRepositoryData addRepositoryData(String name, String dataTypeName, String prefix)
    {
        try
        {
            Node node = _node.addNode(_getFullName(name, prefix), dataTypeName);
            return new JCRRepositoryData(node);
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to create repository data '" + _getFullName(name, prefix) + "' with type '" + dataTypeName + "' in repository data at path '" + this + "'.", e);
        }
    }

    private void _checkDataName(String name, String prefix)
    {
        // Name null
        if (name == null)
        {
            throw new AmetysRepositoryException("Unable to set a value with the invalid data name (null) in repository data at path '" + this + "'.");
        }

        // Valid name
        if (!DATA_NAME_PATTERN.matcher(name).matches())
        {
            throw new AmetysRepositoryException("Unable to set a value with the invalid data name '" + name + "'. Only [a-zA-Z][a-zA-Z0-9-_]* are allowed.");
        }
        
        // Has value
        if (hasValue(name, prefix))
        {
            // Remove existing value to avoid values with same name (node + property)
            removeValue(name, prefix);
        }
    }
    
    public void removeValue(String name, String prefix) throws UnknownDataException
    {
        try
        {
            String dataName = _getFullName(name, prefix);
            
            if (_node.hasProperty(dataName))
            {
                _node.getProperty(dataName).remove();
            }
            else if (_node.hasNode(dataName))
            {
                // Remove all existing nodes
                NodeIterator it = _node.getNodes(dataName);
                while (it.hasNext())
                {
                    Node next = it.nextNode();
                    next.remove();
                }
            }
            else
            {
                throw new UnknownDataException("No data found for '" + dataName + "' in repository data at path '" + this + "'.");
            }
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to remove data '" + _getFullName(name, prefix) + "' in repository data at path '" + this + "'.", e);
        }
    }
    
    private String _getFullName(String name, String prefix)
    {
        return StringUtils.isEmpty(prefix) ? name : prefix + ":" + name;
    }
    
    /**
     * Retrieves the underlying node.
     * @return the underlying node.
     */
    public Node getNode()
    {
        return _node;
    }
    
    /**
     * Retrieves the current session
     * @return he session
     * @throws AmetysRepositoryException if an error occurs
     */
    public Session getSession() throws AmetysRepositoryException
    {
        try
        {
            return _node.getSession();
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("unable to retrieve the session of the repository data at path '" + this + "'.", e);
        }
    }
    
    public String getDefaultPrefix()
    {
        return _defaultPrefix;
    }
    
    @Override
    public String toString()
    {
        try
        {
            return _node.getPath() + " (" + _node.getIdentifier() + ")";
        }
        catch (RepositoryException e)
        {
            throw new AmetysRepositoryException("Unable to get infos about the repository data of node '" + _node + "'.", e);
        }
    }
}
