/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.data.repositorydata.impl;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;

import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.data.UnknownDataException;
import org.ametys.plugins.repository.data.repositorydata.ModifiableRepositoryData;
import org.ametys.plugins.repository.data.repositorydata.RepositoryData;

/**
 * Class for data values management in memory
 */
public class MemoryRepositoryData implements ModifiableRepositoryData
{
    private String _fullName;
    private Map<String, Pair<String, Object>> _data;
    private String _defaultPrefix;
    
    /**
     * Creates a memory repository data with the given name
     * @param name the name of the current data
     */
    public MemoryRepositoryData(String name)
    {
        this(name, RepositoryConstants.NAMESPACE_PREFIX);
    }
    
    /**
     * Creates a memory repository data with the given name
     * @param name the name of the current data
     * @param defaultPrefix prefix to use for properties and child nodes
     */
    public MemoryRepositoryData(String name, String defaultPrefix)
    {
        _fullName = _getFullName(name, defaultPrefix);
        _defaultPrefix = defaultPrefix;
        _data = new HashMap<>();
    }
    
    public String getString(String name, String prefix)
    {
        if (hasValue(name, prefix))
        {
            Pair<String, Object> value = _data.get(_getFullName(name, prefix));
            if (RepositoryData.STRING_REPOSITORY_DATA_TYPE.equals(value.getLeft()) && value.getRight() instanceof String)
            {
                return (String) value.getRight();
            }
            else
            {
                throw new AmetysRepositoryException("Unable to get the value of string data '" + prefix + ":" + name + "' in repository data named '" + _fullName + "'.");
            }
        }
        else
        {
            throw new UnknownDataException("Unknown data '" + prefix + ":" + name + "' in repository data named '" + _fullName + "'.");
        }
    }

    public String[] getStrings(String name, String prefix)
    {
        if (hasValue(name, prefix))
        {
            Pair<String, Object> value = _data.get(_getFullName(name, prefix));
            if (RepositoryData.STRING_REPOSITORY_DATA_TYPE.equals(value.getLeft()) && value.getRight() instanceof String[])
            {
                return (String[]) value.getRight();
            }
            else
            {
                throw new AmetysRepositoryException("Unable to get the value of multiple string data '" + prefix + ":" + name + "' in repository data named '" + _fullName + "'.");
            }
        }
        else
        {
            throw new UnknownDataException("Unknown data '" + prefix + ":" + name + "' in repository data named '" + _fullName + "'.");
        }
    }

    public Calendar getDate(String name, String prefix)
    {
        if (hasValue(name, prefix))
        {
            Pair<String, Object> value = _data.get(_getFullName(name, prefix));
            if (RepositoryData.CALENDAR_REPOSITORY_DATA_TYPE.equals(value.getLeft()) && value.getRight() instanceof Calendar)
            {
                return (Calendar) value.getRight();
            }
            else
            {
                throw new AmetysRepositoryException("Unable to get the value of date data '" + prefix + ":" + name + "' in repository data named '" + _fullName + "'.");
            }
        }
        else
        {
            throw new UnknownDataException("Unknown data '" + prefix + ":" + name + "' in repository data named '" + _fullName + "'.");
        }
    }

    public Calendar[] getDates(String name, String prefix)
    {
        if (hasValue(name, prefix))
        {
            Pair<String, Object> value = _data.get(_getFullName(name, prefix));
            if (RepositoryData.CALENDAR_REPOSITORY_DATA_TYPE.equals(value.getLeft()) && value.getRight() instanceof Calendar[])
            {
                return (Calendar[]) value.getRight();
            }
            else
            {
                throw new AmetysRepositoryException("Unable to get the value of multiple date data '" + prefix + ":" + name + "' in repository data named '" + _fullName + "'.");
            }
        }
        else
        {
            throw new UnknownDataException("Unknown data '" + prefix + ":" + name + "' in repository data named '" + _fullName + "'.");
        }
    }
    
    public Long getLong(String name, String prefix)
    {
        if (hasValue(name, prefix))
        {
            Pair<String, Object> value = _data.get(_getFullName(name, prefix));
            if (RepositoryData.LONG_REPOSITORY_DATA_TYPE.equals(value.getLeft()) && value.getRight() instanceof Long)
            {
                return (Long) value.getRight();
            }
            else
            {
                throw new AmetysRepositoryException("Unable to get the value of long data '" + prefix + ":" + name + "' in repository data named '" + _fullName + "'.");
            }
        }
        else
        {
            throw new UnknownDataException("Unknown data '" + prefix + ":" + name + "' in repository data named '" + _fullName + "'.");
        }
    }

    public Long[] getLongs(String name, String prefix)
    {
        if (hasValue(name, prefix))
        {
            Pair<String, Object> value = _data.get(_getFullName(name, prefix));
            if (RepositoryData.LONG_REPOSITORY_DATA_TYPE.equals(value.getLeft()) && value.getRight() instanceof Long[])
            {
                return (Long[]) value.getRight();
            }
            else
            {
                throw new AmetysRepositoryException("Unable to get the value of multiple long data '" + prefix + ":" + name + "' in repository data named '" + _fullName + "'.");
            }
        }
        else
        {
            throw new UnknownDataException("Unknown data '" + prefix + ":" + name + "' in repository data named '" + _fullName + "'.");
        }
    }

    public Double getDouble(String name, String prefix)
    {
        if (hasValue(name, prefix))
        {
            Pair<String, Object> value = _data.get(_getFullName(name, prefix));
            if (RepositoryData.DOUBLE_REPOSITORY_DATA_TYPE.equals(value.getLeft()) && value.getRight() instanceof Double)
            {
                return (Double) value.getRight();
            }
            else
            {
                throw new AmetysRepositoryException("Unable to get the value of double data '" + prefix + ":" + name + "' in repository data named '" + _fullName + "'.");
            }
        }
        else
        {
            throw new UnknownDataException("Unknown data '" + prefix + ":" + name + "' in repository data named '" + _fullName + "'.");
        }
    }

    public Double[] getDoubles(String name, String prefix)
    {
        if (hasValue(name, prefix))
        {
            Pair<String, Object> value = _data.get(_getFullName(name, prefix));
            if (RepositoryData.DOUBLE_REPOSITORY_DATA_TYPE.equals(value.getLeft()) && value.getRight() instanceof Double[])
            {
                return (Double[]) value.getRight();
            }
            else
            {
                throw new AmetysRepositoryException("Unable to get the value of multiple double data '" + prefix + ":" + name + "' in repository data named '" + _fullName + "'.");
            }
        }
        else
        {
            throw new UnknownDataException("Unknown data '" + prefix + ":" + name + "' in repository data named '" + _fullName + "'.");
        }
    }

    public Boolean getBoolean(String name, String prefix)
    {
        if (hasValue(name, prefix))
        {
            Pair<String, Object> value = _data.get(_getFullName(name, prefix));
            if (RepositoryData.BOOLEAN_REPOSITORY_DATA_TYPE.equals(value.getLeft()) && value.getRight() instanceof Boolean)
            {
                return (Boolean) value.getRight();
            }
            else
            {
                throw new AmetysRepositoryException("Unable to get the value of boolean data '" + prefix + ":" + name + "' in repository data named '" + _fullName + "'.");
            }
        }
        else
        {
            throw new UnknownDataException("Unknown data '" + prefix + ":" + name + "' in repository data named '" + _fullName + "'.");
        }
    }

    public Boolean[] getBooleans(String name, String prefix)
    {
        if (hasValue(name, prefix))
        {
            Pair<String, Object> value = _data.get(_getFullName(name, prefix));
            if (RepositoryData.BOOLEAN_REPOSITORY_DATA_TYPE.equals(value.getLeft()) && value.getRight() instanceof Boolean[])
            {
                return (Boolean[]) value.getRight();
            }
            else
            {
                throw new AmetysRepositoryException("Unable to get the value of multiple boolean data '" + prefix + ":" + name + "' in repository data named '" + _fullName + "'.");
            }
        }
        else
        {
            throw new UnknownDataException("Unknown data '" + prefix + ":" + name + "' in repository data named '" + _fullName + "'.");
        }
    }

    public ModifiableRepositoryData getRepositoryData(String name, String prefix)
    {
        if (hasValue(name, prefix))
        {
            Pair<String, Object> value = _data.get(_getFullName(name, prefix));
            if (value.getRight() instanceof ModifiableRepositoryData[])
            {
                ModifiableRepositoryData[] allRepositoryData = (ModifiableRepositoryData[]) value.getRight();
                if (allRepositoryData.length >= 1)
                {
                    return ((ModifiableRepositoryData[]) value.getRight())[0];
                }
                else
                {
                    throw new UnknownDataException("Unknown data '" + prefix + ":" + name + "' in repository data named '" + _fullName + "'.");
                }
            }
            else
            {
                throw new AmetysRepositoryException("Unable to get the value of repository data '" + prefix + ":" + name + "' in repository data named '" + _fullName + "'.");
            }
        }
        else
        {
            throw new UnknownDataException("Unknown data '" + prefix + ":" + name + "' in repository data named '" + _fullName + "'.");
        }
    }
    
    public RepositoryData[] getAllRepositoryData(String name, String prefix)
    {
        if (hasValue(name, prefix))
        {
            Pair<String, Object> value = _data.get(_getFullName(name, prefix));
            if (value.getRight() instanceof ModifiableRepositoryData[])
            {
                return (ModifiableRepositoryData[]) value.getRight();
            }
            else
            {
                throw new AmetysRepositoryException("Unable to get the value of multiple repository data '" + prefix + ":" + name + "' in repository data named '" + _fullName + "'.");
            }
        }
        else
        {
            throw new UnknownDataException("Unknown data '" + prefix + ":" + name + "' in repository data named '" + _fullName + "'.");
        }
    }

    public Set<String> getDataNames(String prefix)
    {
        final String finalPrefix;
        if (prefix == null)
        {
            finalPrefix = "*:";
        }
        else if (StringUtils.isEmpty(prefix))
        {
            finalPrefix = prefix;
        }
        else
        {
            finalPrefix = prefix + ":";
        }
        
        return _data.keySet().stream()
                             .filter(name -> name.startsWith(finalPrefix))
                             .map(name -> name.substring(finalPrefix.length()))
                             .collect(Collectors.toSet());
    }
    
    public String getName()
    {
        int prefixLength = _fullName.indexOf(":") + 1;
        return prefixLength > 0 ? _fullName.substring(prefixLength) : _fullName;
    }

    public boolean hasValue(String name, String prefix)
    {
        return _data.containsKey(_getFullName(name, prefix));
    }

    public String getType(String name, String prefix) throws UnknownDataException
    {
        Pair<String, Object> value = _data.get(_getFullName(name, prefix));
        if (value != null)
        {
            return value.getLeft();
        }
        else
        {
            throw new UnknownDataException("No data found for '" + prefix + ":" + name + "' in repository data named '" + _fullName + "'.");
        }
    }

    public boolean isMultiple(String name, String prefix) throws UnknownDataException
    {
        if (hasValue(name, prefix))
        {
            Pair<String, Object> value = _data.get(_getFullName(name, prefix));
            if (RepositoryConstants.MULTIPLE_ITEM_NODETYPE.equals(value.getLeft()))
            {
                return true;
            }
            else if (value.getRight().getClass().isArray())
            {
                if (value.getRight() instanceof ModifiableRepositoryData[])
                {
                    return ((ModifiableRepositoryData[]) value.getRight()).length > 1;
                }
                else
                {
                    return true;
                }
            }
            else
            {
                return false;
            }
        }
        else
        {
            throw new UnknownDataException("No data found for '" + prefix + ":" + name + "' in repository data named '" + _fullName + "'.");
        }
    }

    public String getDefaultPrefix()
    {
        return _defaultPrefix;
    }

    public ModifiableRepositoryData addRepositoryData(String name, String dataTypeName, String prefix)
    {
        ModifiableRepositoryData repositoryData = new MemoryRepositoryData(name, prefix);
        List<ModifiableRepositoryData> allRepositoryData = new ArrayList<>();
        
        // Has value
        if (hasValue(name, prefix))
        {
            Pair<String, Object> value = _data.get(_getFullName(name, prefix));
            allRepositoryData.addAll(Arrays.asList((ModifiableRepositoryData[]) value.getRight()));
        }
        
        allRepositoryData.add(repositoryData);
        ModifiableRepositoryData[] result = allRepositoryData.toArray(new ModifiableRepositoryData[allRepositoryData.size()]);
        _data.put(_getFullName(name, prefix), new ImmutablePair<>(dataTypeName, result));

        return repositoryData;
    }

    public void rename(String newName, String prefix)
    {
        _fullName = _getFullName(newName, prefix);
    }

    public void setValue(String name, String value, String prefix)
    {
        _setValue(name, RepositoryData.STRING_REPOSITORY_DATA_TYPE, value, prefix);
    }

    public void setValues(String name, String[] values, String prefix)
    {
        _setValue(name, RepositoryData.STRING_REPOSITORY_DATA_TYPE, values, prefix);
    }

    public void setValue(String name, Calendar value, String prefix)
    {
        _setValue(name, RepositoryData.CALENDAR_REPOSITORY_DATA_TYPE, value, prefix);
    }

    public void setValues(String name, Calendar[] values, String prefix)
    {
        _setValue(name, RepositoryData.CALENDAR_REPOSITORY_DATA_TYPE, values, prefix);
    }

    public void setValue(String name, Long value, String prefix)
    {
        _setValue(name, RepositoryData.LONG_REPOSITORY_DATA_TYPE, value, prefix);
    }

    public void setValues(String name, Long[] values, String prefix)
    {
        _setValue(name, RepositoryData.LONG_REPOSITORY_DATA_TYPE, values, prefix);
    }

    public void setValue(String name, Double value, String prefix)
    {
        _setValue(name, RepositoryData.DOUBLE_REPOSITORY_DATA_TYPE, value, prefix);
    }

    public void setValues(String name, Double[] values, String prefix)
    {
        _setValue(name, RepositoryData.DOUBLE_REPOSITORY_DATA_TYPE, values, prefix);
    }

    public void setValue(String name, Boolean value, String prefix)
    {
        _setValue(name, RepositoryData.BOOLEAN_REPOSITORY_DATA_TYPE, value, prefix);
    }

    public void setValues(String name, Boolean[] values, String prefix)
    {
        _setValue(name, RepositoryData.BOOLEAN_REPOSITORY_DATA_TYPE, values, prefix);
    }
    
    private void _setValue(String name, String dataTypeName, Object value, String prefix)
    {
        String fullName =  _getFullName(name, prefix);
        
        if (value == null)
        {
            throw new NullPointerException("Unable to set a 'null' data for '" + fullName + "' in repository data named '" + _fullName + "'.");
        }

        _checkDataName(name);
     
        // Has value
        if (hasValue(name, prefix))
        {
            // Remove existing value to avoid values with same name (node + property)
            removeValue(name, prefix);
        }
        
        _data.put(fullName, new ImmutablePair<>(dataTypeName, value));
    }
    
    private void _checkDataName(String name)
    {
        // Name null
        if (name == null)
        {
            throw new AmetysRepositoryException("Unable to set a value with the invalid data name (null) in repository data named '" + _fullName + "'.");
        }

        // Valid name
        if (!DATA_NAME_PATTERN.matcher(name).matches())
        {
            throw new AmetysRepositoryException("Unable to set a value with the invalid data name '" + name + "'. Only [a-zA-Z][a-zA-Z0-9-_]* are allowed.");
        }
    }

    public void removeValue(String name, String prefix) throws UnknownDataException
    {
        if (hasValue(name, prefix))
        {
            _data.remove(_getFullName(name, prefix));
        }
        else
        {
            throw new UnknownDataException("No data found for '" + _getFullName(name, prefix) + "' in repository data named '" + _fullName + "'.");
        }
    }
    
    private String _getFullName(String name, String prefix)
    {
        return StringUtils.isEmpty(prefix) ? name : prefix + ":" + name;
    }

    public InputStream getStream(String name, String prefix)
    {
        throw new UnsupportedOperationException("Streams are not supported by the memory repository data");
    }
    
    public Long getStreamLength(String name, String prefix)
    {
        throw new UnsupportedOperationException("Streams are not supported by the memory repository data");
    }

    public InputStream[] getStreams(String name, String prefix)
    {
        throw new UnsupportedOperationException("Streams are not supported by the memory repository data");
    }
    
    public void setValue(String name, InputStream value, String prefix)
    {
        throw new UnsupportedOperationException("Streams are not supported by the memory repository data");
    }
    
    public void setValues(String name, InputStream[] values, String prefix)
    {
        throw new UnsupportedOperationException("Streams are not supported by the memory repository data");
    }
    
    @Override
    public String toString()
    {
        return _fullName;
    }
}
