/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.data.type;

import java.lang.reflect.Array;

import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.data.UnknownDataException;
import org.ametys.plugins.repository.data.repositorydata.ModifiableRepositoryData;
import org.ametys.plugins.repository.data.repositorydata.RepositoryData;
import org.ametys.runtime.model.exception.BadItemTypeException;

/**
 * Interface for complex types of elements stored in the repository
 * The complex types ares those which need nodes to be stored
 * @param <T> Type of the element value
 */
public interface ComplexRepositoryElementType<T> extends RepositoryElementType<T>
{
    /** Data name for storing the type identifier of an element with multiple values*/
    public static String TYPE_ID_DATA_NAME = "typeId";
    
    @SuppressWarnings("unchecked")
    public default Object read(RepositoryData parentData, String name) throws BadItemTypeException
    {
        if (!parentData.hasValue(name))
        {
            return null;
        }
        
        if (!isCompatible(parentData, name))
        {
            throw new BadItemTypeException("Try to get " + getId() + " value from the non '" + getId() + "' data '" + name + "' on '" + parentData + "'");
        }
        
        if (parentData.isMultiple(name))
        {
            RepositoryData multipleParentData = parentData.getRepositoryData(name);
            
            return multipleParentData.getDataNames()
                                     .stream()
                                     .map(singleDataName -> multipleParentData.getRepositoryData(singleDataName))
                                     .map(singleData -> readSingleValue(singleData))
                                     .toArray(size -> (T[]) Array.newInstance(getManagedClass(), size));
        }
        else
        {
            RepositoryData data = parentData.getRepositoryData(name);
            return !isSingleValueEmpty(data) ? readSingleValue(data) : null;
        }
    }
    
    public default boolean hasNonEmptyValue(RepositoryData parentData, String name) throws BadItemTypeException
    {
        if (!parentData.hasValue(name))
        {
            return false;
        }
        
        if (!isCompatible(parentData, name))
        {
            throw new BadItemTypeException("Try to check " + getId() + " value from the non '" + getId() + "' data '" + name + "' on '" + parentData + "'");
        }
        
        if (parentData.isMultiple(name))
        {
            RepositoryData multipleParentData = parentData.getRepositoryData(name);
            
            return !multipleParentData.getDataNames().isEmpty();
        }
        else
        {
            RepositoryData data = parentData.getRepositoryData(name);
            return !isSingleValueEmpty(data);
        }
    }
    
    /**
     * Checks if the single value is empty
     * @param singleValueData repository data containing the value to check
     * @return <code>true</code> if the value is empty, <code>false</code> otherwise
     */
    public boolean isSingleValueEmpty(RepositoryData singleValueData);
    
    /**
     * Read the single value in the given repository data
     * @param singleValueData repository data containing the value
     * @return the value. Can return null if the given data does not contain the necessary elements
     */
    public T readSingleValue(RepositoryData singleValueData);
    
    @SuppressWarnings("unchecked")
    public default void write(ModifiableRepositoryData parentData, String name, Object value) throws BadItemTypeException
    {
        if (value == null)
        {
            if (parentData.hasValue(name) && parentData.isMultiple(name))
            {
                parentData.removeValue(name);
                ModifiableRepositoryData multipleParentData = parentData.addRepositoryData(name, RepositoryConstants.MULTIPLE_ITEM_NODETYPE);
                // Add the type identifier in a property to know what is the type of the data even if there is no value
                multipleParentData.setValue(TYPE_ID_DATA_NAME, getId(), RepositoryConstants.NAMESPACE_PREFIX_INTERNAL);
            }
            else
            {
                emptySingleValue(parentData, name);
            }
        }
        else if (getManagedClass().isInstance(value))
        {
            if (parentData.hasValue(name) && removeValueBeforeWritingIt())
            {
                parentData.removeValue(name);
            }
            
            writeSingleValue(parentData, name, (T) value);
        }
        else if (getManagedClassArray().isInstance(value))
        {
            ModifiableRepositoryData multipleParentData;
            
            if (parentData.hasValue(name) && !removeValueBeforeWritingIt())
            {
                multipleParentData = parentData.getRepositoryData(name);
            }
            else
            {
                if (parentData.hasValue(name))
                {
                    parentData.removeValue(name);
                }
                
                multipleParentData = parentData.addRepositoryData(name, RepositoryConstants.MULTIPLE_ITEM_NODETYPE);
                // Add the type identifier in a property to know what the type of the data even if there is no value
                multipleParentData.setValue(TYPE_ID_DATA_NAME, getId(), RepositoryConstants.NAMESPACE_PREFIX_INTERNAL);
            }
            
            // Loop on all given values to replace the existing ones
            for (int i = 0; i < ((T[]) value).length; i++)
            {
                T singleValue = ((T[]) value)[i];
                if (singleValue == null)
                {
                    throw new IllegalArgumentException("Try to set a null value into the multiple " + getId() + " data '" + name + "' on '" + parentData + "'");
                }
                else
                {
                    writeSingleValue(multipleParentData, String.valueOf(i + 1), singleValue);
                }
            }
            
            // Remove existing entries from the last one to the last of given values
            for (int i = multipleParentData.getDataNames().size(); i > ((T[]) value).length; i--)
            {
                multipleParentData.removeValue(String.valueOf(i));
            }
        }
        else
        {
            StringBuilder message = new StringBuilder().append("Try to set the non ").append(getId()).append(" value '").append(value);
            message.append("' to the ").append(getId()).append(" data '").append(name).append("' on '").append(parentData).append("'");
            throw new BadItemTypeException(message.toString());
        }
    }
    
    /**
     * Determines if the existing value has to be removed before writing the new one
     * @return <code>true</code> if the previous value has to be removed, <code>false</code> otherwise
     */
    public default boolean removeValueBeforeWritingIt()
    {
        return true;
    }
    
    /**
     * Empties the single value into the given repositoryData
     * This method is called by the {@link #write(ModifiableRepositoryData, String, Object)} method, when the given value is <code>null</code>
     * @param parentData repository where to empty the single value.
     * @param name the name of the element to empty
     */
    public default void emptySingleValue(ModifiableRepositoryData parentData, String name)
    {
        if (removeValueBeforeWritingIt())
        {
            if (parentData.hasValue(name))
            {
                parentData.removeValue(name);
            }
            
            parentData.addRepositoryData(name, getRepositoryDataType());
        }
    }
    
    /**
     * Write the single value into the given repository data
     * This method is called by the {@link #write(ModifiableRepositoryData, String, Object)} method, once for each value if the value is an array
     * @param parentData repository where to store the single value.
     * @param name the name of the element to write
     * @param value the single value to write. Can be null. In this case, an empty data must be created
     */
    public void writeSingleValue(ModifiableRepositoryData parentData, String name, T value);
    
    public default boolean isCompatible(RepositoryData parentData, String name) throws UnknownDataException
    {
        if (parentData.hasValue(name + EMPTY_METADATA_SUFFIX, RepositoryConstants.NAMESPACE_PREFIX_INTERNAL))
        {
            return true;
        }
        
        if (getRepositoryDataType().equals(parentData.getType(name)))
        {
            return true;
        }

        if (RepositoryConstants.MULTIPLE_ITEM_NODETYPE.equals(parentData.getType(name)))
        {
            RepositoryData multipleItemNode = parentData.getRepositoryData(name);
            return getId().equals(multipleItemNode.getString(TYPE_ID_DATA_NAME, RepositoryConstants.NAMESPACE_PREFIX_INTERNAL));
        }
            
        return false;
    }
}
