/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.data.type;

import org.ametys.plugins.repository.data.UnknownDataException;
import org.ametys.plugins.repository.data.repositorydata.ModifiableRepositoryData;
import org.ametys.plugins.repository.data.repositorydata.RepositoryData;
import org.ametys.runtime.model.exception.BadItemTypeException;
import org.ametys.runtime.model.type.ModelItemGroupType;

/**
 * Interface for types of model item groups stored in the repository
 */
public interface RepositoryModelItemGroupType extends RepositoryModelItemType, ModelItemGroupType
{
    /**
     * Read the group in the given repository data
     * @param parentData repository data containing the group
     * @param name the name of the group to read
     * @return the repository data representing the read group
     * @throws BadItemTypeException if the reading value doesn't match this element type
     */
    public default RepositoryData read(RepositoryData parentData, String name) throws BadItemTypeException
    {
        if (!parentData.hasValue(name))
        {
            return null;
        }
        
        if (!isCompatible(parentData, name))
        {
            throw new BadItemTypeException("Try to get " + getId() + " value from the non '" + getId() + "' data '" + name + "' on '" + parentData + "'");
        }
        
        return parentData.getRepositoryData(name);
    }
    
    public default boolean hasNonEmptyValue(RepositoryData parentData, String name) throws BadItemTypeException
    {
        if (!parentData.hasValue(name))
        {
            return false;
        }
        
        if (!isCompatible(parentData, name))
        {
            throw new BadItemTypeException("Try to check " + getId() + " value from the non '" + getId() + "' data '" + name + "' on '" + parentData + "'");
        }
        
        RepositoryData groupData = parentData.getRepositoryData(name);
        return !groupData.getDataNames().isEmpty();
    }
    
    /**
     * Add a group into the given repository data
     * @param parentData repository where to add the group
     * @param name the name of the group to add
     * @return the repository data representing the added group
     */
    public default ModifiableRepositoryData add(ModifiableRepositoryData parentData, String name)
    {
        if (parentData.hasValue(name))
        {
            parentData.removeValue(name);
        }
        
        return parentData.addRepositoryData(name, getRepositoryDataType());
    }
    
    default boolean isMultiple(RepositoryData parentData, String name) throws UnknownDataException
    {
        return false;
    }
}
