/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.repository.data.type;

import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.data.UnknownDataException;
import org.ametys.plugins.repository.data.repositorydata.ModifiableRepositoryData;
import org.ametys.plugins.repository.data.repositorydata.RepositoryData;
import org.ametys.runtime.model.exception.BadItemTypeException;
import org.ametys.runtime.model.type.ModelItemType;

/**
 * Interface for types of model items stored in the repository
 */
public interface RepositoryModelItemType extends ModelItemType
{
    /** Suffix of the metadata used to check if a data is present but empty */
    public static final String EMPTY_METADATA_SUFFIX = "__isEmpty";
    
    /**
     * Checks if the value in the given repository data is compatible with the item type 
     * @param parentData repository data containing the data to check
     * @param name the name of the data to check
     * @return <code>true</code> if the data type matches the current type, <code>false</code> otherwise
     * @throws UnknownDataException if there is no data in the parent repository data with this name
     */
    public default boolean isCompatible(RepositoryData parentData, String name) throws UnknownDataException
    {
        return parentData.hasValue(name + EMPTY_METADATA_SUFFIX, RepositoryConstants.NAMESPACE_PREFIX_INTERNAL) || getRepositoryDataType().equals(parentData.getType(name));
    }
    
    /**
     * Retrieves the type of repository data used for this type
     * @return the type of repository data used for this type
     */
    public String getRepositoryDataType();
    
    /**
     * Checks if the value in the given repository data is multiple 
     * @param parentData repository data containing the data to check
     * @param name the name of the data to check
     * @return <code>true</code> if the stored data is multiple, <code>false</code> otherwise
     * @throws UnknownDataException if there is no data in the parent repository data with this name
     */
    public default boolean isMultiple(RepositoryData parentData, String name) throws UnknownDataException
    {
        return parentData.isMultiple(name);
    }
    
    /**
     * Check if there is a value, even empty, in the given repository data
     * @param parentData repository data containing the value
     * @param name the name of the element to check
     * @return <code>true</code> if there is value, <code>false</code> otherwise
     * @throws BadItemTypeException if the reading value doesn't match this element type
     */
    public default boolean hasValue(RepositoryData parentData, String name) throws BadItemTypeException
    {
        return parentData.hasValue(name) && isCompatible(parentData, name);
    }
    
    /**
     * Check if there is a non empty value in the given repository data
     * @param parentData repository data containing the value
     * @param name the name of the element to check
     * @return <code>true</code> if there is a non empty value, <code>false</code> otherwise
     * @throws BadItemTypeException if the reading value doesn't match this element type
     */
    public boolean hasNonEmptyValue(RepositoryData parentData, String name) throws BadItemTypeException;
    
    /**
     * Remove the value into the given repository data
     * @param parentData repository where the value to remove is stored.
     * @param name the name of the element to remove
     */
    public default void remove(ModifiableRepositoryData parentData, String name)
    {
        if (parentData.hasValue(name))
        {
            parentData.removeValue(name);
        }
        
        if (parentData.hasValue(name + EMPTY_METADATA_SUFFIX, RepositoryConstants.NAMESPACE_PREFIX_INTERNAL))
        {
            parentData.removeValue(name + EMPTY_METADATA_SUFFIX, RepositoryConstants.NAMESPACE_PREFIX_INTERNAL);
        }
    }
}
