/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.repository.dom;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import org.w3c.dom.Element;
import org.w3c.dom.Node;

import org.ametys.core.util.dom.AbstractWrappingAmetysElement;
import org.ametys.core.util.dom.AmetysAttribute;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.TraversableAmetysObject;
import org.ametys.plugins.repository.data.ametysobject.ModelLessDataAwareAmetysObject;
import org.ametys.plugins.repository.metadata.CompositeMetadata;
import org.ametys.plugins.repository.metadata.CompositeMetadata.MetadataType;
import org.ametys.plugins.repository.metadata.MetadataAwareAmetysObject;
import org.ametys.runtime.model.type.ElementType;
import org.ametys.runtime.model.type.ModelItemType;

/**
 * Implementation of {@link Element} wrapping an {@link AmetysObject}.<br>
 * Only methods useful for XPath processing are implemented.
 * @param <A> the actual type of the wrapped {@link AmetysObject}.
 */
public class AmetysObjectElement<A extends AmetysObject> extends AbstractWrappingAmetysElement<A>
{
    /**
     * Constructor.
     * @param object the underlying {@link AmetysObject}.
     */
    public AmetysObjectElement(A object)
    {
        this(object, null);
    }
    
    /**
     * Constructor.
     * @param object the underlying {@link AmetysObject}.
     * @param parent the parent {@link Element}.
     */
    public AmetysObjectElement(A object, AmetysObjectElement<? extends TraversableAmetysObject> parent)
    {
        super(object, parent);
    }

    @Override
    public String getTagName()
    {
        return _object.getName();
    }
    
    /**
     * This implementation returns all non-composite metadata.
     */
    @Override
    protected Map<String, AmetysAttribute> _lookupAttributes()
    {
        Map<String, AmetysAttribute> result = new HashMap<>();
        
        if (_object instanceof ModelLessDataAwareAmetysObject)
        {
            ModelLessDataAwareAmetysObject dataHolder = (ModelLessDataAwareAmetysObject) _object;
            
            for (String name : dataHolder.getDataNames())
            {
                ModelItemType type = dataHolder.getType(name);
                if (type instanceof ElementType && !dataHolder.isMultiple(name))
                {
                    Object value = dataHolder.getValue(name);
                    @SuppressWarnings("unchecked")
                    String valueAsString = ((ElementType) type).toString(value);
                    result.put(name, new AmetysAttribute(name, name, null, valueAsString, this));
                }
            }
        }
        else if (_object instanceof MetadataAwareAmetysObject)
        {
            CompositeMetadata metadata = ((MetadataAwareAmetysObject) _object).getMetadataHolder();
            
            String[] names = metadata.getMetadataNames();
            
            for (String name : names)
            {
                MetadataType type = metadata.getType(name);
                if (type != MetadataType.COMPOSITE && type != MetadataType.USER && type != MetadataType.BINARY && type != MetadataType.RICHTEXT && type != MetadataType.OBJECT_COLLECTION && !metadata.isMultiple(name))
                {
                    String value = metadata.getString(name);
                    result.put(name, new AmetysAttribute(name, name, null, value, this));
                }
            }
        }
        
        return result;
    }
    
    @Override
    public boolean hasChildNodes()
    {
        if (_object instanceof TraversableAmetysObject)
        {
            AmetysObjectIterable<AmetysObject> children = ((TraversableAmetysObject) _object).getChildren();
            
            return children.iterator().hasNext();
        }
        
        return false;
    }
    
    @SuppressWarnings("unchecked")
    @Override
    public Node getFirstChild()
    {
        if (_object instanceof TraversableAmetysObject)
        {
            AmetysObjectIterable<AmetysObject> children = ((TraversableAmetysObject) _object).getChildren();
            Iterator<AmetysObject> it = children.iterator();
            
            if (!it.hasNext())
            {
                return null;
            }
            
            return new AmetysObjectElement<>(it.next(), (AmetysObjectElement<TraversableAmetysObject>) this);
        }
        
        return null;
    }
    
    @SuppressWarnings("unchecked")
    @Override
    public Node getNextSibling()
    {
        if (_parent == null)
        {
            return null;
        }
        
        TraversableAmetysObject parent = (TraversableAmetysObject) ((AbstractWrappingAmetysElement) _parent).getWrappedObject();
        
        AmetysObjectIterable<AmetysObject> children = parent.getChildren();
        
        boolean isNext = false;
        AmetysObject nextSibling = null;
        Iterator<AmetysObject> it = children.iterator();
        
        while (nextSibling == null && it.hasNext())
        {
            AmetysObject child = it.next();
            
            if (isNext)
            {
                nextSibling = child;
            }
            else if (_object.getId().equals(child.getId()))
            {
                isNext = true;
            }
        }
        
        return nextSibling == null ? null : new AmetysObjectElement<>(nextSibling, (AmetysObjectElement< ? extends TraversableAmetysObject>) _parent);
    }
}
