/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.repository.dom;

import java.io.File;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

import org.ametys.core.util.dom.AbstractWrappingAmetysElement;
import org.ametys.core.util.dom.AmetysAttribute;
import org.ametys.plugins.repository.metadata.CompositeMetadata;
import org.ametys.plugins.repository.metadata.CompositeMetadata.MetadataType;

/**
 * DOM layer on top if a {@link File} hierarchy.
 */
public class CompositeMetadataElement extends AbstractWrappingAmetysElement<CompositeMetadata>
{
    String _name;
    
    /**
     * Constructor.
     * @param metadata the underlying {@link CompositeMetadata}.
     * @param name the name of underlying {@link CompositeMetadata}
     */
    public CompositeMetadataElement(CompositeMetadata metadata, String name)
    {
        this(metadata, name, null);
    }

    /**
     * Constructor.
     * @param metadata the underlying {@link CompositeMetadata}.
     * @param name the name of underlying {@link CompositeMetadata}
     * @param parent the parent {@link Element}.
     */
    public CompositeMetadataElement(CompositeMetadata metadata, String name, CompositeMetadataElement parent)
    {
        super(metadata, parent);
        _name = name;
    }
    
    @Override
    public String getTagName()
    {
        char c = _name.charAt(0);
        if (c >= '0' && c <= '9')
        {
            String hex = Integer.toHexString(c);
            return "_x" + StringUtils.leftPad(hex, 4, '0') + "_" + _name.substring(1);
        }
        else
        {
            return _name;
        }
    }
    
    @Override
    protected Map<String, AmetysAttribute> _lookupAttributes()
    {
        Map<String, AmetysAttribute> result = new HashMap<>();
        result.put("name", new AmetysAttribute("name", "name", null, _name, this));
        result.put("type", new AmetysAttribute("type", "type", null, MetadataType.COMPOSITE.toString(), this));
        return result;
    }
    
    @Override
    public boolean hasChildNodes()
    {
        String[] childNames = _object.getMetadataNames();
        return childNames != null ? childNames.length > 0 : false;
    }
    
    @Override
    public Node getFirstChild()
    {
        String[] childNames = _object.getMetadataNames();
        
        if (childNames != null && childNames.length > 0)
        {
            if (_object.getType(childNames[0]).equals(MetadataType.COMPOSITE))
            {
                return new CompositeMetadataElement(_object.getCompositeMetadata(childNames[0]), childNames[0], this);
            }
            else if (_object.getType(childNames[0]).equals(MetadataType.USER))
            {
                return new UserMetadataElement(_object.getCompositeMetadata(childNames[0]), childNames[0], this);
            }
            else
            {
                return new MetadataElement(childNames[0], _object, this);
            }
        }
        
        return null;
    }
    
    @Override
    public Node getNextSibling()
    {
        if (_parent == null)
        {
            return null;
        }
        
        CompositeMetadataElement parentElement = (CompositeMetadataElement) _parent;
        
        CompositeMetadata parent = parentElement.getWrappedObject();
        
        String[] children = parent.getMetadataNames();
        
        boolean isNext = false;
        String nextSibling = null;
        int i = 0;
        
        while (nextSibling == null && i < children.length)
        {
            String child = children[i++];
            
            if (isNext)
            {
                nextSibling = child;
            }
            else if (_name.equals(child))
            {
                isNext = true;
            }
        }
        
        if (nextSibling != null)
        {
            MetadataType type = parent.getType(nextSibling);
            switch (type)
            {
                case COMPOSITE:
                    return new CompositeMetadataElement(parent.getCompositeMetadata(nextSibling), nextSibling, parentElement);
                case USER:
                    return new UserMetadataElement(parent.getCompositeMetadata(nextSibling), nextSibling, parentElement);
                case MULTILINGUAL_STRING:
                    return new MultilingualStringElement(parent.getMultilingualString(nextSibling), nextSibling, parentElement);
                default:
                    return new MetadataElement(nextSibling, parent);
            }
        }
        
        return null;
    }
}
